/*
gdpc - a program for visualising molecular dynamic simulations  
Copyright (C) 1999 Jonas Frantz   

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include "parameters.h"


/*****************************************************************/
/* This function is called if the help parameter has been given. */
/* It prints a helppage for the usage of gdpc.                   */
/*****************************************************************/ 
void printhelp() 
{
    printf(" Original idea from dpc by Kai Nordlund.\n\n");
    printf(" usage: gdpc [options] columns input-file\n");
    printf("\ts 600 600        Set drawable screenarea (default: 600 600)\n");
    printf("\tcube 100         Set the volume to be drawn\n");
    printf("\tm 0/1/2          Set drawingmode to use(default: 1)\n");
    printf("\td 10             Set drawingradius of each atom (default: 2)\n");
    printf("\tx 0 50           Set the size to be drawn in x-direction\n");
    printf("\ty 0 60           Set the size to be drawn in y-direction\n");
    printf("\tz 0 70           Set the size to be drawn in z-direction\n");
    printf("\tsleep 1          Delay each frame a given amount of seconds\n");
    printf("\terase            Clear drawable area after each frame (default: off)\n");
    printf("\tw                Use white background (default: off)\n");
    printf("\tf colnum string  Just process linues with <string> in <colnum>\n");
    printf("\tgreyscale        Use greyscale colorset\n");
    printf("\tcolorinv         Use inverted colors\n");
    printf("\tcoldcolors       Use cold colorset\n");
    printf("\tcoldcolors2      Use cold colorset 2\n");
    printf("\tv 1/2            Vary atomsize according to z-column data\n");
    printf("\tsortr            Sort atoms in reverse order\n");
    printf("\tusetypes         Color atoms depending on their type.\n");
    printf("\ttifdump name     Dumps an image of each frame, name followed by frametime.\n");
    printf("\tjpgdump name     Dumps an image of each frame, name followed by frametime.\n");
    printf("\tdumpnum          Dumped images are named after framenumber.\n");
    printf("\txyz              Input file is in xyz format (default: off)\n");
    printf("\t2 3 4 5          Identify columns of infile with x,y,z,t\n\n");
    printf(" The drawingmodes are :          0 plain rektangles\n");
    printf("                                 1 plain circles\n");
    printf("                                 2 rendered balls\n");
    printf(" Varying of the size modes are : 1 Size decreases with z\n");
    printf("                                 2 Size increases with z\n");
    printf(" If neither cube,x,y or z maxsize is determined by input coordinates.\n");
    printf(" Atoms are automatically sorted by x, y and z.\n");
    printf(" The filename given to the tifdump/jpgdump parameter should be given without extension,\n");
    printf(" gdpc adds the time of the frame and the extension to the end of the filename.\n");
    printf(" If input file is in xyz format the t column will be ignored\n");
    printf(" The usetypes parameter is not relevant if not used with xyz input file, and will then\n");
    printf(" be ignored.\n");
    printf(" The only necessary input data is the column representations and the input file.\n");
    printf("\n");
}

/*******************************************************************/
/* This function handles the parameters given at the command line, */
/* and sets the variables accordingly.                             */
/*******************************************************************/
gboolean handleargs(int args,char **argv,struct GlobalParams *params) 
{
gint argl=0, inttmp;
gchar c[50];
gint control;
gfloat tmp;
gboolean setxcol=FALSE,setycol=FALSE,setzcol=FALSE,settcol=FALSE,setfile=FALSE;

    while (args-1>argl) {
	strncpy(c,argv[argl+1],strlen(argv[argl+1])+1);

	if (!strcmp(c,"s")) {
	    control=sscanf(argv[argl+2],"%d",&(params->absxsize));      
	    if (control==0) {
		printf("First parameter is invalid or missing for option: s\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    control=sscanf(argv[argl+3],"%d",&(params->absysize));      
	    if (control==0) {
		printf("Second parameter is invalid or missing for option: s\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if (params->absxsize<100 || params->absysize<100) {
		printf("The window should be at least 100*100 pixels.\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    argl+=3;
	}
	else if (!strcmp(c,"x")) {
	    control=sscanf(argv[argl+2],"%f",&(params->xmin));
	    if (control==0) {
		printf("First parameter is invalid or missing for option: x\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    control=sscanf(argv[argl+3],"%f",&(params->xmax));
	    if (control==0) {
		printf("Second parameter is invalid or missing for option: x\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if(params->xmin>=params->xmax) {
		printf("xmin must be smaller than xmax!\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    argl+=3;
	}
	else if (!strcmp(c,"y")) {
	    control=sscanf(argv[argl+2],"%f",&(params->ymin));      
	    if (control==0) {
		printf("First parameter is invalid or missing for option: y\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    control=sscanf(argv[argl+3],"%f",&(params->ymax));
	    if (control==0) {
		printf("Second parameter is invalid or missing for option: y\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if(params->ymin>=params->ymax) {
		printf("ymin must be smaller than ymax!\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    argl+=3;
	}
	else if (!strcmp(c,"z")) {
	    control=sscanf(argv[argl+2],"%f",&(params->zmin));      
	    if (control==0) {
		printf("First parameter is invalid or missing for option: z\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    control=sscanf(argv[argl+3],"%f",&(params->zmax));
	    if (control==0) {
		printf("Second parameter is invalid for or missing option: z\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if(params->zmin>=params->zmax) {
		printf("zmin must be smaller than zmax!\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    argl+=3;
	}
	else if (!strcmp(c,"cube")) {
	    control=sscanf(argv[argl+2],"%f",&tmp);
	    if(control==0) {
		printf("Invalid or missing parameter for option: cube\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if(tmp<=0.0) {
		printf("cube value must be larger than 0.0!\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if(params->xmin==65535.0) { 
		params->xmin=-tmp;
		params->xmax=tmp;
	    }
	    if(params->ymin==65535.0) { 
		params->ymin=-tmp;
		params->ymax=tmp;
	    }
	    if(params->zmin==65535.0) {
		params->zmin=-tmp;
		params->zmax=tmp;	
	    }
	    argl+=2;
	}

	else if (!strcmp(c,"erase")) { params->erase=TRUE; argl++; }
	else if (!strcmp(c,"w")) { params->whitebg=TRUE; argl++; }
	else if (!strcmp(c,"colorinv")) { params->colorset=1; argl++; }
	else if (!strcmp(c,"coldcolors")) { params->colorset=2; argl++; }
	else if (!strcmp(c,"coldcolors2")) { params->colorset=3; argl++; }
	else if (!strcmp(c,"greyscale")) { params->colorset=4; argl++; }         
	else if (!strcmp(c,"sort")) { params->sort=1; argl++; }
	else if (!strcmp(c,"sortr")) { params->sort=2; argl++; }
	else if (!strcmp(c,"xyz")) { params->fxyz=TRUE; argl++; }
	else if (!strcmp(c,"dumpnum")) { params->dumpnum=TRUE; argl++; }

	else if (!strcmp(c,"m")) {
	    control=sscanf(argv[argl+2],"%d",&(params->mode));
	    if (control==0) {
		printf("Invalid or missing parameter for option: m\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if (params->mode>2 || params->mode<0) {
		printf("Unknown drawingmode : %d\n",params->mode);
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    argl+=2;
	}      
	else if (!strcmp(c,"d")) {
	    control=sscanf(argv[argl+2],"%d",&(params->radius));
	    if(control==0) {
		printf("Invalid or missing parameter for option: d\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if (params->radius<2) params->radius=2;
	    if (params->radius>25) params->radius=25;
	    argl+=2;
	}
	else if (!strcmp(c,"sleep")) {
	    control=sscanf(argv[argl+2],"%f",&tmp);
	    if (control==0) {
		printf("Invalid or missing parameter for option: sleep\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    argl+=2;
	    params->interval=(gint) 1000*tmp;
	    if (params->interval < MININTERVAL) params->interval=MININTERVAL;
	}
	else if (!strcmp(c,"v")) {
	    control=sscanf(argv[argl+2],"%d",&(params->vary));
	    if (control==0) {
		printf("Invalid or missing parameter for option: v\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if (params->vary>2 || params->vary<1) {
		printf("Vary has only two valid modes: 1 or 2 !\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    argl+=2;
	}
	else if (!strcmp(c,"f")) {
	    control=sscanf(argv[argl+2],"%d",&(params->scol));
	    if (control==0) {
		printf("Invalid or missing parameter for option: f\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    if (params->scol<1) {
		printf("Column must be greater than zero!\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    sscanf(argv[argl+3],"%s",params->fstring);
	    argl+=3;
	}
	else if (!strcmp(c,"tifdump")) {
	    control=sscanf(argv[argl+2],"%s",params->dumpname);
	    if (control==0) {
		printf("Invalid or missing parameter for option: tifdump\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    params->tifjpg=TRUE;
	    argl+=2;
	}
	else if (!strcmp(c,"jpgdump")) {
	    control=sscanf(argv[argl+2],"%s",params->dumpname);
	    if (control==0) {
		printf("Invalid or missing parameter for option: jpgdump\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    params->tifjpg=FALSE;
	    argl+=2;
	}
	else if (!strcmp(c,"usetypes")) {
	    params->usetypes=TRUE;
	    argl++;
	}	
	else if (!strcmp(c,"help")) {
	    printhelp();
	    return FALSE;
	}
	else if (sscanf(c,"%d",&inttmp)>0) {
	    if (!setxcol) {
		params->xcolumn = inttmp;
		setxcol=TRUE;
	    }
	    else if (!setycol) {
		params->ycolumn = inttmp;
		setycol=TRUE;
	    }
	    else if (!setzcol) {
		params->zcolumn = inttmp;
		setzcol=TRUE;
	    }
	    else if (!settcol) {
		params->tcolumn = inttmp;
		settcol=TRUE;
	    }
	    argl++;
	}
	else if (setxcol && setycol && setzcol && settcol) {
	    control=sscanf(c,"%s",params->file);
	    if (control==0) {
		printf("Invalid or missing filename for input\n");
	    	printf("Use option 'help' for list of all valid command line parameters\n"); 
		return FALSE;
	    }
	    setfile=TRUE;
	    argl+=1;
	}
	else {
	    printf("Unknown option: %s\n",c);
	    printf("Please check your command line arguments.\n");
	    printf("Use option 'help' for list of all valid command line parameters\n"); 
	    return FALSE;
	}
    }

    if (!params->fxyz) params->usetypes=FALSE;
    if (setxcol && setycol && setzcol && settcol && setfile) return TRUE;
    else {
	printf("You at least have to specify input data columns and input file if\n");
	printf("you're going to start gdpc from the commandline.\n");
    }
return FALSE;
}
