/*
icosa.c Version 2.3.0. Calculate the mathematics of an icosahedron.
Copyright (C) 2001-2010  dondalah721@yahoo.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define PI M_PI
#define RADIUS 0
#define SIDE 1
#define ANGLE 72.0
#define HALFANGLE 36.0

double pi,rad2deg,deg2rad;

putstx(char *pgm)
   {
   fprintf(stderr,"Usage: %s <flag> <length>\n", pgm);
   fprintf(stderr,"If flag = -s, length = side\n", pgm);
   fprintf(stderr,"If flag = -r, length = radius\n", pgm);
   exit(1);
   } /* putstx */

/* circumscribed radius of icosahedron */
double getsphrad(side)
double side;
   {
   double r;
   double getirad();
   /* r = (1/4) * a * sqrt(10.0 + (2.0 * sqrt(5.0))) */
   r = side * getirad();
   return(r);
   } /* getsphrad */

/* circumscribed radius of 2D pentagon */
double getpradius(side)
double side;
   {
   double r,dvsr,sin36;
   sin36 = sin(HALFANGLE * deg2rad);
   r = 0.5 * side / sin36;
   return(r);
   } /* getpradius */

/* origin to midpoint of strut */
double getd(side)
double side;
   {
   double d,r,rxrx4,sxs;
   r = getsphrad(side);
   rxrx4 = 4 * r * r;
   sxs = side * side;
   d = 0.5 * sqrt(rxrx4 - sxs);
   return(d);
   } /* getd */

/* Chord length equals side */
double getchord(side)
double side;
   {
   double r,d,diff,root,chord;
   r = getsphrad(side);
   d = getd(side);
   diff = (r * r) - (d * d);
   root = sqrt(diff);
   chord = root + root;
   return(chord);
   } /* getchord */

/* Arc of strut in radians */
/* This routine is similar to arclen.c */ 
/* in the lib directory */
double gettheta(side)
double side;
   {
   double doverr,d,r,akos;
   double theta;
   r = getsphrad(side);
   d = getd(side);
   doverr = d / r;
   akos = acos(doverr);
   theta = akos + akos;
   return(theta);
   } /* gettheta */

/* origin to midpoint of strut */
double getd2(side)
double side;
   {
   double r,d,theta,kos;
   r = getsphrad(side);
   theta = gettheta(side);
   kos = cos(theta * 0.5);
   d = r * kos;
   return(d);
   } /* getd2 */

/* chord length based on Radius */
double getchord2(side)
double side;
   {
   double r,d,theta,rsintho2;
   r = getsphrad(side);
   theta = gettheta(side);
   rsintho2 = r * sin(theta * 0.5);
   d = rsintho2 + rsintho2;
   return(d);
   } /* getchord2 */

/* inscribed radius of 2D pentagon */
double getheight(side)
double side;
   {
   double rad,radxrad,hlfside,hgt;
   /* circumscribed radius of 2D pentagon */
   rad = getpradius(side);
   radxrad = rad * rad;
   hlfside = 0.5 * side;
   hgt = sqrt(radxrad - (hlfside * hlfside));
   return(hgt);
   } /* getheight */

/* arc of 2D pentagon side in radians */
double getstheta(side)
double side;
   {
   double rad,hgt,radtheta,theta;
   /* circumscribed radius of 2D pentagon */
   rad = getpradius(side);
   /* inscribed radius of 2D pentagon */
   hgt = getheight(side);
   radtheta = acos(hgt / rad);
   return(radtheta + radtheta);
   } /* getstheta */

/* midpoint of 2D pentagon side to opposite vertex */
double getpdpth(side)
double side;
   {
   double rad,radxrad,hgt,radtheta,theta;
   double pdpthxpdpth,hgtxhgt,phgt;
   double pdpth;
   /* inscribed radius of 2D pentagon */
   hgt = getheight(side);
   /* circumscribed radius of 2D pentagon */
   rad = getpradius(side);
   /* midpoint of 2D pentagon side to opposite vertex */
   pdpth = rad + hgt;
   return(pdpth);
   } /* getpdpth */

/* height of pentagon over its center when */
/* the side is the fulcrum */
double getpheight(side)
double side;
   {
   double rad,radxrad,hgt,radtheta,theta;
   double pdpthxpdpth,hgtxhgt,phgt;
   double pdpth;
   /* midpoint of 2D pentagon side to opposite vertex */
   pdpth = getpdpth(side);
   pdpthxpdpth = pdpth * pdpth;
   /* inscribed radius of 2D pentagon */
   hgt = getheight(side);
   hgtxhgt = hgt * hgt;
   /* height of pentagon over its center when */
   /* the side is the fulcrum */
   phgt = sqrt(pdpthxpdpth - hgtxhgt);
   return(phgt);
   } /* getpheight */

/* circumscribed icosahedron radius */
double getiradius(side)
double side;
   {
   double rad,phgt,hgt,hlfside,penthgt,eqhgt;
   hlfside = side * 0.5;
   /* height of equilateral triangle */
   eqhgt = sqrt((side * side) - (hlfside * hlfside));
   /* inscribed 2D pentagon radius */
   hgt = getheight(side);
   /* height of 3D polar pentagon over its center */
   penthgt = sqrt((eqhgt * eqhgt) - (hgt * hgt));
   /* height of pentagon over its center when */
   /* the side is the fulcrum */
   phgt = getpheight(side);
   rad = 0.5 * (penthgt + phgt);
   return(rad);
   } /* getiradius */

/* height of north pole over horizontal pentagon */
double geteqhgt(side)
double side;
   {
   double hlfside,hgt,eqhgt,phgt;
   /* inscribed radius of 2D pentagon */
   hgt = getheight(side);
   hlfside = 0.5 * side;
   /* height of equilateral triangle */
   eqhgt = sqrt((side * side) - (hlfside * hlfside));
   phgt = sqrt((eqhgt * eqhgt) - (hgt * hgt));
   return(phgt);
   } /* geteqhgt */

/* angle of equilateral triangle face over horizontal */
double geteqtheta(side)
double side;
   {
   double hgt,hlfside,eqhgt,akos;
   /* inscribed radius of 2D pentagon */
   hgt = getheight(side);
   /* height of equilateral triangle over horizontal */
   hlfside = 0.5 * side;
   eqhgt = sqrt((side * side) - (hlfside * hlfside));
   akos = acos(hgt / eqhgt);
   return(akos);
   } /* geteqtheta */

/* angle of equilateral triangle side over horizontal */
double getstruttheta(side)
double side;
   {
   double prad,akos;
   /* circumscribed radius of 2D pentagon */
   prad = getpradius(side);
   akos = acos(prad / side);
   return(akos);
   } /* getstruttheta */

/* vertical distance from origin to 2D pentagon */
double getvert(side)
double side;
   {
   double irad,eqhgt,vert;
   /* circumscribed radius of icosahedron */
   irad = getsphrad(side);
   /* vertical distance from 2D pentagon */
   /* to the north pole */
   eqhgt = geteqhgt(side);
   /* vertical distance from origin to */
   /* 2D pentagon */
   vert = irad - eqhgt;
   return(vert);
   } /* getvert */

/* Angle at center of sphere between center of */
/* 2D pentagon to the midpoint of the side. */
/* This equals the distance from the north pole */
/* to the midpoint of the great circle in radians. */
double getgcmidpt(side)
double side;
   {
   double hgt,irad,eqhgt,vert,theta;
   /* inscribed radius of 2D pentagon */
   hgt = getheight(side);
   /* vertical distance from origin to */
   /* 2D pentagon */
   vert = getvert(side);
   theta = atan(hgt / vert);
   return(theta);
   } /* getgcmidpt */

/* circumscribed circumference of icosahedron */
double getcircum(side)
double side;
   {
   double irad,circum;
   /* circumscribed radius of icosahedron */
   irad = getsphrad(side);
   circum = 2.0 * PI * irad;
   return(circum);
   } /* getcircum */

/* circumscribed area of icosahedron at equator */
double geteqarea(side)
double side;
   {
   double irad,area;
   /* circumscribed radius of icosahedron */
   irad = getsphrad(side);
   area = PI * irad * irad;
   return(area);
   } /* geteqarea */

/* circumscribed circumference of 2D pentagon */
double getpcircum(side)
double side;
   {
   double prad,circum;
   /* circumscribed radius of 2D pentagon*/
   prad = getpradius(side);
   circum = 2.0 * PI * prad;
   return(circum);
   } /* getparea */

/* circumscribed area of 2D pentagon */
double getparea(side)
double side;
   {
   double prad,area;
   /* circumscribed radius of 2D pentagon*/
   prad = getpradius(side);
   area = PI * prad * prad;
   return(area);
   } /* getparea */

double getgoldr(side)
double side;
   {
   int i;
   double ratio,ratiodeg,prev,curr,new;
   prev = 0.0;
   curr = 1.0;
   new = prev + curr;
   prev = curr;
   curr = new;
   printf("Calculate golden proportion\n");
   for (i=0;i<45;i++)
      {
      new = prev + curr;
      ratio = new / curr;
      printf("side %f prev %f curr %f\n"
         "   new %f ratio %25.21f\n",
         side, prev, curr, new, ratio);
      prev = curr;
      curr = new;
      }
   ratio = 0.5 * (1 + sqrt(5.0));
   ratiodeg = ratio * rad2deg;
   printf("(1+sqrt(5))/2 %25.21f\n", ratio);
   return(ratio);
   } /* getgoldr */

void calcmath(side)
double side;
   {
   double d,r,h,theta,radius,chord,height;
   double circum,area;
   radius = getpradius(side);
   printf("Circumscribed radius of 2D pentagon\n");
   printf("side %f radius %f\n", side, radius);
   radius = getsphrad(side);
   printf("Circumscribed radius of icosahedron\n");
   printf("side %f radius %f\n", side, radius);
   height = getd(side);
   printf("Origin to midpoint of icosahedron strut\n");
   printf("side %f height %f\n", side, height);
   chord = getchord(side);
   printf("Icosahedron side based on radius\n");
   printf("side %f chord %f\n", side, chord);
   theta = gettheta(side);
   printf("Icosahedron arc of side\n");
   printf("side %f theta %f\n", side, theta * rad2deg);
      radius = getd2(side);
   printf("Origin to midpoint of icosahedron strut\n");
   printf("side %f r*cos(theta/2) %f\n", side, radius);
   chord = getchord2(side);
   printf("Icosahedron side based on radius\n");
   printf("side %f 2*r*sin(theta/2) %f\n", side, chord);
   height = getheight(side);
   printf("Inscribed radius of 2D pentagon\n");
   printf("side %f radius %f\n", side, height);
   theta = getstheta(side);
   printf("Arc of 2D pentagon side\n");
   printf("side %f theta %f\n", side, theta * rad2deg);
   height = getpdpth(side);
   printf("Midpoint of 2D pentagon side to opposite vertex\n"); 
   printf("side %f distance %f\n", side, height);
   height = getpheight(side);
   printf("Height of pentagon over its center when "
      "the side is the fulcrum\n");
   printf("side %f height %f\n", side, height);
   height = geteqhgt(side);
   printf("Height of equilateral triangle apex ");
   printf("over horizontal pentagon\n");
   printf("side %f height %f\n", side, height);
   radius = getiradius(side);
   printf("Icosahedron circumscribed radius\n");
   printf("equals half of sum of previous two heights.\n");
   printf("side %f icosahedron radius %f\n", side, radius);
   theta = geteqtheta(side);
   printf("Angle of equilateral triangle face ");
   printf("over horizontal pentagon\n");
   printf("side %f theta %f\n", side, theta * rad2deg);
   theta = getstruttheta(side);
   printf("Angle of equilateral triangle side ");
   printf("over horizontal pentagon\n");
   printf("side %f theta %f\n", side, theta * rad2deg);
   height = getvert(side);
   printf("Vertical distance from origin to 2D pentagon\n"); 
   printf("side %f height %f\n", side, height);
   theta = getgcmidpt(side);
   printf("Angle at origin from north pole "
      "to midpoint of great circle.\n");
   printf("Equals angle at origin of inscribed "
      "radius of 2D pentagon.\n");
   printf("side %f theta %f\n", side, theta * rad2deg);
   circum = getcircum(side);
   printf("circumscribed circumference of icosahedron\n");
   printf("side %f circum %f\n", side, circum);
   area = geteqarea(side);
   printf("circumscribed area of icosahedron at equator\n");
   printf("side %f area %f\n", side, area);
   circum = getpcircum(side);
   printf("circumscribed circumference of 2D pentagon\n");
   printf("side %f circum %f\n", side, circum);
   area = getparea(side);
   printf("circumscribed area of 2D pentagon\n");
   printf("side %f area %f\n", side, area);
   height = getgoldr(side);
   } /* calcmath */

void rad2side(radius)
double radius;
   {
   double side;
   double getirad();
   side = radius / getirad();
   printf("Calculate side based on radius\n");
   printf("radius %f side %f\n", radius, side);
   calcmath(side);
   } /* rad2side */

main(argc,argv)
int argc;
char **argv;
   {
   int flg,rslt;
   double s,r;
   char swtch[16];
   pi = PI;
   rad2deg = 180.0 / pi;
   deg2rad = pi / 180.0;
   flg = -1;
   if (argc != 3) putstx(*argv);
   strcpy(swtch,*(argv+1));
   rslt = strcmp(swtch,"-s");
   if (!rslt)
      {
      s = atof(*(argv+2));
      if (s <= 0.0)
         {
         fprintf(stderr,"Side length <= 0\n");
         putstx(*argv);
         } /* invalid side length */
      else flg = SIDE;
      printf("side %f\n", s);
      } /* if -s */
   rslt = strcmp(swtch,"-r");
   if (!rslt)
      {
      r = atof(*(argv+2));
      if (r <= 0.0)
         {
         fprintf(stderr,"Radius length <= 0\n");
         putstx(*argv);
         } /* invalid radius length */
      else flg = RADIUS;
      printf("radius %f\n", r);
      } /* if -r */
   if (flg == SIDE) calcmath(s);
   else if (flg == RADIUS) rad2side(r);
   else
      {
      fprintf(stderr,"Invalid flag %s\n", *(argv+1));
      exit(1);
      } /* invalid flag */
   return(0);
   } /* main */
