# install.pl -- install with string substitution
# Copyright (C) 2000 Tim Freeman
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
# 
# You should have received a copy of the GNU Library General Public
# License along with this library in the file COPYING.txt; if not,
# write to the Free Software Foundation, Inc., 59 Temple Place -
# Suite 330, Boston, MA 02111-1307, USA
#
# The author can be reached by email at tim@infoscreen.com, or by
# paper mail at:
#
# Tim Freeman
# 655 S. FairOaks Ave., Apt B-316
# Sunnyvale, CA 94086
#

# Use path to find perl if this is invoked by hand,
# since this guy is invoked to edit the true name of
# perl into other scripts and therefore doesn't know the name.
# See Programming Perl, page 220.
eval '(exit $?0)' && eval 'exec perl -w -S $0 ${1+"$@"}'
& eval 'exec perl -w -S $0 $argv:q'
if 0;

sub usage {
    die <<"EOM";
usage: $0 [<string1>=<val1> <string2>=<val2> ...] [-v] [-e <errorstr>] <mode> <from> <to>
   or: $0 [<string1>=<val1> <string2>=<val2> ...] [-v] [-e <errorstr>] <mode> <from1> <from2> ... <fromn> <dir>
   or: $0 [<string1>=<val1> <string2>=<val2> ...] [-v] [-e <errorstr>] <mode> -d <dir>
Install a file or directory, while doing string substitutions.
-e <errorstr> means to report an error if the given string is in the
   file after we are done with all substitutions. 
-v means verbose.
-d means to install a directory, otherwise install a file.
<stringi>=<vali> means to substitute <vali> for <stringi> in the file.
   (It has no effect if we are installing a directory.)
<mode> is the permissions to give to <to> or <dir>, in octal.
<from> is the file to copy.
<to> is the file to copy <from> into after doing the substitutions.

The substitutions are done in order from longest to shortest, with no
particular order used for several strings with the same length.
EOM
}

&usage unless @ARGV;

sub parsesubst {
    local ($subst) = @_;
    $subst =~ /^([^=]+)=([^=]*)$/ ||
	die "Ill-formatted subst $subst in parsesubst";
    return ($1, $2);
}
    
@sortsubsts=();
$verbose = 0;
$errorstring = undef;

while (@ARGV && ($ARGV[0] =~ /^[^=]+=[^=]*$/ ||
		 '-v' eq $ARGV[0] || '-e' eq $ARGV[0])) {
    if ('-v' eq $ARGV[0]) {
	$verbose = 1;
	shift;
    } elsif ('-e' eq $ARGV[0]) {
	shift;
	$errorstring = shift;
    } else {
	push (@sortsubsts, shift (@ARGV));
    }
}

sub sortfun {
    local ($keya) = &parsesubst ($a);
    local ($keyb) = &parsesubst ($b);
    return length($keyb) <=> length($keya);
}

@sortsubsts = sort sortfun @sortsubsts;

@substs=();
while (@sortsubsts) {{
    local ($key, $val) = parsesubst (shift (@sortsubsts));
    push (@substs, $key, $val);
}}

die "No mode specified" unless defined ($ARGV[0]) && $ARGV[0] =~ /^[0-7]+$/;
# For error message.
$preoctmode = shift;
$mode = oct ($preoctmode);

# Slurp whole files, otherwise we have loops within loops which is too slow.
undef ($/);
# Regexps span lines.  But don't get a warning in perl 5.
local ($oldwarn) = $^W;
$^W = 0;
eval ('$* = 1');
$^W = $oldwarn;

# Create every file by renaming a file, so we have atomic changes
# and can maybe safely install into a running system.
sub fakename {
    local ($fname) = @_;
    # Put the trailing tilde on the end so "clean" will clean it up.
    return "$fname.fake$$~";
}

sub fixfakename {
    local ($fname) = @_;
    local ($fakename) = &fakename ($fname);
    if (!rename ($fakename, $fname)) {
	unlink ($fakename);
	die "Can't rename $fakename to $fname: $!";
    }
}

sub install_one {
    local ($from, $to, $mode) = @_;
    print "Creating file $to.\n" if $verbose;
    local ($faketo) = &fakename ($to);
    unlink ($faketo);
    die "Can't delete $faketo before creating: $!" if -e $faketo;
    if (-B $from) {{
	# If it's a binary file, don't substitute,
	# and be prepared to deal with very large files.
	local ($buf) = "";
	local ($r);
	open (IN, "<$from");
	open (OUT, ">$faketo") || die "Can't write $faketo: $!";
	for (;;) {
	    $r = read (IN, $buf, 100000);
	    die "Read error on $from: $!" unless defined ($r);
	    last unless $r;
	    print OUT $buf;
	}
	close (IN);
	close (OUT);
    }} else {{
	# If it is a text file, read it in all at once, otherwise we
	# have to loop on each line which would be slow.
	open (IN, "<$from") || die "Can't read $from: $!";
	local ($text) = <IN>;
	close (IN);
	local ($val, $string);
	local (@mysubsts) = @substs;
	for (;;) {
	    last unless @mysubsts;
	    $string = shift (@mysubsts);
	    $val = shift (@mysubsts);
	    $text =~ s/$string/$val/g;
	}
	local ($faketo) = &fakename ($to);
	open (OUT, ">$faketo") || die "Can't write $faketo: $!";
	print OUT $text;
	close (OUT);
	if ($errorstring && ($text =~ /$errorstring/o)) {
	    die "After substitutions, $faketo contains $errorstring."
	}
    }}
    chmod ($mode, $faketo) || die "Can't set permissions on $faketo to $preoctmode: $!";
    &fixfakename ($to);
}

if ($ARGV[0] eq "-d") {{
    shift;
    local ($to);
    foreach $to (@ARGV) {
	print "Creating directory $to.\n" if $verbose;
	if (!-d $to) {
	    mkdir ($to, $mode)
		|| die "Can't create directory $to with mode $preoctmode: $!";
	}
	# The mode is overridden by the current umask, so always do a chmod.
	chmod ($mode, $to)
	    || die "Can't set permissions on directory $to to $preoctmode: $!";
    }
}} elsif (@ARGV==2 && ! -d ($ARGV[1])) {
    &install_one ($ARGV[0], $ARGV[1], $mode);
} else {{
    local ($dir) = pop (@ARGV);
    die "$dir should be a directory" unless -d $dir;
    local ($from, $tail);
    foreach $from (@ARGV) {
	if ($from =~ m!/([^/]+)$!) {
	    $tail = $1;
	} else {
	    $tail = $from;
	}
	&install_one ($from, "$dir/$tail", $mode);
    }
}}
