// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//
#ifndef __XBallDrawerConfiguration_h__
#include "XBallDrawerConfiguration.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __PositiveFloatSlotValue_h__
#include "PositiveFloatSlotValue.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __PhysicsObject_h__
#include "PhysicsObject.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __BoringAtom_h__
#include "BoringAtom.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __XObjectDrawer_h__
#include "XObjectDrawer.h"
#endif

#ifndef __BallConfiguration_h__
#include "BallConfiguration.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __XLineDrawer_h__
#include "XLineDrawer.h"
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __Color_h__
#include "Color.h"
#endif

#ifndef __AtomConfiguration_h__
#include "AtomConfiguration.h"
#endif

#ifndef __TheObjectDrawerFactory_h__
#include "TheObjectDrawerFactory.h"
#endif

#ifndef __AtomInfo_h__
#include "AtomInfo.h"
#endif

#ifndef __DepthClipInfo_h__
#include "DepthClipInfo.h"
#endif

#ifndef __BoundingSphere_h__
#include "BoundingSphere.h"
#endif

#ifndef __BoolSlotValue_h__
#include "BoolSlotValue.h"
#endif

namespace {
  class XWireFrameDrawerConfiguration
    : public XBallDrawerConfiguration
  {
    static String clickRadiusName () {
      return "Radius for mouse clicks and atoms without bonds";
    }
    static String minFadeName () {
      return "Remaining intensity for wires near background";
    }
    static String alwaysBallName () {
      return "Always draw balls";
    }
  public:
    XWireFrameDrawerConfiguration () {
      initializeSlot (clickRadiusName (), NEW (PositiveFloatSlotValue (0.5)));
      initializeSlot (minFadeName (), NEW (PositiveFloatSlotValue (0.25)));
      initializeSlot (alwaysBallName (), NEW (BoolSlotValue (false)));
    }
    Float getClickRadius () const {
      return PositiveFloatSlotValue::getSlot (this, clickRadiusName ());
    }
    void setClickRadius (Float f) {
      PositiveFloatSlotValue::setSlot (this, clickRadiusName (), f);
    }
    Float getMinFade () const {
      return PositiveFloatSlotValue::getSlot (this, minFadeName ());
    }
    void setMinFade (Float f) {
      assert (0 < f && f <= 1);
      PositiveFloatSlotValue::setSlot (this, minFadeName (), f);
    }
    bool getAlwaysBall () const {
      return BoolSlotValue::getSlot (this, alwaysBallName());
    }
    void setAlwaysBall (bool b) {
      BoolSlotValue::setSlot (this, alwaysBallName(), b);
    }
    SP<Configuration> copy () const {
      return NEW (XWireFrameDrawerConfiguration (*this));
    }
    bool hasTheSameTypeAs (const Configuration *c) const {
      return 0 != dynamic_cast <const XWireFrameDrawerConfiguration *> (c);
    }
  };
  SP<const XObjectDrawer> genBallDrawer (const Color &c,
					 XWireFrameDrawerConfiguration *conf)
  {
    SP<XBallDrawerConfiguration> xbc =
      dynamic_cast <XBallDrawerConfiguration *> (&*(conf->copy ()));
    assert (xbc);
    // Copy should have copied the physics object configuration too.
    assert (xbc->getPhysicsObjectConfiguration () !=
	    conf->getPhysicsObjectConfiguration ());
    SP<BallConfiguration> bc = NEW (BallConfiguration ());
    // The radius of the ball, when we draw it, is our click radius.
    bc->setVisibleRadius (conf->getClickRadius ());
    bc->setMaterial (c);
    xbc->setPhysicsObjectConfiguration (&*bc);
    SP<const Configurable> result1 =
      &* (FactoryTable::load ("ObjectDrawer", "XBallDrawer")->makeIt (&*xbc));
    assert (result1);
    // The atom factory is supposed to prevent us from creating very many of
    // these, so doing a dynamic_cast each time we create is okay.
    SP<const XObjectDrawer> result2 =
      dynamic_cast <const XObjectDrawer *> (&*result1);
    return result2;
  }
  class XWireFrameDrawer
    : public XObjectDrawer
  {
    // Delegate to the XBallDrawer instead of being a subclass of it, since
    // that's cleaner and the class isn't public.
    const SP<const XObjectDrawer> m_ballDrawer;
    const Float m_clickRadius;
    const Color m_color;
    const Float m_minFade;
    const bool m_alwaysDrawBalls;
  public:
    XWireFrameDrawer (const Color &c, XWireFrameDrawerConfiguration *conf)
      : m_ballDrawer (genBallDrawer (c, conf)),
	m_clickRadius (conf->getClickRadius ()),
	m_color (c),
	m_minFade (conf->getMinFade ()),
	m_alwaysDrawBalls (conf->getAlwaysBall())
    {}
    bool intersect (const Float *state,
		    const Vec3 &v1, const Vec3 &v2, Vec3 *result,
		    int objectIndex, const SceneGraph *sg)
      const
    {
      // We might try to support clicking on the bonds here, but not today.
      return m_ballDrawer->intersect (state, v1, v2, result,
				      objectIndex, sg);
    }
    bool graphicsIntersect (const Float *state, const BoundingBox &box,
			    int objectIndex, const SceneGraph *sg)
      const {
      return m_ballDrawer->graphicsIntersect (state, box, objectIndex, sg);
    }
    bool intersect (const Float *state, const Vec3 &v,
		    int objectIndex, const SceneGraph *sg) const {
      return m_ballDrawer->intersect (state, v, objectIndex, sg);
    }
    // Bug: have a bunch of wireframe atoms near each other, one with a really
    // long bond.  The old scheme used the graphicsBoundingSphere to determine
    // what you could select, so the outcome was that no matter where in the
    // vicinity you clicked, you select the one with the long bond.  The
    // solution is to have a separate method for determining the bounding
    // sphere radius for the purpose of doing selections.
    // FIXME This will break down once we have objects with truly non-spherical
    // sensitive areas, like bouncing cubes.  The right primitive would say
    // whether an orthogonal transformation applied to an x,y,z box intersects
    // the object.  Yuck, complicated.  Another option is to give up on
    // selecting a frustrum and only allow selecting the one object under the
    // mouse.
    BoundingSphere selectionBoundingSphere (const Float *state,
					    int objectIndex,
					    const SceneGraph *sg)
      const
    {
      (void) objectIndex; (void) sg;
      return BoundingSphere (m_clickRadius, boundingSphereCenter (state));
    }

    Float graphicsBoundingSphereRadius (int objectIndex,
					const SceneGraph *sg) const {
      // Return the maximum of m_clickRadius and half the distance to the
      // furthest bonded atom.
      const Vec3 &myCenter =
	BoringAtom::center (sg->objectState (objectIndex));
      int count;
      const int *links;
      sg->getLinkManager ()->getLinks (objectIndex, count, links);
      Float maxBondLengthSquared = 0;
      for (int i = 0; i < count; i++) {
	const int link = links [i];
	if (-1 != link) {
	  const PhysicsObject *po = sg->object (link);
	  if (po->isInstanceOf (BoringAtom::staticClassId)) {
	    const Vec3 &otherCenter =
	      BoringAtom::center (sg->objectState (link));
	    maxBondLengthSquared =
	      FloatUtil::max (maxBondLengthSquared,
			      (otherCenter - myCenter).lengthSquared ());
	  }
	}
      }
      if (maxBondLengthSquared > 4 * FloatUtil::sqr (m_clickRadius)) {
	return FloatUtil::sqrt (maxBondLengthSquared)/2;
      } else {
	return m_clickRadius;
      }
    }
    Vec3 boundingSphereCenter (const Float *state) const {
      return m_ballDrawer-> boundingSphereCenter (state);
    }
    // Draw lines from the present atom (with center specified by state) to its
    // neighbors (with neighbor numbers specified by the first count integers
    // found at links, and the meanings of the neighbor numbers given by sg).
    // vo is the frame to draw on.  
    // dci should be maintained with information about the number of pixels at
    // each depth.
    // finished is an input indicating which depth bins are finished (so maybe
    // we could skip drawing them).
    // FIXME Why don't I glue together dci, finished, and vo into an object
    // so I don't have to pass around so much cruft?
    void drawLines (const Float *state, const int *links, int count,
		    const SceneGraph *sg,
		    const Canvas *vo, const DepthClipInfo &dci,
		    const BinSet &finished) const {
      // Draw some lines
      // Note that where differs from where2 because where has already been
      // translated so its depth is measured from the near Z plane. 
      const Vec3i where2 = vo->transformWorldToLHScreen
	(BoringAtom::center (state)).truncateToVec3i ();
      for (int i = 0; i < count; i++) {
	const int link = links [i];
	if (-1 != link &&
	    sg->object (link)->isInstanceOf (BoringAtom::staticClassId)) {
	  Vec3i otherCenter =
	    vo->transformWorldToLHScreen
	    (BoringAtom::center (sg->objectState (link))).truncateToVec3i();
	  Vec3i otherEnd = (where2 + otherCenter) / 2;
	  XLineDrawer::drawFadingLine (vo, where2, otherEnd, m_color,
				       dci, finished, m_minFade);
	}
      }
    }
    void draw (const Float *state, const Canvas *vo,
	       const Vec3i &where, const DepthClipInfo &dci,
	       const BinSet &finished, int objectIndex,
	       const SceneGraph *sg) const {
      const LinkManager *lm = sg->getLinkManager ();
      int count;
      const int *links;
      lm->getLinks (objectIndex, count, links);
      if (m_alwaysDrawBalls) {
	m_ballDrawer->draw (state, vo, where, dci, finished,
			    objectIndex, sg);
	drawLines (state, links, count, sg, vo, dci, finished);
      } else {	
	bool foundBond = false;
	for (int i = 0; i < count; i++) {
	  const int link = links [i];
	  if (-1 != link &&
	      sg->object (link)->isInstanceOf (BoringAtom::staticClassId)) {
	    foundBond = true;
	    break;
	  }
	}
	if (!foundBond) {
	  m_ballDrawer->draw (state, vo, where, dci, finished,
			      objectIndex, sg);
	} else {
	  drawLines (state, links, count, sg, vo, dci, finished);
	}
      }
    }
    bool isVisible () const {
      return true;
    }
  };
  class XWireFrameDrawerFactory
    : public TypedFactory <XWireFrameDrawerConfiguration, XWireFrameDrawer>
  {
  public:
    XWireFrameDrawerFactory ()
      :
      TypedFactory <XWireFrameDrawerConfiguration, XWireFrameDrawer> ("XWireFrameDrawer")
    {}
    SP<XWireFrameDrawerConfiguration> typedDefaultConfiguration () const {
      return NEW (XWireFrameDrawerConfiguration ());
    }
    SP<XWireFrameDrawer> makeIt (XWireFrameDrawerConfiguration *c) const {
      SP<AtomConfiguration> ac = 
	dynamic_cast <AtomConfiguration *>
	(&*c->getPhysicsObjectConfiguration ());
      assert (ac);
      const int num = ac->getSymbolNumber ();
      return NEW (XWireFrameDrawer (AtomInfo::CPKColor (num), c));
    }
  };
  class XBallWireFrameDrawerFactory
    : public TypedFactory <XWireFrameDrawerConfiguration, XWireFrameDrawer>
  {
  public:
    XBallWireFrameDrawerFactory ()
      :
      TypedFactory <XWireFrameDrawerConfiguration, XWireFrameDrawer>
    ("XBallWireFrameDrawer")
    {}
    SP<XWireFrameDrawerConfiguration> typedDefaultConfiguration () const {
      SP<XWireFrameDrawerConfiguration> result =
	NEW (XWireFrameDrawerConfiguration ());
      result->setAlwaysBall (true);
      return result;
    }
    SP<XWireFrameDrawer> makeIt (XWireFrameDrawerConfiguration *c) const {
      SP<AtomConfiguration> ac = 
	dynamic_cast <AtomConfiguration *>
	(&*c->getPhysicsObjectConfiguration ());
      assert (ac);
      const int num = ac->getSymbolNumber ();
      return NEW (XWireFrameDrawer (AtomInfo::CPKColor (num), c));
    }
  };
  const bool useless =
  (AtomConfiguration::registerDrawer ("WireFrame", "Wire Frame"),
   AtomConfiguration::registerDrawer ("BallWireFrame", "Ball and Wire"),
   TheObjectDrawerFactory::registerObjectDrawer ("WireFrame"),
   TheObjectDrawerFactory::registerObjectDrawer ("BallWireFrame"),
   FactoryTable::store ("ObjectDrawer", NEW (XWireFrameDrawerFactory ())),
   FactoryTable::store ("ObjectDrawer", NEW (XBallWireFrameDrawerFactory ())),
   true);

}
