// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

// perror.
#include <stdio.h>

// Next three so I can call select.
#include <sys/time.h>
#include <sys/types.h>
// Next one for select and usleep.
#include <unistd.h>

// For XImage, among others.
#ifndef __X11_Xlib_h__
#include <X11/Xlib.h>
#define __X11_Xlib_h__
#endif

#ifndef __X11_Xutil_h__
#include <X11/Xutil.h>
#define __X11_Xutil_h__
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __TopLevelConfiguration_h__
#include "TopLevelConfiguration.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __GenericObjectDrawer_h__
#include "GenericObjectDrawer.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __TheObjectDrawerFactory_h__
#include "TheObjectDrawerFactory.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __ConstantRecursiveSlotValue_h__
#include "ConstantRecursiveSlotValue.h"
#endif

#ifndef __Color_h__
#include "Color.h"
#endif

#ifndef __Stopwatch_h__
#include "Stopwatch.h"
#endif

#ifndef __Integrator_h__
#include "Integrator.h"
#endif

#ifndef __Evaluator_h__
#include "Evaluator.h"
#endif

#ifndef __DispatcherConfiguration_h__
#include "DispatcherConfiguration.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __BoolSlotValue_h__
#include "BoolSlotValue.h"
#endif

#ifndef __RungaKuttaFactory_h__
#include "RungaKuttaFactory.h"
#endif

#ifndef __IMWheel_h__
#include "IMWheel.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __InputDevice_h__
#include "InputDevice.h"
#endif

#ifndef __XKeyboardMouse_h__
#include "XKeyboardMouse.h"
#endif

#ifndef __stdlib_h__
#include <stdlib.h>
#define __stdlib_h__
#endif

#ifndef __PositiveIntSlotValue_h__
#include "PositiveIntSlotValue.h"
#endif

#ifndef __KeyboardMouseEvent_h__
#include "KeyboardMouseEvent.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __ObjectLink_h__
#include "ObjectLink.h"
#endif

#ifndef __XCanvas_h__
#include "XCanvas.h"
#endif

#ifndef __float_h__
#include <float.h>
#define __float_h__
#endif

#ifndef __XCanvasConfiguration_h__
#include "XCanvasConfiguration.h"
#endif

namespace {
  // Can't typedef ... Depth, since Xlib.h has a different definition.
  // typedef Canvas::Depth Depth;
  class XTopLevelConfiguration
    : public TopLevelConfiguration
  {
    static String shmName () {
      return "Use MIT Shared Memory Extension";
    }
    static String dbeName () {
      return "Use Double Buffering Extension";
    }
    static String pollName () {
      return "Sleep between polls (microseconds)";
    }
    static String busyBugName () {
      return "Busy device bug threshhold";
    }
  public:
    XTopLevelConfiguration ()
      : TopLevelConfiguration (// Prefix for object drawers to use.
			       "X",
			       // Prefixes for input device drivers to load.
			       Dynavec <String> (
#ifdef linux
						 "Linux",
#endif
						 "X"))
    {
      {
	const bool shmInitValue =
#ifdef TRY_MIT_SHM
	  true
#else
	  false
#endif
	  ;
	initializeSlot (shmName(), NEW (BoolSlotValue (shmInitValue)));
      }
      {
	const bool dbeInitValue =
#ifdef TRY_DBE
	  true
#else
	  false
#endif
	  ;
	initializeSlot (dbeName (), NEW (BoolSlotValue (dbeInitValue)));
      }
      // Only let them configure the poll interval if there are devices that
      // want polling.  Otherwise, people might believe that things are less
      // efficient than they are, or they might try hard to optimize the
      // polling interval when it has no effect.
      Dynavec <SP <InputDevice> > inputDevices = getInputDevices ();
      bool needPoll = false;
      for (int i = 0; i < inputDevices.size(); i++) {
	if (inputDevices [i]->fdForSelect() == -1
	    &&
	    0 != dynamic_cast <XKeyboardMouse *> (&*inputDevices [i])) {
	  needPoll = true;
	}
      }
      if (needPoll) {
	// 50000 microseconds = 1/20 second.
	initializeSlot (pollName (), NEW (PositiveIntSlotValue (50000)));
      }
      initializeSlot (busyBugName (), NEW (PositiveIntSlotValue (100)));
    }
    bool getShm () const {
      return BoolSlotValue::getSlot (this, shmName());
    }
    void setShm (bool value) {
      return BoolSlotValue::setSlot (this, shmName(), value);
    }
    bool getDbe () const {
      return BoolSlotValue::getSlot (this, dbeName());
    }
    void setDbe (bool value) {
      return BoolSlotValue::setSlot (this, dbeName(), value);
    }
    // If there are no polling input devices, then calling either of the next
    // two is illegal, and they'll fail if we're compiling with debugging
    // turned on.
    int getPollInterval () {
      return PositiveIntSlotValue::getSlot (this, pollName ());
    }
    void setPollInterval (int pollInterval) {
      PositiveIntSlotValue::setSlot (this, pollName (), pollInterval);
    }
    // A select device or a poll device can have a bug where it appears busy
    // all the time.  To avoid hanging, if any such devices exhibit this bug,
    // we print a message disable them.  Otherwise we would hang.
    int getBusyBug () {
      return PositiveIntSlotValue::getSlot (this, busyBugName ());
    }
    void setBusyBug (int busyBug) {
      PositiveIntSlotValue::setSlot (this, busyBugName (), busyBug);
    }
  };

  class XTopLevelImpl
    : public TopLevel
  {
    // Whether we need to redraw the screen.
    bool m_needDraw;
    bool m_wantExit;
    bool m_wantNewWindowSystem;
    SP<SceneGraph> m_sg;
    SP<Integrator> m_integrator;
    bool m_integrating;
    bool m_ballIsGrabbed;
    Times m_times;
    Atom m_deleteWindowAtom;
    // This file doesn't need to know the pixel size for the XCanvas.
    SP<XCanvas> m_vo;
    SP<XTopLevelConfiguration> m_conf;
    SP<Stopwatch> m_frameStart;
    // All of the input devices.  Every input device in m_inputDevices is in
    // exactly one of m_pollInputDevices, m_selectInputDevices, or it is
    // m_keyboardMouse. 
    Dynavec <SP <InputDevice> > m_inputDevices;
    // The devices that need polling.
    Dynavec <SP <InputDevice> > m_pollInputDevices;
    // The devices that should have their file descriptors included when we
    // call select.
    Dynavec <SP <InputDevice> > m_selectInputDevices;
    // m_keyboardMouse will also be on the inputDevices list.  It doesn't
    // matter where.
    SP<XKeyboardMouse> m_keyboardMouse;
    // The device that gave us the most recent input.
    SP<InputDevice> m_lastInputDevice;
  public:
    XTopLevelImpl (SP<XTopLevelConfiguration> conf);
    ~XTopLevelImpl ();
    bool mainLoop ();
    void doExit ();
    void newWindowSystem ();
    SP<SceneGraph> getSceneGraph ();
    SP<const SceneGraph> getSceneGraph () const;
    void setSceneGraph (SP<SceneGraph> sg);
    void setIntegrating (bool b);
    bool getIntegrating () const;
    bool findUnderMouse (int &index, Vec3 &intersectionPoint) const;
    Times getTimes () const;
    void initFungimolState ();
    void closeFungimolState ();
    void draw ();
    void mouse (int button, int state, int winx, int winy);
    void mouseMotion (int x, int y);
    void reshape (int width, int height);
    void timeStep ();
    void handleNonInputXEvent (const XEvent *event);
    void oneXEvent ();
    void allPendingXEvents ();
    void oneBlockingXEvent ();
    void readAllEvents (InputDevice *inp);
    Display *getDisplay () {
      return m_vo->getDisplay ();
    }
    SP<const Event> getEvent () const;
    void grab (SP<RecursiveSlotValue> rsv);
    SP<const InputDevice> getDevice () const;
    Canvas *getCanvas ();
  };

  XTopLevelImpl::XTopLevelImpl (SP<XTopLevelConfiguration> conf)
  : m_conf (conf)
  {
    {
      SP<XCanvasConfiguration> xcc = NEW (XCanvasConfiguration ());
      xcc->setUseShm (conf->getShm());
      xcc->setUseDbe (conf->getDbe());
      m_vo = NEW (XCanvas (xcc));
    }
    m_inputDevices = conf->getInputDevices ();
    for (int i = 0; i < m_inputDevices.size(); i++) {
      InputDevice *inp = m_inputDevices [i];
      XKeyboardMouse *isKeyboardMouse = dynamic_cast <XKeyboardMouse *> (inp);
      if (isKeyboardMouse) {
	m_keyboardMouse = isKeyboardMouse;
      } else if (-1 == inp->fdForSelect ()) {
	m_pollInputDevices.push (inp);
      } else {
	m_selectInputDevices.push (inp);
      }
    }
    if (! m_keyboardMouse) {
      die ("Could not find an XKeyboardMouse input device.");
    }
    ref ();
    // Set up the object drawer.
    // Do it during initialization, instead of at the top of mainLoop, so we
    // can call ViewScene from main, and then ViewScene can get the
    // graphicsBoundingSphere of the scene..
    SP<ConstantRecursiveSlotValue> od = m_conf->getObjectDrawer ();
    assert (od);
    TheObjectDrawerFactory::setTheObjectDrawerFactory (od);
    deref ();
  }

  Canvas *XTopLevelImpl::getCanvas () {
    return &*m_vo;
  }

  SP<const Event> XTopLevelImpl::getEvent () const {
    return m_lastInputDevice->getEvent ();
  }

  void XTopLevelImpl::grab (SP<RecursiveSlotValue> rsv) {
    m_lastInputDevice->grab (rsv);
  }

  SP<const InputDevice> XTopLevelImpl::getDevice () const {
    return &*m_lastInputDevice;
  }
  
  // Handle X events that aren't direct user input, that is, events for which
  // XKeyboardMouse::insertXEvent returns false.
  void XTopLevelImpl::handleNonInputXEvent (const XEvent *event) {
    switch (event->type) {
    case MappingNotify:
      // Xlib R4/R5 Reference Manual, edited by Adrian Nye, OReilly & Assocs,
      // Dec 1993 printing, Appendix E, MappingNotify, page 820 says we
      // should only call XRefreshKeyboardMapping when the request is
      // MappingKeyboard.
      //
      // There used to be a crashing bug here, but it had nothing to do with
      // calling XRefreshKeyboardMapping when the request code wasn't
      // MappingKeyboard.  I was passing (XMappingEvent *) (&event) to
      // XRefreshKeyboardMapping.  Event is a pointer, it's address is a
      // pointer into our stack frame, which can be coerced into a completely
      // useless XMappingEvent * that is nothing at all like
      // &(event->xmapping).
      //
      // Morals of the story:
      // 1. Don't use c-style casts.  
      // 2. Don't use a pointer cast to substitute for taking a field of a
      // union.  
      if (MappingKeyboard == event->xmapping.request) {
	// IMO XRefreshKeyboardMapping should take a const XMappingEvent *, but
	// it takes a non-const XMappingEvent *, hence the const_cast.
	XRefreshKeyboardMapping(const_cast <XMappingEvent *>
				(& (event->xmapping)));
      }
      break;	 
    case Expose:
      // We tend to get a whole bunch of expose's at once, if the window is
      // dragged around.  So just set a flag to be sure we redraw soon.
      m_needDraw = true;
      break;
    case ConfigureNotify:
      reshape (event->xconfigure.width, event->xconfigure.height);
      m_needDraw = true;
      break;
    case ClientMessage:
      if (((unsigned) event->xclient.data.l[0]) == m_deleteWindowAtom) {
	m_wantExit = true;
      }
      break;
    default:
      // Ignore.
      break;
    }
  }

  // Read and handle exactly one X event.
  void XTopLevelImpl::oneXEvent () {
    XEvent event;
    XNextEvent (getDisplay(), &event);
    if (m_keyboardMouse->insertXEvent (&event, m_vo->getHeight ())) {
      bool readResult = m_keyboardMouse->readEvent ();
      // We just successfully fed it an event, so it should have an event.
      (void) readResult;
      assert (readResult);
      m_lastInputDevice = &*m_keyboardMouse;
      m_sg->validate ();
      m_keyboardMouse->dispatchEvent (this);
      m_needDraw = true;
    } else {
      handleNonInputXEvent (&event);
    }
  }

  void XTopLevelImpl::allPendingXEvents () {
    // Used to have a call to XPending here.  Problem is, XPending can read
    // more inputs from the queue, which means that we can then block when we
    // shouldn't in oneBlockingXEvent below.  XQLength never reads.
    while (XQLength (getDisplay ())) {
      oneXEvent ();
    }
  }    

  // Read X events, reading from the file descriptor once.  First deal with all
  // of the events in the queue, then do one blocking XNextEvent, then deal
  // with all of the events in the queue again.
  void XTopLevelImpl::oneBlockingXEvent () {
    allPendingXEvents ();
    oneXEvent ();
    allPendingXEvents ();
  }

  void XTopLevelImpl::readAllEvents (InputDevice *inp) {
    int inputsRead = 0;
    int maxInputs = m_conf->getBusyBug ();
    while (inp->readEvent ()) {
      m_lastInputDevice = inp;
      m_sg->validate ();
      inp->dispatchEvent (this);
      m_needDraw = true;
      inputsRead++;
      if (inputsRead > maxInputs) {
	cerr << "An input device has a readEvent() method that is always "
	     << "returning true" << endl;
	abort ();
      }
    }
  }

  bool XTopLevelImpl::mainLoop () {
    m_wantExit = false;
    m_wantNewWindowSystem = false;
    m_integrating = false;
    m_ballIsGrabbed = false;
    // We'll get an expose event, so we don't need to draw before then.  Before
    // we get the first reconfigure event, the window height is unknown, so
    // we would draw in the wrong place anyway.
    m_needDraw = false;
    // Need to know when the window is deleted, so we can exit before
    // the window system connection is closed from the other end.
    m_deleteWindowAtom = XInternAtom (getDisplay (),
				      "WM_DELETE_WINDOW", False);
    initFungimolState ();
    while (!m_wantExit && !m_wantNewWindowSystem) {
      // Desiderata:
      //
      // if there's an X input, and we can stuff it into the XKeyboardMouse,
      // stuff it in and dispatch.
      //
      // for each select-able device, include the fd into the select mask when
      // we call X, and if the bit is set then do a read on the device, and if
      // the read got something then dispatch on the device.
      // 
      // For each non select-able device, poll.
      // 
      // If there are no devices that require polling, and we are not doing
      // integration, then the select should block.
      // 
      // If there are devices that need polling, and we are not doing
      // integration, and none of the devices that want select are ready, we
      // should wait for the polling interval before continuing, so we don't
      // monopolize the CPU when there is nothing to do.
      // 
      // If there are multiple sources of input available simultaneously, I
      // don't care much how I prioritize them.
      //
      bool waitForSelect = true;
      Display *const display = getDisplay ();
      // It's slightly simpler if there isn't any X input waiting.
      allPendingXEvents ();
      // If there are input devices that want polling, we can't wait on the
      // select.
      if (0 != m_pollInputDevices.size()) waitForSelect = false;
      // If the screen needs updating, we can't wait before updating it.
      if (m_needDraw) waitForSelect = false;
      // If we're animating, we can't wait.
      if (m_integrating || m_sg->anyAnimations ()) waitForSelect = false;
      // Set up and do the select.
      fd_set fds;
      FD_ZERO (&fds);
      FD_SET (ConnectionNumber(display), &fds);
      int maxfd = ConnectionNumber(display);
      for (int i = 0; i < m_selectInputDevices.size(); i++) {
	maxfd =
	  FloatUtil::max (maxfd, m_selectInputDevices [i]->fdForSelect());
	FD_SET (m_selectInputDevices [i]->fdForSelect(), &fds);
      }
      // Have to flush before we select, otherwise we might hang because the
      // commands to create windows we're expecting events from are waiting on
      // the output buffer.
      m_vo->flush ();
      int rc;
      if (waitForSelect) {
	rc = select(maxfd + 1, &fds, NULL, NULL, 0);
      } else {
	struct timeval timeout;
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	rc = select(maxfd + 1, &fds, NULL, NULL, &timeout);
      }
      // Whether we dispatched to any devices based on the select.
      bool didSelect = false;
      if (0 > rc) {
	perror ("Error from select");
	exit (1);
      } else if (0 < rc) {
	didSelect = true;
	if (FD_ISSET (ConnectionNumber (display), &fds)) {
	  oneBlockingXEvent ();
	}
	for (int i = 0; i < m_selectInputDevices.size(); i++) {
	  InputDevice *inp = m_selectInputDevices [i];
	  assert (inp);
	  assert (inp->fdForSelect() >= 0);
	  if (FD_ISSET (inp->fdForSelect(), &fds)) readAllEvents (inp);
	} // end loop over selectinputdevices.
      } // end if with cases on the return code rc.
      for (int i = 0; i < m_pollInputDevices.size (); i++) {
	readAllEvents (m_pollInputDevices [i]);
      }
      if (m_pollInputDevices.size() > 0 &&
	  ! m_integrating && !m_sg->anyAnimations () && !m_needDraw &&
	  !didSelect) {
	// We had nothing to do but poll input devices on the last round, so
	// wait a little bit.  This ensures that if we are only polling, we
	// don't hog 100% of the CPU.
	usleep (m_conf->getPollInterval ());
      }
      if (!m_wantExit && !m_wantNewWindowSystem) {
	if (m_integrating) timeStep ();
	if (m_needDraw || m_sg->anyAnimations ()) draw ();
      }
    }
    closeFungimolState ();
    m_vo->closeDisplay ();
    m_vo = 0;
    if (m_wantExit) {
      return false;
    } else if (m_wantNewWindowSystem) {
      return true;
    } else {
      assert (0 && "Impossible in XTopLevelImpl");
      return false;
    }
  }

  void XTopLevelImpl::doExit () {
    m_wantExit = true;
  }

  void XTopLevelImpl::newWindowSystem () {
    m_wantNewWindowSystem = true;
  }

  SP<SceneGraph> XTopLevelImpl::getSceneGraph () {
    return m_sg;
  }

  SP<const SceneGraph> XTopLevelImpl::getSceneGraph () const {
    return &*m_sg;
  }

  void XTopLevelImpl::setSceneGraph (SP<SceneGraph> sg) {
    m_sg = sg;
    if (m_integrator) {
      m_integrator->setEvaluator (sg->getEvaluator());
    }
  }

  void XTopLevelImpl::setIntegrating (bool b) {
    m_integrating = b;
  }

  bool XTopLevelImpl::getIntegrating () const {
    return m_integrating;
  }

  // FIXME An obsolete interface.  We have input devices other than the
  // mouse.  Using the SelectCube action is better for new code.
  bool XTopLevelImpl::findUnderMouse (int &index, Vec3 &intersectionPoint)
    const
  {
    const KeyboardMouseEvent *const e =
      dynamic_cast <const KeyboardMouseEvent *>
      (&*m_keyboardMouse->getEvent ());
    const int x = e->getX();
    const int y = e->getY();
    const Vec3 vnear =
      m_vo->transformLHScreenToWorld (Vec3 (x, y, m_vo->getNearPlane ()));
    const Vec3 vfar =
      m_vo->transformLHScreenToWorld (Vec3 (x, y, m_vo->getFarPlane ()));
    Dynavec<int> indices;
    Dynavec<Vec3> intersectionPoints;
    m_sg->objectsOnLine (vnear, vfar, indices, intersectionPoints);
    int closestIndex = -1;
    if (indices.size()) {
      Float closestDistance = FLT_MAX;
      // Start the loop at 0 so that if we have only one object under the
      // mouse, and it isn't selectable, then we don't find it under the
      // mouse.
      for (int i = 0; i < indices.size(); i++) {
	Float newDistance = (intersectionPoints[i]-vnear).lengthSquared();
	// FIXME Do not allow selecting objects that intersect vnear.
	if (newDistance < closestDistance &&
	    SelectionManager::isSelectable (ObjectLink (m_sg, indices[i]))) {
	  closestIndex = i;
	  closestDistance = newDistance;
	}
      }
    }
    if (-1 != closestIndex) {
      intersectionPoint = intersectionPoints[closestIndex];
      index = indices [closestIndex];
      return true;
    } else {
      return false;
    }
  }

  XTopLevelImpl::Times XTopLevelImpl::getTimes () const {
    return m_times;
  }

  void XTopLevelImpl::initFungimolState () {
    m_integrator = RungaKuttaFactory::mkRungaKutta ();
    m_integrator->setEvaluator (m_sg->getEvaluator ());
    m_integrator->start (0.0, 0.01);
  }

  void XTopLevelImpl::closeFungimolState () {
    m_integrator = 0;
  }

  XTopLevelImpl::~XTopLevelImpl () {
    TheObjectDrawerFactory::setTheObjectDrawerFactory (0);
  }

  void XTopLevelImpl::draw () {
    if (m_frameStart) {
      m_times.interFrameTime = m_frameStart->millis ();
    }
    m_sg->animateFrame (getCanvas ());
    m_frameStart = NEW (Stopwatch ());
    SP<Stopwatch> start = NEW(Stopwatch ());
    if (m_sg->draw (&*m_vo)) {
      // FIXME Nothing important to do with the return value from draw
      // So get rid of it maybe?
    }
    m_times.lastDrawTime = start->millis ();
    m_needDraw = false;
  }

  void XTopLevelImpl::reshape (int width, int height) {
    m_vo->setWidth (width);
    m_vo->setHeight (height);
  }

  void XTopLevelImpl::timeStep () {
    Float time = 0;
    if (m_integrating && !m_ballIsGrabbed) {
      SP<Stopwatch> totalTime = NEW(Stopwatch ());
      int steps = getConfiguration()->getStepsPerFrame();
      for (int i = 0; i < steps; i++) {
	m_sg->animateTimeStep ();
	SP<Stopwatch> start;
	if (i == steps - 1) start = NEW(Stopwatch ());
	time = m_integrator->observe ();
	if (i == steps - 1) m_times.lastIntTime = start->millis();
      }
      m_times.totalIntTime = totalTime->millis();
      m_needDraw = true;
    } else {
      m_times.lastIntTime = 0;
      m_times.totalIntTime = 0;
    }
  }
  
  class XTopLevelFactory
    : public Factory
  {
  public:
    XTopLevelFactory ()
      : Factory ("XTopLevel")
    {}
    SP<Configuration> defaultConfiguration () const {
      return NEW (XTopLevelConfiguration ());
    }
    SP<Configurable> makeIt (SP<Configuration> c) const {
      SP<XTopLevelConfiguration> tlc =
	dynamic_cast <XTopLevelConfiguration *> (&*c);
      assert (tlc);
      return (NEW (XTopLevelImpl (tlc)));
    }
  };

  static const bool useless =
  (FactoryTable::store ("TopLevel", NEW (XTopLevelFactory ())),
   GenericObjectDrawer::registerWindowSystem ("X"),
   true);
} // namespace
