// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//
#ifndef __XLineDrawer_h__
#include "XLineDrawer.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __Vec3i_h__
#include "Vec3i.h"
#endif

#ifndef __Color_h__
#include "Color.h"
#endif

#ifndef __DepthClipInfo_h__
#include "DepthClipInfo.h"
#endif

#ifndef __BinSet_h__
#include "BinSet.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

namespace {
  typedef Canvas::Depth Depth;
  template <class Pixel>
  struct PointStuff {
    const int m_width;
    const int m_height;
    Depth *const m_depthBuffer;
    Pixel *const m_frameBuffer;
    const int m_frameBufferPixelsPerLine;
    const Pixel m_p;
    const DepthClipInfo &m_dci;
    PointStuff (const Canvas *vo, const Color &c, const DepthClipInfo &dci) 
      : m_width (vo->getWidth ()),
	m_height (vo->getHeight ()),
	m_depthBuffer (vo->getDepthBuffer ()),
	m_frameBuffer ((Pixel *) (vo->getFrameBuffer ())),
	m_frameBufferPixelsPerLine (vo->getFrameBufferPixelsPerLine ()),
	m_p (vo->packColor (c)),
	m_dci (dci)
    {}
  };
  template <class Pixel>
  inline void point (const int x, const int y, const int z,
		     const PointStuff <Pixel> &ps) {
    if (0 <= x && x < ps.m_width
	&& 0 <= y && y < ps.m_height
	// No far Z plane clipping.  The depthShift is adjusted in
	// XDepthClippingDrawer so it can't happen.
	&& 0 <= z) {
      const int screenY = ps.m_height - y - 1;
      Depth *toDepth = ps.m_depthBuffer + screenY * ps.m_width + x;
      if (z < *toDepth) {
	ps.m_dci.binAtScreenCoord (x, y, *toDepth)--;
	 *toDepth = z;
	 ps.m_dci.binAtScreenCoord (x, y, z)++;
	 Pixel *toPix =
	   ps.m_frameBuffer
	   + screenY * ps.m_frameBufferPixelsPerLine
	   + x;
	 *toPix = ps.m_p;
       }
     }
  }
  template <class Pixel>
  void drawHorizontally (const Vec3i &v1, const Vec3i &v2,
			 const PointStuff <Pixel> &ps) {
    Vec3i left;
    Vec3i right;
    if (v1[0] <= v2[0]) {
      left = v1;
      right = v2;
    } else {
      left = v2;
      right = v1;
    }
    assert (left[0] <= right [0]);
    const int dx = right [0] - left [0];
    if (0 == dx) {
      point (left [0], left [1], left [2], ps);
    } else {
      const int dy = right [1] - left [1];
      const int dz = right [2] - left [2];
      for (int deltax = 0; deltax <= dx; deltax++) {
	point (left [0] + deltax,
	       // FIXME Get the next two to round instead of truncate.
	       left [1] + (deltax * dy) / dx,
	       left [2] + (deltax * dz) / dx, ps);
      }
    }
  }
  template <class Pixel>
  void drawVertically (const Vec3i &v1, const Vec3i &v2,
		       const PointStuff <Pixel> &ps)
  {
    // Y increases upward.
    Vec3i top;
    Vec3i bottom;
    if (v1[1] <= v2[1]) {
      bottom = v1;
      top = v2;
    } else {
      bottom = v2;
      top = v1;
    }
    assert (bottom[1] <= top [1]);
    const int dy = top [1] - bottom [1];
    if (0 == dy) {
      point (bottom [0], bottom [1], bottom [2], ps);
    } else {
      const int dx = top [0] - bottom [0];
      const int dz = top [2] - bottom [2];
      for (int deltay = 0; deltay <= dy; deltay++) {
	point (// FIXME Get the next two to round instead of truncate.
	       bottom [0] + (deltay * dx) / dy,
	       bottom [1] + deltay,
	       bottom [2] + (deltay * dz) / dy, ps);
      }
    }
  }
  template <class Pixel>
  void drawLineTemplate (const Canvas *vo, const Vec3i &v1, const Vec3i &v2,
			 const Color &color,
			 const DepthClipInfo &dci) {
    Vec3i w1, w2;
    w1 = v1; w1 [2] += dci.m_clipZ;
    w2 = v2; w2 [2] += dci.m_clipZ;
    PointStuff <Pixel> ps (vo, color, dci);
    if (FloatUtil::abs (w1[0] - w2[0]) > FloatUtil::abs (w1 [1] - w2 [1])) {
      drawHorizontally (w1, w2, ps);
    } else {
      drawVertically (w1, w2, ps);
    }
  }
}

void XLineDrawer::drawLine (const Canvas *vo, const Vec3i &v1, const Vec3i &v2,
			    const Color &color,
			    const DepthClipInfo &dci, const BinSet &finished) {
  (void) finished;
  switch (vo->pixelBytes ()) {
  case 4:
    drawLineTemplate <Canvas::Pixel32> (vo, v1, v2, color, dci);
    break;
  case 2:
    drawLineTemplate <Canvas::Pixel16> (vo, v1, v2, color, dci);
    break;
  default:
    die (String ("Can't draw lines at ") + vo->pixelBytes () +
	 String (" bytes per pixel"));
  }
}

// Next one is like drawLine, except we'll make the line dimmer as
// it is drawn further away, until a line near the background is faded to
// backgroundFade times its original value.  backgroundFade should be more
// than zero (so you can see the line) and less than or equal to 1.
// The fade is determined by the depth of v1.
void XLineDrawer::drawFadingLine
(const Canvas *vo, const Vec3i &v1, const Vec3i &v2,
 const Color &color,
 const DepthClipInfo &dci, const BinSet &finished, Float backgroundFade)
{
  assert (0.0 < backgroundFade && backgroundFade <= 1.0);
  const int myDepth = v1[2]+dci.m_clipZ;
  const Float myDepthFrac = ((Float) myDepth) / dci.m_backgroundDepth;
  assert (0.0 <= myDepthFrac && myDepthFrac < 1.0);
  const Float nearness = 1.0 - myDepthFrac;
  const Float brightness =
    backgroundFade + (1.0 - backgroundFade) * nearness;
  Color myColor = color * brightness;
  drawLine (vo, v1, v2, myColor, dci, finished);
}
