// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __string_h__
#include <string.h>
#define __string_h__
#endif

#ifndef __limits_h__
#include <limits.h>
#define __limits_h__
#endif

#ifndef __Drawer_h__
#include "Drawer.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __PhysicsObject_h__
#include "PhysicsObject.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __SimpleConfiguration_h__
#include "SimpleConfiguration.h"
#endif

#ifndef __Evaluator_h__
#include "Evaluator.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __XObjectDrawer_h__
#include "XObjectDrawer.h"
#endif

#ifndef __ObjectDrawer_h__
#include "ObjectDrawer.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __BinSet_h__
#include "BinSet.h"
#endif

#ifndef __Point_h__
#include "Point.h"
#endif

#ifndef __InvisibilityMarker_h__
#include "InvisibilityMarker.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __DepthClipInfo_h__
#include "DepthClipInfo.h"
#endif

namespace {
  
  typedef Canvas::ScreenRect ScreenRect;
  typedef XObjectDrawer::DepthCount DepthCount;
  const int clipBufWidth = XObjectDrawer::clipBufWidth;
  const int clipBufWidthShift = XObjectDrawer::clipBufWidthShift;
  const int clipBufDepth = XObjectDrawer::clipBufDepth;
  const int clipBufDepthShift = XObjectDrawer::clipBufDepthShift;
  const int widthShift = XObjectDrawer::widthShift;

  struct ObjInfo {
    // Next object in the chain from nearest to farthest.
    int m_next;
    // Integer position of the object.  We could just store the depth at
    // a cost of one extra transform per object, or store nothing at
    // all at the cost of two extra transforms.  Compared with just
    // storing the depth, storing the entire coordinates saves ~4 ms/frame.
    // With memorized coordinates, sample takes 637-638 ms/frame.
    // Without memorized coordinates, takes 641-642 ms/frame.
    // So keep the memorized coordinates. 
    Vec3i m_pos;
    int m_radius;
    // The nearest depth of the object.  That is, the z coordinate
    // from m_pos, minus the radius of the graphics bounding sphere,
    // minus the appropriate constant so that the minimum m_depth is 0.
    int m_depth;
    // Whether we are going to draw the object.
    bool m_draw;
    // Return the z bin for this object, assuming the other fields
    // have their proper values.
    inline int zBin (DepthClipInfo &dci) {
      // If we're doing Z clipping, then m_depth can be negative, but the z bin
      // should never be negative.
      return FloatUtil::max (m_depth, 0) >> dci.m_depthShift;
    }
    // Return a Rect that says which bins in the x-y direction (not z) may be
    // drawn into by the object.
    inline Rect xyBins (const DepthClipInfo &dci,
			const Canvas *vo) const {
      (void) dci; // dci used to have an m_widthShift field we used here.
      const int x = m_pos [0];
      const int y = m_pos [1];
      // Give one pixel of slack in the radius.  Balls seem to need it.
      // Imagine: you have a center pixel, and a radius of 2.  So you draw two
      // pixels on the left side of the center pixel, and two on the right
      // side, so now you have a diameter of 5, counting the center pixel.  We
      // could squeeze it to a diameter of 4, but then we have to be careful to
      // leave out the pixel on the correct end of the ball, otherwise we still
      // overwrite our boundaries.  Better to just give a pixel of slack.
      const int radius = m_radius + 1;
      return Rect (FloatUtil::max (0, (x-radius) >> widthShift),
		   FloatUtil::max (0, (y-radius) >> widthShift),
		   (FloatUtil::min (vo->getWidth () - 1, (x+radius))
		    >> widthShift) + 1,
		   (FloatUtil::min (vo->getHeight () - 1, (y+radius))
		    >> widthShift) + 1);
    }
  };
  
  // Return true if the rect r could be grown in the given direction
  // by adding elements in b.
  bool growLeft (BinSet &b, Rect &r) {
    if (r.minx > 0) {
      Rect newLeft (r.minx-1, r.miny, r.minx, r.maxy);
      if (b.isElement (newLeft)) {
	r = Rect (r.minx-1, r.miny, r.maxx, r.maxy);
	return true;
      }
    }
    return false;
  }

  bool growDown (BinSet &b, Rect &r) {
    if (r.miny > 0) {
      Rect newDown (r.minx, r.miny-1, r.maxx, r.miny);
      if (b.isElement (newDown)) {
	r = Rect (r.minx, r.miny-1, r.maxx, r.maxy);
	return true;
      }
    }
    return false;
  }

  bool growRight (BinSet &b, Rect &r) {
    if (r.maxx < clipBufWidth) {
      Rect newRight (r.maxx, r.miny, r.maxx + 1, r.maxy);
      if (b.isElement (newRight)) {
	r = Rect (r.minx, r.miny, r.maxx + 1, r.maxy);
	return true;
      }
    }
    return false;
  }

  bool growUp (BinSet &b, Rect &r) {
    if (r.maxy < clipBufWidth) {
      Rect newUp (r.minx, r.maxy, r.maxx, r.maxy + 1);
      if (b.isElement (newUp)) {
	r = Rect (r.minx, r.miny, r.maxx, r.maxy + 1);
	return true;
      }
    }
    return false;
  }

  // Convert b to a bunch of nonoverlapping Rect's.  Try to make the
  // result a small number of large rects.  When we're done, b is
  // empty. 
  void convertToRects (BinSet &b,  Dynavec <Rect> &rects) {
    rects.extendTo (0);
    while (! b.isEmpty ()) {
      Point p = b.anyElement ();
      Rect r (p.x, p.y, p.x+1, p.y+1);
      // Because I have |'s here instead of ||'s, we'll tend to get
      // squarish rects instead of long narrow ones.
      //while (growLeft (b, r) | growUp (b, r) | growDown (b, r) |
      //growRight (b,r)) {
      // Do nothing.
      //};
      // Long horizontal runs are good.
      while (growLeft (b, r));
      while (growRight (b, r));
      while (growUp (b, r));
      while (growDown (b, r));
      rects.push (r);
      b.remove (r);
    }
  }

  class DepthSort {
    // We'll have many linked lists, one per depth bin.  m_start [bin]
    // gives the position of the object in that bin.  This is
    // logically local to depthSort, but I put it here so I don't have
    // to repeatedly allocate.
    Dynavec <int> m_start;
    // Pointers to the end of each linked list, so we can efficiently
    // string them together.  This is logically local to depthSort,
    // but I put it here so I don't have to repeatedly allocate.
    Dynavec <int *> m_end;
  public:
    DepthSort () {
      m_start.extendTo (clipBufDepth);
      m_end.extendTo (clipBufDepth);
    }
    // Returns a pointer to the head of the list.
    int sortIntoBins (DepthClipInfo &dci, Dynavec <ObjInfo> &info) {
      // Surprisingly enough, we don't need to initialize the start
      // array.  Do it anyway for debug code, for determinism when debugging.
      int bins = m_start.size ();
#ifndef NDEBUG
      for (int i = 0; i < bins; i++) {
	m_start [i] = -1;
      }
#endif
      // The last address of each list is the start address, since
      // the lists have to start out empty.
      assert (m_end.size() == bins);
      for (int i = 0; i < bins; i++) {
	m_end [i] = &m_start [i];
      }
      // Put each object in its bin.  Add it to the end of the
      // linked list.
      {
	int size = info.size();
	for (int i = 0; i < size; i++) {
	  ObjInfo &oi = info [i];
	  if (oi.m_draw) {
	    int bin = oi.zBin (dci);
	    *m_end [bin] = i;
	    m_end [bin] = &info[i].m_next;
	  }
	}
      }
      // Put a known value at the end of the last list.
      *m_end [bins-1] = -1;
      // Concatenate all of the linked lists.  
      for (int i = bins-2; i >=0; i--) {
	*m_end [i] = m_start [i+1];
      }
      return m_start [0];
    }
  };

  class XDepthClippingDrawer
    : public Drawer
  {
    // Stuff about each object.
    Dynavec <ObjInfo> m_info;
    // Stuff for depth-sorting, partitioned into its own private class
    // because nothing else here needs to know.
    DepthSort m_ds;
    // Number of garbage collections on the scene graph.  We may have
    // to resize m_next, m_start, and m_end when the size of the scene
    // graph changes.
    int m_generation;
    // The dirty bins for the back buffer.
    BinSet m_dirty;
    // The bins we have to draw.  That is, the bins that have graphics in this
    // time.
    BinSet m_toDraw;
    // Used briefly to keep track of the recently finished bins.
    BinSet m_finished;
    // The bins we have to render this time.  We have to render a bin if either
    // we draw something into it, or it was dirty last time.  If it was dirty
    // last time and not drawn this time we have to copy the black pixels, and
    // that is one kind of rendering.
    BinSet m_toRender;
    // Width and height last time we drew.  We ignore the old dirty
    // rectangle if the screen changed size.
    int m_oldWidth, m_oldHeight;
    // A bunch of rects used for various things.  A member variable so we don't
    // have to reallocate.
    Dynavec <Rect> m_rects;
    // y for the depth clip buffer increases as we go up on the screen.
    Dynavec <DepthCount> m_depthClipBuffer;
    DepthClipInfo m_dci;
    // Logically local to fillInfo.  Accumulates a list of InvisibilityMarker's
    // so we don't have to scan through the scene twice for them.
    // FIXME Sorting the scene would help here.
    // FIXME The scene graph could maintain and return a list of
    // InvisibilityMarkers much more cheaply than we can rediscover it here, at
    // the cost of more inter-class dependency.
    Dynavec <int> m_invisibilityMarkers;
    // This starts out with the number of pixels available in each bin, and
    // counts down toward zero as the pixels are observed to be drawn.
    int m_unDonePixels [clipBufWidth] [clipBufWidth];

    // Take a pass over the objects, filling in the m_info array,
    // m_dci, and m_toRender.  Also set maxZbin to one more than the number of
    // the last occupied depth bin.
    void fillInfo (SceneGraph *sg, const Canvas *vo, int &maxZbin)
    {
      const Dynavec <Float> &sv = sg->getEvaluator()->stateVec();
      const Float *v = &*sv;
      int minZ = INT_MAX;
      int maxZ = INT_MIN;
      const int width = vo->getWidth ();
      const int height = vo->getHeight ();
      m_toDraw.empty ();
      m_invisibilityMarkers.extendTo (0);
      const int size = sg->maxIndex();
      // Gather up the InvisibilityMarkers, and mark everyone as tentatively
      // worth drawing.
      for (int i = 0; i < size; i++) {
	ObjInfo &oi = m_info [i];
	// Could save some cycles here by having
	// InvisibilityMarker::staticClassId == sg->object(i)->classId ()
	// at the cost of disallowing useful subclasses of InvisibilityMarker.
	if (sg->object(i)->isInstanceOf (InvisibilityMarker::staticClassId)) {
	  m_invisibilityMarkers.push (i);
	} else {
	  oi.m_draw = true;
	}
      }
      // Don't draw the objects that are linked to by InvisibilityMarker's that
      // are invisible.
      {
	const int *const ims = &*m_invisibilityMarkers;
	const int imsize = m_invisibilityMarkers.size();
	SP<LinkManager> lm = sg->getLinkManager();
	for (int i = 0; i < imsize; i++) {
	  if (!((InvisibilityMarker *)(sg->object(ims[i])))->getVisible ()) {
	    int neighborCount;
	    const int *neighbors;
	    lm->getLinks (ims[i], neighborCount, neighbors);
	    for (int j = 0; j < neighborCount; j++) {
	      const int n = neighbors [j];
	      if (-1 != n) {
		m_info[n].m_draw = false;
	      }
	    }
	  }
	}
      }
      // Think further about the objects that we haven't yet decided not to
      // draw.
      for (int i = 0; i < size; i++) {
	ObjInfo &oi = m_info [i];
	if (oi.m_draw) {
	  const PhysicsObject *const o = sg->object(i);
	  const ObjectDrawer *const od = o->getObjectDrawer ();
	  if (!od->isVisible()) {
	    // Make sure invisible objects don't cause dirty bins.
	    oi.m_draw = false;
	  } else {
	    Vec3i pos =
	      vo->iTransformWorldToLHScreen
	      (o->boundingSphereCenter (v+sg->objectStateIndex(i)));
	    const int radius = 
	      FloatUtil::ceiling (o->getObjectDrawer()
				  ->graphicsBoundingSphereRadius(i, sg)
				  * vo->getScale ());
	    oi.m_radius = radius;
	    oi.m_pos = pos;
	    const int x = pos [0];
	    const int y = pos [1];
	    const int z = pos [2];
	    if (x + radius >= 0 && x - radius < width &&
		y + radius >= 0 && y - radius < height &&
		z + radius >= 0) {
	      oi.m_draw = true;
	      minZ = FloatUtil::min (minZ, z - radius);
	      maxZ = FloatUtil::max (maxZ, z + radius);
	      // We will draw it, so add some dirty bins.
	      m_toDraw.insert (oi.xyBins (m_dci, vo));
	    } else {
	      // Don't draw due to viewport clipping.
	      oi.m_draw = false;
	    }
	  }
	}
      }
      if (INT_MAX == minZ) { 
	m_dci.m_clipZ = minZ; // Empty scene.
	maxZbin = 0;
	return;
      } else if (minZ < 0) {
	// The near plane is at z = 0, so if minZ < 0 then we'll really draw
	// those pixels at minZ = 0.
	minZ = 0;
      }
      m_dci.m_clipZ = -minZ;
      for (int i = 0; i < size; i++) {
	ObjInfo &oi = m_info [i];
	// Testing oi.m_draw isn't necessary if we aren't bothered by
	// integer overflow or reading unintialized memory.
	if (oi.m_draw) {
	  oi.m_pos [2] -= minZ;
	  oi.m_depth = oi.m_pos [2] - oi.m_radius;
	}
      }
      // Allocate one depth for the background plane.
      maxZ++;
      m_dci.m_backgroundDepth = maxZ - minZ;
      // Don't let depthShift start at 0, since the whole point is
      // that the depthClipBuffer is faster to work with than the real
      // depth buffer.
      {
	int depth = maxZ - minZ;
	assert (depth >= 0);
	int depthShift = 3;
	while ((depth >> depthShift) >= clipBufDepth) depthShift++;
	m_dci.m_depthShift = depthShift;
	maxZbin = (depth >> depthShift) + 1;
      }
    }

    // How many (x,y) coordinates are on the screen that fall into the
    // given bin.
    int depthBinPixels (int xbin, int ybin, const Canvas *vo) {
      const int width = vo->getWidth ();
      const int height = vo->getHeight ();
      const int minX = xbin << widthShift;
      const int maxX = FloatUtil::min ((xbin + 1) << widthShift, width);
      const int minY = ybin << widthShift;
      const int maxY = FloatUtil::min ((ybin + 1) << widthShift, height);
      return (maxX - minX) * (maxY - minY);
    }

    // Clear the colors for the dirty bins from the color buffer, and set
    // m_toRender to the part of the screen we need to render to eventually
    // propagate this to the screen.
    void clearDirtyColors (const Canvas *vo) {
      if (vo->getWidth () == m_oldWidth && vo->getHeight () == m_oldHeight) {
	m_toRender = m_dirty;
	convertToRects (m_dirty, m_rects);
	for (int i = 0; i < m_rects.size (); i++) {
	  vo->clearColorRect (m_rects [i].screenRect (vo, m_dci));
	}
      } else {
	m_oldWidth = vo->getWidth ();
	m_oldHeight = vo->getHeight ();
	vo->clearColorRect (ScreenRect (0, 0, m_oldWidth, m_oldHeight));
	// m_toRender has to be the whole screen.
	m_toRender.empty ();
	m_toRender.insert
	  (Rect (0, 0,
		 ((m_oldWidth - 1) >> widthShift) + 1,
		 ((m_oldHeight - 1) >> widthShift) + 1));
      }
    }
  public:
    XDepthClippingDrawer ()
      : m_generation (-1),
	m_dirty (BinSet ()),
	m_toDraw (BinSet ()),
	m_finished (BinSet ()),
	m_toRender (BinSet ()),
	m_oldWidth (-1)
    {
      m_depthClipBuffer.extendTo (clipBufDepth * clipBufWidth * clipBufWidth);
      m_dci.m_buffer = &*m_depthClipBuffer;
    }
    bool draw (SceneGraph *sg, const Canvas *a_vo);
  };

  bool XDepthClippingDrawer::draw (SceneGraph *sg, const Canvas *vo) {
    {
      const int size = sg->maxIndex();
      if (SceneGraph::garbageCollectionCount() != m_generation ||
	  size != m_info.size ()) {
	m_generation = SceneGraph::garbageCollectionCount();
	m_info.extendTo (size);
      }
    }
    {
      const int maxsize = 1 << (widthShift + clipBufWidthShift);
      static bool griped = false;
      if (vo->getWidth () > maxsize || vo->getHeight () > maxsize) {
	if (!griped) {
	  griped = true;
	  message (String ("Your window is too big.  Please resize it to ")+
		   maxsize+" by "+maxsize+" or less");
	}
	return false;
      } else {
	if (griped) {
	  griped = false;
	  message ("Resuming drawing");
	}
      }
    }
    // Next one gives m_toRender an initial value for this frame, either
    // derived from m_dirty or from scratch if we give up on carefully
    // interpreting the dirty buffer.
    clearDirtyColors (vo);
    int maxZbin;
    // Next one sets up m_toDraw.
    fillInfo (sg, vo, maxZbin);
    // Render if it was dirty before or we will draw to it this time.
    // m_toRender is not updated again this frame.
    m_toRender.doUnion (m_toDraw);
    // Clear out the depth clip buffer, for the bins that we are going to
    // draw into.  Also set up m_unDonePixels.
    {
      int drawSize = m_toDraw.numElements ();
      for (int i = 0; i < drawSize; i++) {
	Point p = m_toDraw.getElement (i);
	DepthCount *binStart = &m_dci.binAtBinCoord (p.x, p.y, 0);
	memset ((void *) binStart, 0, maxZbin * sizeof (DepthCount));
	m_unDonePixels [p.x] [p.y] = depthBinPixels (p.x, p.y, vo);
      }
    }
    // It is dirty next time if we drew it this time.  Compute this now because
    // we're about to destroy m_toDraw.  m_dirty is not updated again this
    // frame.
    m_dirty = m_toDraw;
    // Clear out the depth buffer, destroying m_toDraw. 
    convertToRects (m_toDraw, m_rects);
    for (int i = 0; i < m_rects.size(); i++) {
      vo->clearDepthRect (m_rects[i].screenRect (vo, m_dci),
			  m_dci.m_backgroundDepth);
    }
    {
      // Sort approximately by depth.  "i" will be the number of the first
      // object in the list that is sorted by depth.
      int i = m_ds.sortIntoBins (m_dci, m_info);
      // Draw and depth clip.
      const Dynavec <Float> &sv = sg->getEvaluator()->stateVec();
      const Float *v = &*sv;
      // We reuse m_toDraw to hold the xy bins that have been dirtied but not
      // filled by this frame so far.
      m_toDraw.empty ();
      for (int depthBin = 0; depthBin < maxZbin; depthBin ++) {
	for (;
	     i != -1 && m_info[i].zBin (m_dci) == depthBin;
	     i = m_info[i].m_next) {
	  Rect r = m_info[i].xyBins (m_dci, vo);
	  for (int binY = r.miny; binY < r.maxy; binY++) {
	    for (int binX = r.minx; binX < r.maxx; binX++) {
	      assert (m_toRender.isElement (binX, binY));
	      assert (m_unDonePixels [binX] [binY] >= 0);
	      if (m_unDonePixels [binX] [binY]) {
		// Some pixels that might be drawn are not done, so we have
		// to draw.
		const PhysicsObject *obj = sg->object (i);
		assert (obj);
		const ObjectDrawer *od = obj->getObjectDrawer();
		assert (dynamic_cast <const XObjectDrawer *> (od));
		const XObjectDrawer *xod =
		  static_cast <const XObjectDrawer *> (od);
		xod->draw (v+sg->objectStateIndex(i), vo,
			   m_info[i].m_pos, m_dci, m_finished,i, sg);
		m_toDraw.insert (r);
		goto done;
	      }
	    }
	  }
	done: ;
	}
	// We've drawn everything in the depth bin.  Update m_unDonePixels
	// for all of the things we drew.
	// m_finished will hold the bins that we have dirtied that are now
	// finished.  Can't remove them directly from toDraw, since that
	// would cause things in toDraw to move around as we're trying to
	// enumerate them.
	m_finished.empty ();
	{
	  int size = m_toDraw.numElements();
	  for (int elt = 0; elt < size; elt++) {
	    Point p = m_toDraw.getElement (elt);
	    int &undone = m_unDonePixels [p.x] [p.y];
	    undone -= m_dci.binAtBinCoord (p.x, p.y, depthBin);
	    if (0 == undone) {
	      m_finished.insert (p);
	    }
#ifndef NDEBUG
	    int total = 0;
	    for (int bin = 0; bin <= depthBin; bin++) {
	      total += m_dci.binAtBinCoord (p.x, p.y, bin);
	    }
	    assert (total + m_unDonePixels [p.x] [p.y] ==
		    depthBinPixels (p.x, p.y, vo));
#endif
	  }
	}
	{ // If they're finished, then take them out of toDraw.
	  int size = m_finished.numElements ();
	  for (int elt = 0; elt < size; elt++) {
	    m_toDraw.remove (m_finished.getElement (elt));
	  }
	}
	// Either we're at the end, or the next object has a larger bin.
	// If this isn't true, then sortIntoBins isn't doing its job.
	assert (-1 == i || m_info[i].zBin (m_dci) > depthBin);
      } // End loop over bins.
    } // End scope for i, v, sv.
    // Render.
    convertToRects (m_toRender, m_rects);
    for (int i = 0; i < m_rects.size(); i++) {
      vo->drawRect (m_rects [i].screenRect (vo, m_dci));
    }
    vo->sync ();
    if (vo->doubleBuffered ()) {
      vo->swapBuffers ();
    }
    return false;
  } // End of draw.
  
  class SortingDrawerFactory : public Factory {
  public:
    SortingDrawerFactory () : Factory ("XDepthClippingDrawer") {
    }
    SP<Configuration> defaultConfiguration () const {
      return NEW (SimpleConfiguration ());
    }
    SP<Configurable> makeIt (SP<Configuration> c) const {
      (void) c;
      return NEW (XDepthClippingDrawer ());
    }
  };

  static const bool useless =
  (FactoryTable::store ("Drawer", NEW (SortingDrawerFactory ())),
   true);

}
