// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __XCanvas_h__
#include "XCanvas.h"
#endif

// For exec.
#ifndef __unistd_h__
#include <unistd.h>
#define __unistd_h__
#endif

// For strerror.
#ifndef  __string_h__
#include <string.h>
#define __string_h__
#endif

// For errno.
#ifndef __errno_h__
#include <errno.h>
#define __errno_h__
#endif

// Next one defines, among other things, Bool, which is used in Xutil.h.
#ifndef __X11_Xlib_h__
#include <X11/Xlib.h>
#define __X11_Xlib_h__
#endif

// For XVisualInfo.  Tsk tsk, this uses Bool without making any effort to
// ensure that it is defined.
#ifndef __X11_Xutil_h__
#include "X11/Xutil.h"
#define __X11_Xutil_h__
#endif

// malloc.
#ifndef __stdlib_h__
#include <stdlib.h>
#define __stdlib_h__
#endif

// memset.
#ifndef __string_h__
#include <string.h>
#define __string_h__
#endif

// MIT Shared Memory extension code follows the document, which is here on my
// machine:
// /usr/src/RPM/BUILD/XFree86-3.3.5/xc/doc/hardcopy/Xext/mit-shm.PS.gz
// I had to install the XFree86 source rpm to get this, and then tell rpm to
// unpack it.  "rpm -bp" unpacks sources for a source rpm.
#ifdef TRY_MIT_SHM
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#endif

// Double Buffering extension is documented on my machine at:
// /usr/src/RPM/BUILD/XFree86-3.3.5/xc/doc/man/Xext/dbe/DBE.man.
#ifdef TRY_DBE
#include <X11/extensions/Xdbe.h>
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __BoundingSphere_h__
#include "BoundingSphere.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

// For cerr.
#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __Matrix4_h__
#include "Matrix4.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __Color_h__
#include "Color.h"
#endif

#ifndef __XCanvasConfiguration_h__
#include "XCanvasConfiguration.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

typedef Canvas::LongestPixel LongestPixel;

struct XCanvas::XCanvasData {
  int m_bytesPerPixel;
  mutable int m_viewPointChangeCount;
  int m_width;
  int m_height;
  Float m_scale;
  Vec3 m_origin;
  Quaternion m_rotation;
  Display *m_display;
  XVisualInfo m_xvi;
  bool m_useShm;
  bool m_useDbe;
  Matrix4 m_worldToScreen;
  Matrix4 m_screenToWorld;
  mutable bool m_valid;
  XImage *m_frameBuffer;
  Depth *m_depthBuffer;
#ifdef TRY_MIT_SHM
  XShmSegmentInfo m_shminfo;
#endif
#ifdef TRY_DBE
  XdbeBackBuffer m_backBuffer;
#else
  Window m_backBuffer;
#endif
  Window m_window;
  GC m_gc;

  inline int pixelBytes () const {
    return m_bytesPerPixel;
  }
  inline void invalidate () {
    m_viewPointChangeCount++;
    m_valid = false;
  }
  void findVisual ();
  void validate ();
  void freeVisualBuffers ();
  XCanvasData (XCanvasConfiguration *xcc);
  void drawRect (const ScreenRect &r) const;
  inline int getFrameBufferBytesPerLine () const {
    return m_frameBuffer->bytes_per_line;
  }
  inline void checkRect (const ScreenRect &r) const {
    (void) r;
    assert (r.x >= 0 && r.y >= 0 &&
	    r.x + r.width <= m_width &&
	    r.y + r.height <= m_height);
  }
  void closeDisplay ();
#ifdef TRY_DBE
  bool XCanvas::XCanvasData::findDBVisual ();
#endif
  bool XCanvas::XCanvasData::findNonDBVisual ();
  ~XCanvasData () {
    freeVisualBuffers ();
    closeDisplay ();
  }
};

void XCanvas::XCanvasData::closeDisplay () {
  invalidate();
  if (m_display) {
    // Because freeVisualBuffers can call XShmDetach, we have to do
    // it before calling XCloseDisplay.
    freeVisualBuffers();
    // If we didn't close the display, we should unmap and destroy the
    // window.
    assert (m_gc);
    XFreeGC (m_display, m_gc);
    m_gc = 0;
    XCloseDisplay (m_display); 
    m_display = 0;
  }
}

void XCanvas::drawFrame () const {
  drawRect (ScreenRect (0, 0, getWidth(), getHeight()));
  if (m_data->m_useDbe) {
    swapBuffers ();
  }
  // Have to sync here, not flush, because if we are using shared memory and we
  // start to reuse the pixmap before the operation is finished, it can finish
  // with the wrong bits.
  sync ();
}

void XCanvas::XCanvasData::drawRect (const ScreenRect &r) const {
  checkRect (r);
#ifdef TRY_MIT_SHM
  if (m_useShm) {
    // This saves us 45 ms/frame, full screen, over using XPutImage.
    XShmPutImage (m_display, m_backBuffer, m_gc, m_frameBuffer, r.x,
		  r.flipY (m_height), r.x, r.flipY (m_height),
		  r.width, r.height, False);
  } else {
#endif
    // Next one takes 27 ms for a 600x600 image.
    XPutImage (m_display, m_backBuffer, m_gc, m_frameBuffer, r.x,
	       r.flipY (m_height), r.x, r.flipY (m_height),
	       r.width, r.height);
#ifdef TRY_MIT_SHM
  }
#endif
}

void XCanvas::drawRect (const ScreenRect &r) const {
  m_data->drawRect (r);
}

void XCanvas::clearFrame () const {
  int width = getWidth ();
  int height = getHeight ();
  assert (width > 0);
  assert (height > 0);
  clearColorRect (ScreenRect (0, 0, width, height));
  clearDepthRect (ScreenRect (0, 0, width, height), (Depth) -1);
}

int XCanvas::pixelBytes () const {
  return m_data->pixelBytes ();
}

void XCanvas::clearColorRect (const ScreenRect &r) const
{
  m_data->validate ();
  m_data->checkRect (r);
  const int bytesToJump = m_data->getFrameBufferBytesPerLine ();
  char *rowStart =
    ((char *) getFrameBuffer() + r.x * pixelBytes ()) +
    r.flipY (getHeight()) * bytesToJump;
  const int bytesToClear = r.width * pixelBytes ();
  for (int row = 0; row < r.height; row++) {
    memset (rowStart, 0, bytesToClear);
    rowStart += bytesToJump;
  }
}

void XCanvas::clearDepthRect (const ScreenRect &r, Depth d) const {
  m_data->checkRect (r);
  XCanvas::Depth *depthBuffer = getDepthBuffer();
  assert (depthBuffer);
  Depth *start =
    depthBuffer + r.x + r.flipY (m_data->m_height) * m_data->m_width;
  Depth *end = start + r.height * m_data->m_width;
  for (; start < end; start += m_data->m_width) {
    Depth *rowEnd = start + r.width;
    for (Depth *rowStart = start; rowStart < rowEnd; rowStart++) {
      // Unrolling should help here.  Four shorts per double, perhaps?
      *rowStart = d;
    }
  }
}

void XCanvas::sync () const {
  XSync (m_data->m_display, False);
}

void XCanvas::flush () const {
  XFlush (m_data->m_display);
}

namespace {
  // Whether we know how to render to a given number of bits per pixel.
  bool isGoodDepth (const int depth) {
    return depth == 32 || depth == 24 || depth == 16 || depth == 15;
  }
}

#ifdef TRY_DBE
// Find a double-buffered visual.
// The interface to XdebScreenVisualInfo does not resemble at all any of the
// non-double-buffered functions for searching for visuals, so no shared code
// between findDBVisual and findNonDBVisual.  :-(.
bool XCanvas::XCanvasData::findDBVisual () {
  int num_screens = 1;
  bool result = false;
  Drawable d = DefaultRootWindow (m_display);
  XdbeScreenVisualInfo *svi = XdbeGetVisualInfo (m_display, &d,
						 &num_screens);
  for (int i = 0; i < svi->count; i++) {
    VisualID visual = svi->visinfo[i].visual;
    // FIXME With another loop we could pick the fastest double buffered visual
    // That is, the one with the greatest svi->visinfo[i].perflevel.
    // But with no hardware assisted double buffering implementation on hand,
    // it's untestable.
    if (isGoodDepth (svi->visinfo[i].depth)) {
      XVisualInfo templt; // template is a C++ keyword.
      templt.visualid = visual;
      int ret = 0; // return is a C++ keyword.
      XVisualInfo *xvi = XGetVisualInfo (m_display, VisualIDMask,
					 &templt, &ret);
      assert (1 == ret);
      assert (xvi);
      if (TrueColor == xvi->c_class) {
	m_xvi = *xvi;
	result = true;
	XFree (xvi);
	break;
      } else {
	XFree (xvi);
      }
    }
  }
  XdbeFreeVisualInfo (svi);
#ifndef NDEBUG
  if (result) {
    message (String ("Found a double buffered visual with depth ") +
	     m_xvi.depth);
  } else {
    message ("Found no double buffered visual.");
  }
#endif
  return result;
}
#endif

// Return true if we found a non depth-buffered visual.
bool XCanvas::XCanvasData::findNonDBVisual () {
  Status status =
    XMatchVisualInfo (m_display, DefaultScreen (m_display),
		      32, TrueColor, &m_xvi) ||
    XMatchVisualInfo (m_display, DefaultScreen (m_display),
		      24, TrueColor, &m_xvi) ||
    XMatchVisualInfo (m_display, DefaultScreen (m_display),
		      16, TrueColor, &m_xvi) ||
    XMatchVisualInfo (m_display, DefaultScreen (m_display),
		      15, TrueColor, &m_xvi);
  return 0 != status;
}

// Set m_xvi to the visual we want, or exit if we can't find a good visual, or
// exec another version of fungimol if we think it would do a better job.
void XCanvas::XCanvasData::findVisual () {
  bool done = false;
#ifdef TRY_DBE
  if (m_useDbe) {
    if (XCanvas::XCanvasData::findDBVisual ()) {
      done = true;
    } else {
      m_useDbe = false;
    }
  }
#endif
  if (!done) {
    if (XCanvas::XCanvasData::findNonDBVisual ()) {
      done = true;
    }
  }
  if (!done) {
    cerr << "Can't find a 32, 24, 16, or 15 bit per pixel TrueColor "
	 << "visual on display "
	 << XDisplayName (0) << endl;
    exit (1);
  }
  // Round up to the next byte.
  m_bytesPerPixel = (m_xvi.depth + 7) / 8;
  // But if it's odd, make it even.
  m_bytesPerPixel += m_bytesPerPixel % 2;
}

#ifdef TRY_DBE
namespace {
  // FIXME Let the user tweak mySwapAction.
  // since different swap actions may have optimal performance on different
  // hardware.  Requires the XCanvas to report the swap action to the code
  // drawing to the canvas, and then that code does the right thing with it.
  // If this is XdbeUntouched, it's really slow.
  const XdbeSwapAction mySwapAction =
  // XdbeUntouched
  XdbeCopied
  // XdbeUndefined
  ;
}
#endif

XCanvas::XCanvasData::XCanvasData (XCanvasConfiguration *xcc)
  : m_viewPointChangeCount (0),
    // m_width and m_height are initialized below, immediately after we create
    // the window.
    // This scale causes the differential gear to approximately be the full
    // height screen, a convenient test case.
    m_scale (17.27),
    m_origin (Vec3 (-1, -1, 128)),
    m_useDbe (false),
    m_valid (false),
    m_frameBuffer (0),
    m_depthBuffer (0)
{
  m_display = XOpenDisplay (0);
  if (!m_display) {
    cerr << "Could not open display " << XDisplayName (0) << endl;
    exit (1);
  }
  { // Decide whether to try to use shared memory.
    static bool griped = 0;
    m_useShm = xcc->getUseShm ();
#ifdef TRY_MIT_SHM
    if (m_useShm) {
      if (!XShmQueryExtension (m_display)) {
	m_useShm = false;
	if (!griped) {
	  griped = true;
	  cerr << "Can't use MIT Shared Memory extension because the "
	       << "server does not support it," << endl
	       << "or perhaps it is not a local display."<< endl;
	}
      } 
    }
#else
    if (m_useShm) {
      m_useShm = false;
      if (!griped) {
	griped = true;
	cerr << "Can't use MIT Shared Memory extension because it was "
	     << "compiled out."
	     << endl
	     << "Use -DTRY_MIT_SHM." << endl;
      }
    }
#endif
  }
  { // Decide whether to use double buffering.
    static bool griped = 0;
    m_useDbe = xcc->getUseDbe ();
#ifdef TRY_DBE
    if (m_useDbe) {
      int major, minor;
      if (!XdbeQueryExtension (m_display, &major, &minor)) {
	m_useDbe = false;
	if (!griped) {
	  griped = true;
	  cerr << "We will not be using the double buffering extension "
	       << "because " << endl << "the server does not support it."
	       << endl;
	}
      }
    }
#else
    if (m_useDbe) {
      m_useDbe = false;
      if (!griped) {
	griped = true;
	cerr << "We will not be using the double buffering extension because"
	     << endl << "it was compiled out.  Use -DTRY_DBE." << endl;
      }
    }
#endif
  }
  // Set m_xvi to the visual we want, or exit if we can't find a good visual.
  findVisual ();
  { // Create and map the window
    unsigned long mask = CWEventMask | CWBackPixel;
    XSetWindowAttributes attr;
    attr.background_pixel = 0;
    attr.event_mask = StructureNotifyMask | ExposureMask | PointerMotionMask
      | KeyPressMask | KeyReleaseMask
      | ButtonPressMask | ButtonReleaseMask;
    // Nothing highly significant in these numbers, it's just the shape of the
    // convenient rendering benchmark I use.  This fits on my screen next to an
    // 80-column emacs window.
    // static const int initialWidth = 702;
    // static const int initialHeight = 926;
    // These smaller numbers fit my portable's smaller screen.
    static const int initialWidth = 525;
    static const int initialHeight = 670;
    // Sometimes I want to have a small enough width and height so I see
    // meaningful things happen as I single-step through the code.
    // static const int initialWidth = 20;
    // static const int initialHeight = 40;
    m_window = XCreateWindow
      (m_display,
       RootWindow (m_display, DefaultScreen (m_display)),
       0, 0, initialWidth, initialHeight, 0,
       m_xvi.depth, InputOutput, m_xvi.visual, mask, &attr);
#ifdef TRY_DBE
    if (m_useDbe) {
      m_backBuffer = XdbeAllocateBackBufferName (m_display, m_window,
						 mySwapAction);
    } else {
#endif
      m_backBuffer = m_window;
#ifdef TRY_DBE
    }
#endif
    // We may draw the screen before we get the first ConfigureNotify event, so
    // we need the width and height of the screen to be correct from the very
    // beginning.
    m_width = initialWidth;
    m_height = initialHeight;
    Atom deleteWindowAtom = XInternAtom (m_display, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(m_display, m_window, &deleteWindowAtom, 1);
    XmbSetWMProperties (m_display, m_window, "Fungimol",
			"Fungimol", 0, 0, 0, 0, 0);
    XMapWindow (m_display, m_window);
  }
  {
    XGCValues values;
    values.function = GXcopy;
    m_gc = XCreateGC (m_display, m_window, GCFunction, &values);
  }
}

Float XCanvas::getNearPlane () const {
  return 0;
}

Float XCanvas::getFarPlane () const {
  return (Float) ((Depth) (-1));
}

void XCanvas::XCanvasData::freeVisualBuffers () {
  if (m_frameBuffer) {
#ifdef TRY_MIT_SHM
    if (m_useShm) {
      XShmDetach (m_display, &m_shminfo);
      m_frameBuffer->data = (char*)NULL;
      shmdt (m_shminfo.shmaddr);
    }
#endif
    XDestroyImage (m_frameBuffer);
    m_frameBuffer = 0;
    assert (m_depthBuffer);
    delete [] m_depthBuffer;
  }
}

XCanvas::XCanvas (XCanvasConfiguration *xcc)
  : m_data (NEW (XCanvasData (xcc)))
{}

XCanvas::~XCanvas () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

int XCanvas::viewPointChangeCount () const {
  return m_data->m_viewPointChangeCount;
}

void XCanvas::setWidth (int width) {
  XCanvasData &d = *m_data;
  if (width != d.m_width) {
    d.invalidate ();
    d.m_width = width;
    d.freeVisualBuffers ();
  }
}

void XCanvas::setHeight (int height) {
  XCanvasData &d = *m_data;
  if (height != d.m_height) {
    d.invalidate ();
    d.m_height = height;
    d.freeVisualBuffers ();
  }
}

int XCanvas::getWidth () const {
  return m_data->m_width;
}

int XCanvas::getHeight () const {
  return m_data->m_height;
}

void XCanvas::setScale (Float scale) {
  m_data->invalidate ();
  m_data->m_scale = scale;
}

Float XCanvas::getScale () const {
  return m_data->m_scale;
}

Display *XCanvas::getDisplay () {
  return m_data->m_display;
}

void XCanvas::closeDisplay () {
  m_data->closeDisplay ();
}

bool XCanvas::doubleBuffered () const {
  return m_data->m_useDbe;
} 

void XCanvas::swapBuffers () const {
#ifdef TRY_DBE
  assert (m_data->m_useDbe);
  XdbeSwapInfo si;
  si.swap_window = m_data->m_window;
  si.swap_action = mySwapAction;
  Status s = XdbeSwapBuffers (m_data->m_display, &si, 1);
  if (0 == s) {
    die (String ("Failed to swap buffers, status is ") + (int) s);
  }
#else
  assert (0 &&
	  "swapBuffers was called but double buffering was compiled out.");
#endif
}

const Display *XCanvas::getDisplay () const {
  return m_data->m_display;
}

Vec3 XCanvas::transformWorldToLHScreen (const Vec3 &where) const
{
  m_data->validate ();
  return m_data->m_worldToScreen * where;
}

Vec3 XCanvas::transformLHScreenToWorld (const Vec3 &where) const
{
  m_data->validate ();
  return m_data->m_screenToWorld * where;
}

Vec3i XCanvas::iTransformWorldToLHScreen (const Vec3 &where) const
{
  return (transformWorldToLHScreen (where)).truncateToVec3i ();
}

namespace {
  inline Vec3 flipZ (const Vec3 &v) {
    return Vec3 (v[0], v[1], -v[2]);
  }
}

Vec3 XCanvas::transformWorldToRHScreen (const Vec3 &where) const
{
  return flipZ (transformWorldToLHScreen (where));
}

Vec3 XCanvas::transformRHScreenToWorld (const Vec3 &where) const
{
  return transformLHScreenToWorld (flipZ (where));
}

Matrix4 XCanvas::transformRHScreenToWorld (const Matrix4 &mat) const {
  m_data->validate ();
  return (m_data->m_screenToWorld * Matrix4::flipZ () * mat *
	  Matrix4::flipZ () * m_data->m_worldToScreen);
}

Quaternion XCanvas::transformRHScreenToWorld (const Quaternion &q) const {
  m_data->validate ();
  return ~getRotation() * q * getRotation ();
}

Quaternion XCanvas::transformWorldToRHScreen (const Quaternion &q) const {
  m_data->validate ();
  return getRotation() * q * ~getRotation ();
}

BoundingSphere XCanvas::transformRHScreenToWorld (const BoundingSphere &bs)
  const
{
  m_data->validate ();
  return BoundingSphere (bs.radius()/getScale(),
			 transformRHScreenToWorld (bs.center()));
}

BoundingSphere XCanvas::transformWorldToRHScreen (const BoundingSphere &bs)
  const
{
  m_data->validate ();
  return BoundingSphere (bs.radius()*getScale(),
			 transformWorldToRHScreen (bs.center()));
}

void XCanvas::setOrigin (const Vec3 &o) {
  m_data->invalidate();
  m_data->m_origin = o;
}

Vec3 XCanvas::getOrigin () const {
  return m_data->m_origin;
}

void XCanvas::setRotation (const Quaternion &q) {
  m_data->invalidate();
  m_data->m_rotation = q;
}

Quaternion XCanvas::getRotation () const {
  return m_data->m_rotation;
}

void *XCanvas::getFrameBuffer () const {
  m_data->validate ();
  return (void *) m_data->m_frameBuffer->data;
}

int XCanvas::getFrameBufferPixelsPerLine () const {
  m_data->validate ();
  return m_data->m_frameBuffer->bytes_per_line / pixelBytes ();
}

XCanvas::Depth *XCanvas::getDepthBuffer () const {
  m_data->validate ();
  return m_data->m_depthBuffer;
}

namespace {
  inline LongestPixel packOneColor (Float intensity, unsigned long mask)
  {
    return (LongestPixel) (((unsigned long)(mask * intensity)) & mask);
  }
}

LongestPixel XCanvas::packColor (const Color &c) const {
  const XVisualInfo &xvi = m_data->m_xvi;
  return (packOneColor (c[0], xvi.red_mask) |
	  packOneColor (c[1], xvi.green_mask) |
	  packOneColor (c[2], xvi.blue_mask));
}

namespace {
  inline Float unPackOneColor (LongestPixel p, unsigned long mask) {
    unsigned long relevantBits = p & mask;
    return ((Float) relevantBits) / mask;
  }
}

Color XCanvas::unPackColor (LongestPixel p) const {
  const XVisualInfo &xvi = m_data->m_xvi;
  return Color (unPackOneColor (p, xvi.red_mask),
		unPackOneColor (p, xvi.green_mask),
		unPackOneColor (p, xvi.blue_mask),
		1);
}

void XCanvas::XCanvasData::validate () {
  if (m_valid) {
    assert (m_frameBuffer);
    assert (m_depthBuffer);
  } else {
    m_valid = true;
    if (!m_frameBuffer) {
#ifdef TRY_MIT_SHM
      if (m_useShm) {
	m_frameBuffer = XShmCreateImage (m_display, m_xvi.visual,
					 m_xvi.depth, ZPixmap, 0,
					 &m_shminfo, m_width, m_height);
	if (! m_frameBuffer) {
	  cerr << "XShmCreateImage gave us a null.  Bailing." << endl;
	  exit (1);
	}
	assert (m_height == m_frameBuffer->height);
	m_shminfo.shmid =
	  shmget (IPC_PRIVATE, m_frameBuffer->bytes_per_line * m_height,
		  IPC_CREAT|0777);
	if (-1 == m_shminfo.shmid) {
	  cerr << "shmget gave us -1.  Bailing." << endl;
	  exit (1);
	}
	m_shminfo.shmaddr = (char *) shmat (m_shminfo.shmid, 0, 0);
	if (-1 == (int) m_shminfo.shmaddr) {
	  cerr << "shmat gave us -1.  Bailing." << endl;
	  exit (1);
	}
	m_frameBuffer->data = m_shminfo.shmaddr;
	m_shminfo.readOnly = True;
	bool attached = XShmAttach (m_display, &m_shminfo);
	if (! attached) {
	  cerr << "XShmAttach failed.  Bailing." << endl;
	  exit (1);
	}
	XSync (m_display, False);
	// Remove the shared memory id.
	shmctl (m_shminfo.shmid, IPC_RMID, 0);
      } else {
#endif
	// XDestroyImage is presumably going to use free, so I have to use
	// malloc.
	char *ptr = (char *) malloc (m_width * m_height * pixelBytes ());
	m_frameBuffer = XCreateImage (m_display, m_xvi.visual,
				      m_xvi.depth, ZPixmap, 0, ptr, 
				      m_width, m_height, 8*pixelBytes (),
				      0);
#ifdef TRY_MIT_SHM
      }
#endif
      assert (m_frameBuffer->data);
      m_depthBuffer = NEW (XCanvas::Depth [m_width * m_height]);
    }
    {
      m_worldToScreen =
	// Read this product from right to left.
	Matrix4::flipZ () * Matrix4::scale (m_scale) * ((Matrix4) m_rotation) *
	Matrix4::translate (- m_origin);
    }
    m_screenToWorld = m_worldToScreen.invert ();
  }
}
