// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __ObjectDrawer_h__
#include "ObjectDrawer.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __OneObjectDrawerConfiguration_h__
#include "OneObjectDrawerConfiguration.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __Vec3i_h__
#include "Vec3i.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __BallConfiguration_h__
#include "BallConfiguration.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __XObjectDrawer_h__
#include "XObjectDrawer.h"
#endif

#ifndef __Color_h__
#include "Color.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

#ifndef __Ball_h__
#include "Ball.h"
#endif

#ifndef __BinSet_h__
#include "BinSet.h"
#endif

#ifndef __XBallDrawerConfiguration_h__
#include "XBallDrawerConfiguration.h"
#endif

namespace {
  typedef unsigned short DepthDifference;
  typedef Canvas::Depth Depth;
  typedef XObjectDrawer::DepthCount DepthCount;
  struct DepthPixel {
    // The location on the ball is nearer to the eye than the center
    // of the ball by this much.  We access depths more than pixels, so put
    // this first.
    DepthDifference m_depth;
    // Color for this pixel.
    Canvas::LongestPixel m_pixel;
    DepthPixel (DepthDifference depth, Canvas::LongestPixel pixel)
      : m_depth (depth), m_pixel (pixel)
    {}
    DepthPixel () {}
  };

  // RowEntry has to be a template, because the pointers we increment for
  // drawing a row have to be pointers to the right kind of pixel.
  template <class Pixel>
  class RowEntry {
    Dynavec <DepthPixel> m_depthpixels;
    // Width, in pixels,
    int m_width;
    // X position of start.
    int m_start;
  public:
    RowEntry () {}
    RowEntry (const Dynavec <DepthPixel> &depthpixels, int width, int start)
      : m_depthpixels (depthpixels),
	m_width (width),
	m_start (start)
    {
      assert (depthpixels.size());
      assert (width);
    }
    // The most general case.  x and z clipping, and update the depth clipping
    // buffer.
    inline void drawRowXZDepth (Pixel *toPix, Depth *toDepth, int z,
				int start, int end, int slabPixel,
				DepthCount *toDepthClip, int clipZ,
				int depthShift)
    {
      const int trueStart = FloatUtil::max (start, m_start);
      toPix += trueStart;
      toDepth += trueStart;
      const int shift = trueStart - m_start;
      DepthPixel *fromPix = &*m_depthpixels + shift;
      DepthPixel *endCopy =
	fromPix + FloatUtil::min (end, m_start + m_width) - trueStart;
      for (; fromPix < endCopy; toDepth++, toPix++, fromPix++) {
	const int thisZ = z - fromPix->m_depth;
	if (z < clipZ) {
	  assert (0 == clipZ);
	  // You're looking at a slice of less than half a ball.
	  // Weird case.
	  if (z + fromPix->m_depth > clipZ) {
	    // FIXME Try separating the loads from the stores when
	    // manipulating the clip buffer.  Might make it faster.
	    *toPix = slabPixel;
	    toDepthClip [(*toDepth) >> depthShift] --;
	    *toDepth = clipZ;
	    toDepthClip [clipZ >> depthShift] ++;
	  }
	} else if (thisZ < clipZ) {
	  *toPix = slabPixel;
	  toDepthClip [(*toDepth) >> depthShift] --;
	  *toDepth = clipZ;
	  toDepthClip [clipZ >> depthShift] ++;
	} else {
	  Depth d = *toDepth;
	  if (thisZ < d) {
	    *toPix = fromPix->m_pixel;
	    toDepthClip [d >> depthShift] --;
	    *toDepth = thisZ;
	    toDepthClip [thisZ >> depthShift] ++;
	  }
	}
      }
    }
    
    inline void drawRowXDepthOne (int z,
				  DepthPixel *fromPix, 
				  Depth *toDepth,
				  Pixel *toPix,
				  DepthCount *toDepthClip,
				  int depthShift) {
      int thisZ = z - fromPix->m_depth;
      // You'd think that storing *toDepth into a variable would help,
      // because maybe that variable would wind up in a register.  But we
      // have too few registers at this point, so it doesn't help.
      if (thisZ < *toDepth) {
	*toPix = fromPix->m_pixel;
	toDepthClip [(*toDepth) >> depthShift] --;
	*toDepth = thisZ;
	toDepthClip [(thisZ) >> depthShift] ++;
      }
    }

    // Use this one for depth clipping, but no z clipping.
    // Still have to do x clipping to clip to a bin.
    // Few balls are less than 32 wide, so not much point in depth clipping
    // without x clipping.  X clipping doesn't affect our innermost loop,
    // anyway.
    inline void drawRowXDepth (Pixel *toPix, Depth *toDepth, int z,
			       int start, int end, DepthCount *toDepthClip,
			       int depthShift)
    {
      int trueStart = FloatUtil::max (start, m_start);
      toPix += trueStart;
      toDepth += trueStart;
      int shift = trueStart - m_start;
      DepthPixel *fromPix = &*m_depthpixels + shift;
      int toCopy = FloatUtil::min (end, m_start + m_width) - trueStart;
      assert (32 == 1 << XObjectDrawer::widthShift);
      assert (toCopy <= 32);
      switch (toCopy) {
      case 32:
	drawRowXDepthOne (z, fromPix+31, toDepth+31, toPix+31,
			  toDepthClip, depthShift);
      case 31:
	drawRowXDepthOne (z, fromPix+30, toDepth+30, toPix+30,
			  toDepthClip, depthShift);
      case 30:
	drawRowXDepthOne (z, fromPix+29, toDepth+29, toPix+29,
			  toDepthClip, depthShift);
      case 29:
	drawRowXDepthOne (z, fromPix+28, toDepth+28, toPix+28,
			  toDepthClip, depthShift);
      case 28:
	drawRowXDepthOne (z, fromPix+27, toDepth+27, toPix+27,
			  toDepthClip, depthShift);
      case 27:
	drawRowXDepthOne (z, fromPix+26, toDepth+26, toPix+26,
			  toDepthClip, depthShift);
      case 26:
	drawRowXDepthOne (z, fromPix+25, toDepth+25, toPix+25,
			  toDepthClip, depthShift);
      case 25:
	drawRowXDepthOne (z, fromPix+24, toDepth+24, toPix+24,
			  toDepthClip, depthShift);
      case 24:
	drawRowXDepthOne (z, fromPix+23, toDepth+23, toPix+23,
			  toDepthClip, depthShift);
      case 23:
	drawRowXDepthOne (z, fromPix+22, toDepth+22, toPix+22,
			  toDepthClip, depthShift);
      case 22:
	drawRowXDepthOne (z, fromPix+21, toDepth+21, toPix+21,
			  toDepthClip, depthShift);
      case 21:
	drawRowXDepthOne (z, fromPix+20, toDepth+20, toPix+20,
			  toDepthClip, depthShift);
      case 20:
	drawRowXDepthOne (z, fromPix+19, toDepth+19, toPix+19,
			  toDepthClip, depthShift);
      case 19:
	drawRowXDepthOne (z, fromPix+18, toDepth+18, toPix+18,
			  toDepthClip, depthShift);
      case 18:
	drawRowXDepthOne (z, fromPix+17, toDepth+17, toPix+17,
			  toDepthClip, depthShift);
      case 17:
	drawRowXDepthOne (z, fromPix+16, toDepth+16, toPix+16,
			  toDepthClip, depthShift);
      case 16:
	drawRowXDepthOne (z, fromPix+15, toDepth+15, toPix+15,
			  toDepthClip, depthShift);
      case 15:
	drawRowXDepthOne (z, fromPix+14, toDepth+14, toPix+14,
			  toDepthClip, depthShift);
      case 14:
	drawRowXDepthOne (z, fromPix+13, toDepth+13, toPix+13,
			  toDepthClip, depthShift);
      case 13:
	drawRowXDepthOne (z, fromPix+12, toDepth+12, toPix+12,
			  toDepthClip, depthShift);
      case 12:
	drawRowXDepthOne (z, fromPix+11, toDepth+11, toPix+11,
			  toDepthClip, depthShift);
      case 11:
	drawRowXDepthOne (z, fromPix+10, toDepth+10, toPix+10,
			  toDepthClip, depthShift);
      case 10:
	drawRowXDepthOne (z, fromPix+9, toDepth+9, toPix+9,
			  toDepthClip, depthShift);
      case 9:
	drawRowXDepthOne (z, fromPix+8, toDepth+8, toPix+8,
			  toDepthClip, depthShift);
      case 8:
	drawRowXDepthOne (z, fromPix+7, toDepth+7, toPix+7,
			  toDepthClip, depthShift);
      case 7:
	drawRowXDepthOne (z, fromPix+6, toDepth+6, toPix+6,
			  toDepthClip, depthShift);
      case 6:
	drawRowXDepthOne (z, fromPix+5, toDepth+5, toPix+5,
			  toDepthClip, depthShift);
      case 5:
	drawRowXDepthOne (z, fromPix+4, toDepth+4, toPix+4,
			  toDepthClip, depthShift);
      case 4:
	drawRowXDepthOne (z, fromPix+3, toDepth+3, toPix+3,
			  toDepthClip, depthShift);
      case 3:
	drawRowXDepthOne (z, fromPix+2, toDepth+2, toPix+2,
			  toDepthClip, depthShift);
      case 2:
	drawRowXDepthOne (z, fromPix+1, toDepth+1, toPix+1,
			  toDepthClip, depthShift);
      case 1:
	drawRowXDepthOne (z, fromPix, toDepth, toPix,
			  toDepthClip, depthShift);
      case 0: ;
      }
    }
    // Use this one for no depth clipping or x or z clipping.
    inline void drawRow (Pixel *toPix, Depth *toDepth, int z)
    {
      assert (z >= 0);
      toPix += m_start;
      toDepth += m_start;
      DepthPixel *fromPix = &*m_depthpixels;
      DepthPixel *endCopy = fromPix + m_width;
      for (; fromPix < endCopy; toDepth++, toPix++, fromPix++) {
	Depth thisZ = z - fromPix->m_depth;
	if (thisZ < *toDepth) {
	  *toPix = fromPix->m_pixel;
	  *toDepth = thisZ;
	}
      }
    }
  };

  class XBallDrawer
    : public XObjectDrawer
  {
    mutable Dynavec <RowEntry <Canvas::Pixel16> > m_rows16;
    mutable Dynavec <RowEntry <Canvas::Pixel32> > m_rows32;
    mutable int m_widestRow;
    mutable int m_lastVoGeneration;
    mutable SP<const Canvas> m_lastVo;
    // What to draw when the ball has negative z.
    // Has to be mutable because it can change when m_lastVo changes.
    mutable Canvas::LongestPixel m_slabPixel;
    const Float m_radius;
    const Color m_material;
    const bool m_watch;
    mutable bool m_griped;
  public:
    bool intersect (const Float *state,
		    const Vec3 &v1, const Vec3 &v2, Vec3 *result,
		    int objectIndex, const SceneGraph *sg) const;
    bool graphicsIntersect (const Float *state, const BoundingBox &box,
			    int objectIndex, const SceneGraph *sg)
      const;
    bool intersect (const Float *state, const Vec3 &v,
		    int objectIndex, const SceneGraph *sg) const;
    Float graphicsBoundingSphereRadius (int objectIndex,
					const SceneGraph *sg) const;
    Vec3 boundingSphereCenter (const Float *state) const;

    XBallDrawer (BallConfiguration *conf, XBallDrawerConfiguration *otherConf)
      : m_radius (conf->getVisibleRadius()),
	m_material (conf->getMaterial()),
	m_watch (otherConf->getWatch()),
	m_griped (false)
    {}
    // Return true if we succeeded, false if we failed.
    template <class Pixel>
    bool genNewRaster (const Canvas *vo, Dynavec <RowEntry <Pixel> > &rows)
    const
    {
      m_slabPixel = vo->packColor (m_material);
      rows.extendTo (0);
      m_widestRow = 0;
      int radius = (int) (m_radius * vo->getScale());
      {
	// We have data big enough to accomodate up to (DepthDifference) -1,
	// but if the balls are too big we spend a huge amount of memory
	// precomputing the image of them.
	// Radius may be negative if we had an integer overflow while computing
	// it. 
	const int maxRadius = 255;
	if (radius > maxRadius || radius < 0) {
	  if (!m_griped) {
	    m_griped = true;
	    message (String ("We can't draw balls with a radius bigger than ")
		     +maxRadius+ " pixels");
	    message ("Please scale the scene to make the balls smaller, "
		     "so we can resume drawing them");
	  }
	  return false;
	} else {
	  if (m_griped) {
	    m_griped = false;
	    message ("The radius is small enough now, so we'll resume "
		     "drawing");
	  }
	}
      }
      // If we scale down far enough, radius will be zero, and then if we
      // attempt to do the loop below we'll lose in the makeUnit() call.
      if (radius > 0) {
	// y increases upward.
	for (int y = -radius; y <= radius; y++) {
	  int halfwidth =
	    FloatUtil::floor (FloatUtil::sqrt ((Float) ((radius * radius)
							- (y * y))));
	  int xmin = radius - halfwidth;
	  int xmax = radius + halfwidth;
	  int width = xmax - xmin + 1;
	  m_widestRow = FloatUtil::max (m_widestRow, width);
	  Dynavec <DepthPixel> pixels;
	  for (int pos = 0; pos < width; pos++) {
	    int x = xmin + pos;
	    const Vec3 lightDirection = Vec3 (-1, 1, -1).makeUnit();
	    const Float lightIntensity = 0.3;
	    int dx = x - radius;
	    Float z = FloatUtil::sqrt ((Float) (radius * radius
						- dx * dx - y * y));
	    Vec3 normalVector = Vec3 (dx, y, -z).makeUnit ();
	    Color color = m_material;
	    Float product = normalVector * lightDirection;
	    if (product > 0) {
	      color = color * (1 + product * lightIntensity);
	    }
	    color = (color * 0.75).clamp ();
	    pixels.push (DepthPixel ((DepthDifference) z,
				     vo->packColor (color)));
	  }
	  rows.push (RowEntry <Pixel> (pixels, width, xmin));
	}
      }
      return true;
    }

    // Return true if we succeeded, false if we failed.
    template <class Pixel>
    bool maybeNewRaster (const Canvas *vo,
			 Dynavec <RowEntry <Pixel> > &rows) const {
      int gen = vo->viewPointChangeCount ();
      if (vo != m_lastVo || m_lastVoGeneration != gen) {
	bool result = genNewRaster <Pixel> (vo, rows);
	if (result) {
	  m_lastVo = vo;
	  m_lastVoGeneration = gen;
	}
	return result;
      } else {
	return true;
      }
    }

    // Draw a bin with no Z clipping.  Abstracted as a function so we can
    // inline it a bunch of times with different constant values of depthShift.
    template <class Pixel>
    inline void drawBinDepthNoZClip (int binMinX, int binMaxX,
				     int binMinY, int binMaxY,
				     Depth *toDepth,
				     DepthCount *toDepthClip, Pixel *toPix,
				     const int z,
				     const int framePixelsPerLine,
				     const int frameWidth,
				     const int depthShift,
				     Dynavec <RowEntry <Pixel> > &rows) const {
      for (int row = binMinY; row < binMaxY; row++) {
	rows[row].drawRowXDepth (toPix, toDepth, z,
				   binMinX, binMaxX,
				   toDepthClip, depthShift);
	toPix -= framePixelsPerLine;
	toDepth -= frameWidth;
      }
    }

    template <class Pixel>
    inline void drawTemplate (const Float *state, const Canvas *vo,
			      const Vec3i &where, const DepthClipInfo &dci,
			      const BinSet &finished, int objectIndex,
			      const SceneGraph *sg,
			      Dynavec <RowEntry <Pixel> > &rows) const
    {
      (void) state; (void) finished; (void) objectIndex; (void) sg;
      if (!maybeNewRaster <Pixel> (vo, rows)) return;
      assert (sizeof(Pixel) == vo->pixelBytes ());
      Pixel *frameBuffer = (Pixel *) (vo->getFrameBuffer ());
      const int frameHeight = vo->getHeight ();
      const int frameWidth = vo->getWidth ();
      const int framePixelsPerLine = vo->getFrameBufferPixelsPerLine ();
      int radius = (int) (m_radius * vo->getScale());
      // (startX, startY) are where on the screen (Y increasing *upward* so we
      // can compute bins without going crazy) to put the upper left
      // corner of the square enclosing the ball.
      int startX = where [0] - radius;
      int startY = where [1] - radius;
      int z = where [2];
      const int depthShift = dci.m_depthShift;
      const int widthShift = XObjectDrawer::widthShift;
      // First X coordinate that we want to draw in the bin, relative to
      // startX.
      int binMinX = FloatUtil::max (// Left edge of the screen.
				    -startX,
				    // Left edge of the raster and left edge of
				    // the bin.
				    0);
      // Call drawRow once per bin in the scan line.
      for (;;) {
	// First X coordinate past the end of the bin, relative to startX.
	int binMaxX =
	  FloatUtil::min
	  // Right edge of the bin.
	  (((((startX + binMinX) >> widthShift) + 1) << widthShift) - startX,
	   // Right edge of the screen.
	   frameWidth - startX,
	   // Right edge of the raster.
	   m_widestRow);
	if (binMinX >= binMaxX) break;
	// First Y coordinate in the bin, increasing upward, relative to
	// startY.
	int binMinY = FloatUtil::max
	  (// Bottom edge of screen
	   -startY,
	   // Bottom edge of raster and of the bin.
	   0);
	for (;;) {
	  // Last Y coordinate in the bin, increasing upward, relative to
	  // startY.
	  int binMaxY =
	    FloatUtil::min
	    (// Top of the bin
	     ((((startY + binMinY) >> widthShift) + 1) << widthShift) - startY,
	     // Top of the screen
	     frameHeight - startY,
	     // Top of the raster
	     rows.size ());
	  if (binMinY >= binMaxY) break;
	  if (!finished.isElement ((startX + binMinX) >> widthShift,
				   (startY + binMinY) >> widthShift)) {
	    // Y screen coordinate (increasing downward).
	    int screenY = frameHeight - (startY + binMinY) - 1;
	    Pixel *toPix = startX + frameBuffer + framePixelsPerLine * screenY;
	    Depth *toDepth =
	      startX + vo->getDepthBuffer() + frameWidth * screenY;
	    DepthCount *toDepthClip =
	      &dci.binAtScreenCoord (startX + binMinX, startY + binMinY, 0);
	    if (z > radius + dci.m_clipZ) {
	      // No Z clipping.
	      // Massively duplicate code to decrease the number of variables
	      // in the innermost loop by one.  Code we don't execute is pretty
	      // cheap; it might not even get paged in.
	      switch (depthShift) {
	      case 3:
		drawBinDepthNoZClip (binMinX, binMaxX,
				     binMinY, binMaxY,
				     toDepth, toDepthClip, toPix,
				     z,
				     framePixelsPerLine,
				     frameWidth,
				     3, rows);
		break;
	      case 4:
		drawBinDepthNoZClip (binMinX, binMaxX,
				     binMinY, binMaxY,
				     toDepth, toDepthClip, toPix,
				     z,
				     framePixelsPerLine,
				     frameWidth,
				     4, rows);
		break;
	      case 5:
		drawBinDepthNoZClip (binMinX, binMaxX,
				     binMinY, binMaxY,
				     toDepth, toDepthClip, toPix,
				     z,
				     framePixelsPerLine,
				     frameWidth,
				     5, rows);
		break;
	      case 6:
		drawBinDepthNoZClip (binMinX, binMaxX,
				     binMinY, binMaxY,
				     toDepth, toDepthClip, toPix,
				     z,
				     framePixelsPerLine,
				     frameWidth,
				     6, rows);
		break;
	      case 7:
		drawBinDepthNoZClip (binMinX, binMaxX,
				     binMinY, binMaxY,
				     toDepth, toDepthClip, toPix,
				     z,
				     framePixelsPerLine,
				     frameWidth,
				     7, rows);
		break;
	      case 8:
		drawBinDepthNoZClip (binMinX, binMaxX,
				     binMinY, binMaxY,
				     toDepth, toDepthClip, toPix,
				     z,
				     framePixelsPerLine,
				     frameWidth,
				     8, rows);
		break;
	      default:
		drawBinDepthNoZClip (binMinX, binMaxX,
				     binMinY, binMaxY,
				     toDepth, toDepthClip, toPix,
				     z,
				     framePixelsPerLine,
				     frameWidth,
				     depthShift, rows);
	      break;
	      }
	    } else {
	      // Z clipping.
	      for (int row = binMinY; row < binMaxY; row++) {
		Pixel *thisPix = toPix;
		assert (toDepthClip ==
			&dci.binAtScreenCoord (startX + binMinX,
					       startY + row, 0));
		rows[row].drawRowXZDepth (thisPix, toDepth, z,
					  binMinX, binMaxX,
					  m_slabPixel, toDepthClip,
					  dci.m_clipZ, depthShift);
		toPix -= framePixelsPerLine;
		toDepth -= frameWidth;
	      }
	    } // whether or not Z clipping
	  } // Whether the bin is finished
	  if (m_watch) {
	    vo->drawFrame ();
	  }
	  binMinY = binMaxY;
	} // Loop over y bins
	binMinX = binMaxX;
      } // Loop over x bins
    } // drawTemplate
    void draw (const Float *state, const Canvas *vo,
	       const Vec3i &where, const DepthClipInfo &dci,
	       const BinSet &finished, int objectIndex,
	       const SceneGraph *sg) const;
    bool isVisible () const {
      return true;
    }
  }; // XBallDrawer

  void XBallDrawer::draw (const Float *state, const Canvas *vo,
			  const Vec3i &where, const DepthClipInfo &dci,
			  const BinSet &finished, int objectIndex,
			  const SceneGraph *sg) const
  {
    int pixelBytes = vo->pixelBytes ();
    switch (pixelBytes) {
    case 2:
      drawTemplate <Canvas::Pixel16> (state, vo, where,
				      dci, finished, objectIndex, sg,
				      m_rows16);
      break;
    case 4:
      drawTemplate <Canvas::Pixel32> (state, vo, where,
				      dci, finished, objectIndex, sg,
				      m_rows32);
      break;
    default:
      die (String ("Bad pixel size ") + pixelBytes + (" in XBallDrawer"));
      break;
    }
  }

  bool XBallDrawer::intersect (const Float *state, const Vec3 &v,
			       int objectIndex, const SceneGraph *sg) const
  {
    (void) objectIndex; (void) sg;
    Vec3 distvec = Vec3 (state) - v;
    return distvec.lengthSquared() <= FloatUtil::sqr (m_radius);
  }
  
  
  bool XBallDrawer::graphicsIntersect (const Float *state,
				       const BoundingBox &box,
				       int objectIndex,
				       const SceneGraph *sg) const
  {
    return box.overlaps (graphicsBoundingBox (state, objectIndex, sg));
  }
  
  bool XBallDrawer::intersect (const Float *state,
			       const Vec3 &v1, const Vec3 &v2,
			       Vec3 *result,
			       int objectIndex,
			       const SceneGraph *sg) const
  {
    (void) objectIndex; (void) sg;
    Vec3 lineVec = v2 - v1;
    Float lineLengthSquared = lineVec * lineVec;
    Float lineLength = sqrt (lineLengthSquared);
    if (0.0 == lineLength) {
      // Line segment has zero length.  If v1 and v2 are in the
      // object, then we are allowed to return false; otherwise false
      // is the true answer.
      return false;
    } else {
      Vec3 lineUnitVector = lineVec / lineLength;
      Vec3 ballCenter = Vec3 (state [0], state [1], state [2]);
      Vec3 ballFromV1 = ballCenter - v1;
      Float ballDistAlongLine = ballFromV1 * lineUnitVector;
      if (ballDistAlongLine < 0.0 || ballDistAlongLine > lineLength) {
	return false;
      } else {
	// ballDistFromLine is the distance of the center of the ball
	// from the line.
	// For little balls far away, ballDistFromLineSquared can be negative
	// due to roundoff error. 
	const Float ballDistFromLineSquared =
	  ballFromV1.lengthSquared() - FloatUtil::sqr (ballDistAlongLine);
	const Float ballDistFromLine =
	  ballDistFromLineSquared < 0 ? 0 : sqrt (ballDistFromLineSquared);
	if (ballDistFromLine > m_radius) {
	  return false;
	} else {
	  if (result) {
	    // Find the closest intersection point.  We have a right
	    // triangle.  The hypotenuse is the radius of the ball.  One
	    // corner is the center of the ball.  Another corner is the
	    // closest point along our line to the center of the ball.
	    // The corner we want to compute the position of is the
	    // remaining point in the right triangle.
	    Float interDistAlongLine =
	      ballDistAlongLine -
	      sqrt (FloatUtil::sqr (m_radius) -
		    FloatUtil::sqr (ballDistFromLine));
	    if (interDistAlongLine < 0) {
	      interDistAlongLine = 0;
	    }
	    *result = v1 + interDistAlongLine * lineUnitVector;
	    // Need fuzz to swamp roundoff error.
	    assert (((*result) - ballCenter).lengthSquared()
		    <= FloatUtil::sqr (m_radius) + 0.01);
	  } 
	  return true;
	}
      }
    }
  }
  
  Float XBallDrawer::graphicsBoundingSphereRadius (int objectIndex,
						   const SceneGraph *sg)
    const
  {
    (void) objectIndex; (void) sg;
    return m_radius;
  }
  
  Vec3 XBallDrawer::boundingSphereCenter (const Float *state) const {
    return Ball::center (state);
  }

  class XBallDrawerFactory
    : public Factory
  {
  public:
    XBallDrawerFactory () : Factory ("XBallDrawer")
    {}
    SP<Configuration> defaultConfiguration () const {
      return NEW (XBallDrawerConfiguration ());
    }
    SP<Configurable> makeIt (SP<Configuration> c) const {
      XBallDrawerConfiguration *oodc =
	dynamic_cast <XBallDrawerConfiguration *> (&*c);
      assert (oodc);
      SP<Configuration> conf = oodc->getPhysicsObjectConfiguration ();
      assert (conf);
      BallConfiguration *bi = dynamic_cast <BallConfiguration *> (&*conf);
      assert (bi);
      return NEW (XBallDrawer (bi, oodc));
    }
  };
  static const bool useless =
  (FactoryTable::store ("ObjectDrawer", NEW (XBallDrawerFactory ())),
   true);
}
