// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//
#ifndef __DepthClipInfo_h__
#define __DepthClipInfo_h__

// Everything you need to know to draw into a depth clipping buffer.
#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __XObjectDrawer_h__
#include "XObjectDrawer.h"
#endif

struct DepthClipInfo {
  // The buffer is a 3-d array of bins.  A pixel at coordinates (X,
  // Y) on the screen and depth Z falls into the bin at
  // (X >> widthShift, Y >> widthShift, Z >> m_depthShift).
  // We simply assume that the figure starts at x = 0 and y = 0.  We cause
  // the figure to start with a minimum depth of z = 0 by translating the
  // points before XObjectDrawer::draw (..., DepthClipInfo &dci) is called.
  //
  // The bin contains one integer, which is a count of the number of
  // pixels that fall into the bin.  It is the responsibility of the drawing
  // code for each object to maintain the counts in the depth
  // clipping buffer.  When an XDepthClippingDrawer is drawing the
  // screen, it draws them sorted by increasing bin depth for the
  // least-depth pixel in the object.
  //
  // If we're thinking of drawing an object that extends
  // over a certain range of X and Y bins, and it's minimum Z bin is
  // minZ, but all of the pixels for the X and Y bins have already
  // been drawn with a Z bin less than minZ, then we don't have to
  // draw the object at all.
  //
  // See binAtBinCoord below for computing the bin for given screen
  // coordinates.
  //
  // y for a depth clip buffer increases going upward  on the screen.
  typedef XObjectDrawer::DepthCount DepthCount;
  DepthCount *m_buffer;
  // If the next batch of parameters are meaningless because there
  // are no non-clipped objects in the scene, we reflect that by
  // having m_clipZ as MAX_INT and the other fields are uninitialized.
  // m_clipZ is the minimum depth of the scene.
  int m_clipZ;
  int m_depthShift;
  // m_backgroundDepth + m_clipZ is more than the maximum depth of the scene.
  // The background has to be in the last bin,
  // but it has to be further away than all of the graphics in the scene.
  Canvas::Depth m_backgroundDepth;
  inline DepthCount &binAtBinCoord (int x, int y, int z) const {
    // Next assert checks that DepthCount is really big enough to count what
    // we want it to.
    assert ((DepthCount) (1 << (XObjectDrawer::clipBufWidthShift +
				XObjectDrawer::clipBufWidthShift))
	    == (1 << (XObjectDrawer::clipBufWidthShift +
		      XObjectDrawer::clipBufWidthShift)));
    assert (x >= 0);
    assert (y >= 0);
    assert (z >= 0);
    assert (z < (1 << XObjectDrawer::clipBufDepthShift));
    assert (y < (1 << XObjectDrawer::clipBufWidthShift));
    assert (x < (1 << XObjectDrawer::clipBufWidthShift));
    // If x increases by 1, we need the next bin to be nearby, since as we
    // draw we increase x by 1.
    return
      m_buffer
      [(((y << XObjectDrawer::clipBufWidthShift) + x)
	<< XObjectDrawer::clipBufDepthShift) + z];
  }
  inline DepthCount &binAtScreenCoord (int x, int y, int z) const {
    return binAtBinCoord (x >> XObjectDrawer::widthShift,
			  y >> XObjectDrawer::widthShift,
			  z >> m_depthShift);
  }
};

#endif
