// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Canvas_h__
#define __Canvas_h__

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

class Vec3;
class Vec3i;
class BoundingBox;
class BoundingSphere;
class Quaternion;
class Color;
class Matrix4;
class Canvas
  : public Refcount
{
public:
  // The code in XTopLevel.cpp ought to work with any unsigned integer
  // type as the pixel, so long as we can get a TrueColor visual.
  // The longest pixel type we support.
  typedef int Pixel32;
  typedef short Pixel16;
  typedef int LongestPixel;
  virtual LongestPixel packColor (const Color &c) const = 0;
  virtual Color unPackColor (LongestPixel p) const = 0;
  // Number of bytes in a Pixel.  Real implementations will want to use
  // templates to generate efficient code for a small variety of values of
  // pixelBytes, typically 2 and 4.
  virtual int pixelBytes () const = 0;
  // The Y coordinate for the frame buffer increases as you go down on the
  // screen.
  // Really returns a Pixel *, but this interface 
  virtual void *getFrameBuffer () const = 0;
  // 16 bits seems about the right size for a depth in the depth buffer.
  typedef unsigned short Depth;
  // A rectangle on the screen.  The coordinates have Y increasing upward.
  struct ScreenRect {
    int x;
    int y;
    int width;
    int height;
    ScreenRect (int a_x, int a_y, int a_width, int a_height)
      : x (a_x), y (a_y), width (a_width), height (a_height)
    {}
    int flipY (int frameHeight) const {
      return frameHeight - y - height;
    }
  };
  virtual int viewPointChangeCount () const = 0;
  // The size of the window.
  virtual int getWidth () const = 0;
  virtual int getHeight () const = 0;
  // Scale is the number of pixels per world coordinate.  So for a sphere,
  // multiply the radius by the scale to get the number of pixels to
  // show for the radius.  (It's this way instead of the inverse because
  // multiplication is faster than division.)
  virtual void setScale (Float scale) = 0;
  virtual Float getScale () const = 0;
  // The input vector is world coordinates.  In the output vector, x
  // is horizontal to the right; y is vertically up; and z is depth,
  // increasing as you go further from the eye.  In isOffScreen, we
  // assume that depths less than 0 will not be displayed.
  // Note that the screen coordinate system is left-handed.
  virtual Vec3 transformWorldToLHScreen (const Vec3 &where) const = 0;
  virtual Vec3 transformLHScreenToWorld (const Vec3 &where) const = 0;
  virtual Vec3i iTransformWorldToLHScreen (const Vec3 &where) const = 0;
  // The RHScreen coordinate system is like LHScreen, but we negate the Z
  // coordinate, so we get a right-handed coordinate system.
  virtual Vec3 transformWorldToRHScreen (const Vec3 &where) const = 0;
  virtual Vec3 transformRHScreenToWorld (const Vec3 &where) const = 0;
  // Given a matrix that transforms screen coordinates, return the same matrix
  // transforming world coordinates.
  virtual Matrix4 transformRHScreenToWorld (const Matrix4 &mat) const = 0;
  virtual Quaternion transformRHScreenToWorld (const Quaternion &q) const = 0;
  virtual Quaternion transformWorldToRHScreen (const Quaternion &q) const = 0;
  virtual BoundingSphere transformRHScreenToWorld (const BoundingSphere &bs)
    const = 0;
  virtual BoundingSphere transformWorldToRHScreen (const BoundingSphere &bs)
    const = 0;

  // World coordinate that maps to window coordinates 0,0,0.
  virtual void setOrigin (const Vec3 &o) = 0;
  virtual Vec3 getOrigin () const = 0;
  // The rotation that is applied to the world before it is mapped to the
  // screen.  We rotate the world about its origin.  So we start with a point p
  // in the world, subtract the origin, rotate, scale, and then flip the Z.
  virtual void setRotation (const Quaternion &q) = 0;
  virtual Quaternion getRotation () const = 0;
  // The screen Z coordinate of the nearest pixel we will draw.
  virtual Float getNearPlane () const = 0;
  // Yadda yadda farthest yadda yadda.
  virtual Float getFarPlane () const = 0;

  // The number of pixels between successive lines may be greater than the
  // width of the frame.  Use this to find the next line.
  virtual int getFrameBufferPixelsPerLine () const = 0;
  // The Y coordinate for the depth buffer increases as you go down on the
  // screen. 
  virtual Depth *getDepthBuffer () const = 0;
  // Whether we are double-buffered.
  virtual bool doubleBuffered () const = 0;
  // Copy the back buffer to the front buffer if we are double-buffered.  If we
  // are not double buffered, then the behavior is undefined (it will fail an
  // assertion in a debug build).  After the swap, the back buffer will be
  // unchanged and it will be visible.
  virtual void swapBuffers () const = 0;
  // Copy the framebuffer to the screen, and then XSync.  If we are
  // double-buffered, this draws to the back buffer, then it swaps buffers,
  // then it XSync's.
  virtual void drawFrame () const = 0;
  // Copy the given rectangle from the color buffer to the screen.  If we are
  // double-buffered, this copies the rectangle to the back buffer.
  virtual void drawRect (const ScreenRect &r) const = 0;
  // After a bunch of drawRect calls, you'd better call sync before making
  // further changes to the color buffer, otherwise it may draw wrong.
  virtual void sync () const = 0;
  // Clears out both the color and depth buffers.
  virtual void clearFrame () const = 0;
  // Clear part of the color buffer.
  virtual void clearColorRect (const ScreenRect &r) const = 0;
  // Set part of the depth buffer to the given pixel value.
  virtual void clearDepthRect (const ScreenRect &r, Depth d) const = 0;
};
#endif
