// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __BinSet_h__
#define __BinSet_h__

#ifndef __Point_h__
#include "Point.h"
#endif

#ifndef __Rect_h__
#include "Rect.h"
#endif

#ifndef __string_h__
#include <string.h>
#define __string_h__
#endif

class BinSet {
  // Represents a set of bins on the X-Y plane.  The extent is no more
  // than clipBufWidth by clipBufWidth.
  // FIXME Could do this with 64 64-bit ints and it would be faster, 
  // maybe, but more error prone and it would break totally if I
  // increase clipBufWidth.
  // An insulated implementation of BinSet measurably increases time
  // per frame by a millisecond or so. 
  typedef short BinListIndex;
  static const int clipBufWidth = XObjectDrawer::clipBufWidth;
  // -1 for the elements that aren't in the set, the position in
  // binList for the elements that are in the set.
  BinListIndex m_bins [clipBufWidth] [clipBufWidth];
  Dynavec <Point> m_binList;
  // Check that x and y are in bounds.
  inline void check (int x, int y) const {
    (void) x; (void) y;
    assert (0 <= x && x < clipBufWidth && 0 <= y && y < clipBufWidth);
  }
public:
  // Empty it out.
  // FIXME Optimal strategy depends on whether the set is mostly empty or
  // mostly full.  If it's mostly empty, then just clear out the ones in
  // m_binList.  If it's mostly full, then do it the same way as when we
  // created the object.  clipBufWidth is much more than we normally use, so
  // we do the first strategy.
  void empty () {
    const int size = m_binList.size ();
    for (int i = 0; i < size; i++) {
      Point p = m_binList [i];
      m_bins [p.x] [p.y] = -1;
    }
    m_binList.extendTo (0);
#ifndef NDEBUG
    for (int i = 0; i < clipBufWidth; i++) {
      for (int j = 0; j < clipBufWidth; j++) {
	assert (-1 == m_bins [i] [j]);
      }
    }
#endif
  }
  // Makes an empty one.
  BinSet () {
    assert (sizeof (m_bins) ==
	    clipBufWidth * clipBufWidth * sizeof (BinListIndex));
    assert (clipBufWidth * clipBufWidth  ==
	    (BinListIndex) clipBufWidth * clipBufWidth);
    memset ((void *) m_bins, -1, sizeof (m_bins));
    m_binList.extendTo (0);
  }
  BinSet (const BinSet &b) {
    m_binList = b.m_binList;
    for (int i = 0; i < clipBufWidth; i++) {
      for (int j = 0; j < clipBufWidth; j++) {
	m_bins [i] [j] = b.m_bins [i] [j];
      }
    }
  }
  // Say whether a point is in the set.
  bool isElement (int x, int y) const {
    check (x, y);
    return -1 != m_bins [x] [y];
  }
  // Whether all the elements of the rect are elements.
  bool isElement (const Rect &r) {
    for (int i = r.minx; i < r.maxx; i++) {
      for (int j = r.miny; j < r.maxy; j++) {
	if (!isElement (i, j)) return false;
      }
    }
    return true;
  }
  void remove (int x, int y) {
    check (x, y);
    BinListIndex & where = m_bins [x] [y];
    if (-1 != where) {
      Point &wherePoint = m_binList [where];
      assert (wherePoint.x == x && wherePoint.y == y);
      if (where < m_binList.size () -1) {
	wherePoint = m_binList[m_binList.size () - 1];
	assert (m_binList.size () - 1 ==
		m_bins [wherePoint.x] [wherePoint.y]);
	m_bins [wherePoint.x] [wherePoint.y] = where;
      }
      m_binList.pop ();
      where = -1;
    }
  }
  void remove (const Point &p) {
    remove (p.x, p.y);
  }
  void remove (const Rect &r) {
    for (int i = r.minx; i < r.maxx; i++) {
      for (int j = r.miny; j < r.maxy; j++) {
	remove (i, j);
      }
    }
  }
  void insert (int x, int y) {
    check (x, y);
    BinListIndex &where = m_bins [x] [y];
    if (-1 == where) {
      where = m_binList.size();
      m_binList.push (Point (x, y));
    }
  }
  void insert (const Point &p) {
    insert (p.x, p.y);
  }
  void insert (const Rect &r) {
    for (int i = r.minx; i < r.maxx; i++) {
      for (int j = r.miny; j < r.maxy; j++) {
	insert (i, j);
      }
    }
  }
  bool isEmpty () const {
    return 0 == m_binList.size();
  }
  // anyElement returns any element.  It doesn't remove it from the set.
  Point anyElement () const {
    assert (0 != m_binList.size());
    return m_binList [0];
  }
  // Can't call it "union", that's a keyword.
  void doUnion (const BinSet &b) {
    for (int i = 0; i < b.m_binList.size(); i++) {
      insert (b.m_binList [i]);
    }
  }
  int numElements () const {
    return m_binList.size();
  }
  Point getElement (int i) const {
    return m_binList [i];
  }
};

#endif
