// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __SP_h__
#define __SP_h__

#ifndef __myassert_h__
#include "myassert.h"
#endif

// Smart pointers.  They manage the reference count of the object
// pointed to (see the class Refcount) and free the object when the
// reference count goes to zero.
template <class T> class SP {
  T* m_dest;
  void ref () {
    if (m_dest) {
      m_dest->ref ();
    }
  }
  void deref (T *ptr) {
    if (ptr) {
      ptr->deref ();
      if (0 == ptr->count()) {
	delete ptr;
      }
    }
  }
 public:
  SP (T* ptr) : m_dest (ptr) {
    ref ();
  }
  SP () {
    m_dest = 0;
  }
  SP (const SP &o) : m_dest (o.m_dest) {
    ref ();
  }
  ~SP () {
    deref (m_dest);
    m_dest = 0;
  }
  T& operator* () const {
    assert (m_dest);
    assert (m_dest->count ());
    return *m_dest;
  }
  T* operator-> () const {
    assert (m_dest);
    assert (m_dest->count ());
    return m_dest;
  }
  SP &operator= (T* t) {
    if (t != m_dest) {
      // If our dereference leads to the object being destroyed, and the
      // destructor looks at the value of the smart pointer, we want the
      // destructor to see that the assignment has already been done.
      T* old = m_dest;
      m_dest = t;
      // Increase the reference count on m_dest before dereferencing old, so
      // that if dereferencing old causes it to be destructed and the
      // destructor for old looks at the smart pointer it won't see an
      // object with reference count 0.
      ref();
      deref (old);
    }
    return *this;
  }
  SP &operator= (const SP &p) {
    return (*this = (T*) p);
  }
  operator T* () const {
    return m_dest;
  }
};

#endif
