// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "MemoryUtil.h"

#ifndef NDEBUG
#ifndef __mcheck_h__
#include <mcheck.h>
#define __mcheck_h__
#endif
#endif

// For cout, cerr.
#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

// For malloc.
#ifndef __stdlib_h__
#include <stdlib.h>
#define __stdlib_h__
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

// For strcmp.
#ifndef __string_h__
#include <string.h>
#define __string_h__
#endif

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

namespace {
  
#define HEADERWORD  0xfeeddeed
#define VIRGINWORD  0xbabeface
#define DELETEDWORD 0xdeadbeef
#define FOOTERWORD  0xfadedebb
  
  struct newLeakInfo {
    int line;
    const char *file;
    newLeakInfo *next;
    newLeakInfo *prev;
    size_t givenSize;
    unsigned int header;
  };
  
  // Compare two of them, for sorting, someday.  For now I'm using
  // sort-lines in an emacs buffer so this code isn't called yet. 
  bool operator> (const newLeakInfo &p1, const newLeakInfo &p2) {
    int cmp = strcmp (p1.file, p2.file);
    if (0 == cmp) {
      if (p1.line > p2.line) {
	return true;
      } else if (p1.line < p2.line) {
	return false;
      } else {
	if (&p1 > &p2) {
	  return true;
	} else {
	  return false;
	}
      }
    } else if (0 < cmp) {
      return false;
    } else {
      assert (0 > cmp);
      return true;
    }
  }

  // Statics are initialized to zero, but don't have a run-time
  // initializer here because this code has to work before all of the
  // run-time initializers have been done. 
  newLeakInfo *listHeader;
  
  size_t roundUpSize (size_t size) {
    // 4 bytes for the footer, 3 bytes for rounding up.
    return ((size + 7) >> 2) << 2;
  }
} // namespace

// Code I do not control calls the standard new and new[].  Redirect
// them here to have a useless line number and file name, so we can
// still detect leaks and memory corruption and we don't get confused
// later in delete.
void *operator new (size_t size) {
  return operator new (size, __LINE__, __FILE__);
}

void *operator new [] (size_t size) {
  return operator new [] (size, __LINE__, __FILE__);
}

void *operator new (size_t size, int line, const char *file) {
#ifndef NDEBUG
  static bool did_mcheck;
  if (!did_mcheck) {
    did_mcheck = 1;
    mcheck (0);
    cerr << "Initialized mcheck." << endl;
  }
#endif
  if (MemoryUtil::frequentCheck) {
    MemoryUtil::check ();
  }
  // The number of bytes to actually allocate
  size_t allocSize = roundUpSize (size);
  // Number of ints.
  size_t intSize = allocSize >> 2;
  assert (allocSize >= size);
  assert (0 == allocSize % 4);
  if (! listHeader) {
    listHeader = (newLeakInfo *) malloc (sizeof (newLeakInfo));
    assert (listHeader);
    listHeader->line = -1;
    listHeader->file = 0;
    listHeader->next = listHeader;
    listHeader->prev = listHeader;
    listHeader->givenSize = 0;
    listHeader->header = HEADERWORD;
  }
  newLeakInfo *ptr = (newLeakInfo *) malloc (allocSize+sizeof(newLeakInfo));
  assert (ptr);
  ptr->line = line;
  ptr->file = file;
  // Insert ourselves after listHeader->prev and  before listHeader.
  listHeader->prev->next = ptr;
  ptr->prev = listHeader->prev;
  ptr->next = listHeader;
  ptr->givenSize = size;
  ptr->header = HEADERWORD;
  listHeader->prev = ptr;
  int *result = (int *) (ptr + 1);
  result [intSize - 1] = FOOTERWORD;
  for (int i = intSize - 2; i >= 0; i--) {
    result [i] = VIRGINWORD;
  }
  return (void *) result;
}

void *operator new [] (size_t size, int line, const char *file) {
  return operator new (size, line, file);
}

void operator delete (void *ptr) {
  // delete (0) is okay, and the g++ I/O library actually does it.
  if (!ptr) return;
  if (MemoryUtil::frequentCheck) {
    MemoryUtil::check ();
  }
  newLeakInfo *toFree = ((newLeakInfo *) ptr) - 1;
  if (HEADERWORD == toFree->header) {
    assert (toFree->prev);
    assert (toFree->next);
    newLeakInfo *oldPrev = toFree->prev;
    newLeakInfo *oldNext = toFree->next;

    if (HEADERWORD != oldPrev->header || HEADERWORD != oldNext->header) {
      cerr << "Corrupted memory free list" << endl;
      abort ();
    }
    oldNext->prev = oldPrev;
    oldPrev->next = oldNext;
    int freeIntSize =
      (sizeof (newLeakInfo) + roundUpSize (toFree->givenSize)) >> 2;
    unsigned int *toFreeInt = (unsigned int *) toFree;
    if (FOOTERWORD != toFreeInt [freeIntSize - 1]) {
      cerr << "Corrupted footer in memory allocated at line "
	   << toFree->line << " of "
	   << toFree->file << endl;
      abort ();
    }
    for (int i = 0; i < freeIntSize; i++) {
      toFreeInt [i] = DELETEDWORD;
    }
#if 0
    {
      // The point here is that if you leak everything, then you can
      // put a conditional breakpoint here to stop when a particular
      // address is allocated, and you can be sure that it won't be
      // allocated again.
      static bool complained = false;
      if (!complained) {
	cout << "Remember, we've decided to deliberately leak "
	  "everything, for debugging. " << endl;
	complained = true;
      }
    }
#else
    free ((void *) toFree);
#endif
  } else {
    cerr << "Delete of pointer not from new" << endl;
    abort ();
  }
}

void operator delete [] (void *x) throw () {
  operator delete (x);
}

namespace {
  struct MemoryUtilData
    : public Refcount
  {
    Dynavec <void (*) ()> m_deallocators;
  };
  // With the straightforward code 
  // Data* allDeallocators = 0;
  // we lose if some other class has their static initializers before
  // ours, they register a deallocator, then we get initialized and
  // drop their deallocator.  The more involved code below prevents
  // that.  Same trick was used in FactoryTable.
  MemoryUtilData* initData ();
  MemoryUtilData* allDeallocators = initData ();
  MemoryUtilData* initData () {
    if (allDeallocators) {
      return allDeallocators;
    } else {
      allDeallocators = NEW (MemoryUtilData ());
      allDeallocators->ref();
      return allDeallocators;
    }
  }
  void runDeallocators () {
    if (allDeallocators) {
      for (int i = 0; i < allDeallocators->m_deallocators.size(); i++) {
	allDeallocators->m_deallocators[i] ();
      }
      allDeallocators->deref ();
      assert (0 == allDeallocators->count());
      delete allDeallocators;
      allDeallocators = 0;
    }
  }
}

void MemoryUtil::registerDeallocator (void (*f) ()) {
  initData ();
  allDeallocators->m_deallocators.push (f);
}

void MemoryUtil::listLeaks () {
  runDeallocators ();
  if (listHeader->next == listHeader) {
    cout << "No leaks!" << endl;
  } else {
    // Might like to change the if 0 to ifndef NDEBUG.
#ifndef NDEBUG
    cout << "Leaks were found!" << endl;
#else
    cout << "These leaks were found:" << endl;
#endif
  }
  // Sort it here, someday.  But not now.
#ifndef NDEBUG
  { // Print.
    newLeakInfo *ptr = listHeader->next;
    for (;;) {
      if (listHeader == ptr) break;
      // The format in the next line lets me do gnu-emacs next-error to
      // find the next leak.
      cout << ptr->file << ":" << ptr->line << ": " << ptr->givenSize
	   << " bytes allocated at " << (void *) (ptr+1) << "."
	   << endl;
      ptr = ptr->next;
    }
  }
#endif
}

bool MemoryUtil::frequentCheck = false;

void MemoryUtil::check () {
  newLeakInfo *here = listHeader->next;
  for (;;) {
    if (here == listHeader) break;
    if (HEADERWORD != here->header) {
      cerr << "Corrupted header word at " << &here->header << endl;
      abort ();
    }
    int hereIntSize =
      (sizeof (newLeakInfo) + roundUpSize (here->givenSize)) >> 2;
    unsigned int *hereInt = (unsigned int *) here;
    if (FOOTERWORD != hereInt [hereIntSize - 1]) {
      cerr << "Corrupted footer word at " << &hereInt [hereIntSize - 1] << endl;
      abort ();
    }
    here = here -> next;
  }
}
