// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __FloatUtil_h__
#define __FloatUtil_h__

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

struct FloatUtil {

  static const Float PI;

  // Yucky C header files sometimes define min, max, and abs as macros.
#ifdef max
   #undef max
#endif

#ifdef min
   #undef min
#endif

#ifdef abs
   #undef abs
#endif

  template <class T> static inline T max (T a, T b) {
#ifdef __GNUC__
    return a >? b;
#else
    return (a > b)?a:b;
#endif
  }

  template <class T> static inline T min (T a, T b) {
#ifdef __GNUC__
    return a <? b;
#else
    return (a < b)?a:b;
#endif
  }

  template <class T> static inline T max (T a, T b, T c) {
#ifdef __GNUC__
    return a >? b >? c;
#else
    return max (a, (max (b, c)));
#endif
  }

  template <class T> static inline T min (T a, T b, T c) {
#ifdef __GNUC__
    return a <? b <? c;
#else
    return min (a, (min (b, c)));
#endif
  }

  template <class T> static inline T abs (T a) {
    return max (a, -a);
  }

  template <class T> static inline T sqr (T a) {
    return a * a;
  }

  static Float sqrt (Float a);

  static Float acos (Float a);
  static Float asin (Float a);

  static inline int floor (double x) {
    const int i = (int) x;
    int result;
    if (x > 0) {
      result = i;
    } else if (i == x) {
      return i;
    } else {
      // (int) rounds negative numbers up.
      result = i - 1;
    }
    assert (x >= result);
    assert (1 > (x - result));
    return result;
  }

  static inline int floor (float x) {
    const int i = (int) x;
    int result;
    if (x > 0) {
      result = i;
    } else if (i == x) {
      return i;
    } else {
      // (int) rounds negative numbers up.
      result = i - 1;
    }
    assert (x >= result);
    assert (1 > (x - result));
    return result;
  }

  static inline int ceiling (double x) {
    const int i = (int) x;
    int result;
    if (x < 0) {
      return i;
    } else if (i == x) {
      return i;
    } else {
      return i + 1;
    }
    assert (x <= result);
    assert (1 > (result - x));
    return result;
  }

  static inline int ceiling (float x) {
    const int i = (int) x;
    int result;
    if (x < 0) {
      return i;
    } else if (i == x) {
      return i;
    } else {
      return i + 1;
    }
    assert (x <= result);
    assert (1 > (result - x));
    return result;
  }

  // I think it's really cool to be able to do the following without any
  // #ifdef's.  Three cheers for C++.
  inline static bool isNAN (double d) {
    unsigned long long int dint;
    *((double *) &dint) = d;
    unsigned long long int nandmask = 0x7ff00000;
    nandmask = nandmask << 32;
    return (dint & nandmask) == nandmask;
  }
  
  inline static bool isNAN (float d) {
    unsigned int dint;
    *((float *) &dint) = d;
    unsigned long int nandmask = 0x7f800000;
    return (dint & nandmask) == nandmask;
  }

};

#endif
