// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Dynavec_h__
#define __Dynavec_h__

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

// Dynamically resizing vectors.
// C++ book says that the argument for the formal parameter T doesn't
// really have to be a class.
template <class T> class Dynavec {
  T* v;
  int used;
  int allocated;
  static const int defaultInitial = 2;
  T defaultArray [defaultInitial];
  inline void checkme () const {
#ifndef NDEBUG
    assert (v);
    assert (0 <= used);
    assert (used <= allocated);
#endif
  }
  inline void init () {
    v = defaultArray;
    used = 0;
    allocated = defaultInitial;
    checkme ();
  }
  inline void copy (const Dynavec &other) {
    extendTo (other.size ());
    for (int i = 0; i < other.size(); i++) {
      v [i] = other [i];
    }
    checkme ();
  }
 public:
  Dynavec (const Dynavec &v) {
    init ();
    copy (v);
  }
  Dynavec & operator=(const Dynavec &v) {
    if (&v != this) {
      copy (v);
    }
    return *this;
  }
  Dynavec () {
    init ();
  }
  Dynavec (const T &e0) {
    init ();
    extendTo (1);
    v [0] = e0;
  }
  Dynavec (const T &e0, const T &e1) {
    init ();
    extendTo (2);
    v [0] = e0;
    v [1] = e1;
  }
  Dynavec (const T &e0, const T &e1, const T &e2) {
    init ();
    extendTo (3);
    v [0] = e0;
    v [1] = e1;
    v [2] = e2;
  }
  Dynavec (const T &e0, const T &e1, const T &e2, const T &e3) {
    init ();
    extendTo (4);
    v [0] = e0;
    v [1] = e1;
    v [2] = e2;
    v [3] = e3;
  }
  int size () const {
    return used;
  }
  void extendTo (int newSize) {
    checkme ();
    if (newSize > allocated) {
      int toAlloc = newSize;
      if (toAlloc < allocated * 2) {
	toAlloc = allocated * 2;
      }
      T *newv = NEW (T [toAlloc]);
      for (int i = 0; i < used; i++) {
	newv [i] = v[i];
      }
      if (defaultArray != v) {
	delete [] v;
      }
      v = newv;
      allocated = toAlloc;
    }
    used = newSize;
    checkme ();
  }
  void extendTo (int newSize, const T& newValue) {
    const int oldSize = size ();
    extendTo (newSize);
    for (int i = oldSize; i < newSize; i++) {
      v [i] = newValue;
    }
  }
  void push (const T& elt) {
    extendTo (used+1);
    v [used - 1] = elt;
    checkme ();
  }
  void push () {
    extendTo (used+1);
    checkme ();
  }
  T pop () {
    checkme ();
    assert (0 < used);
    used--;
    return v [used];
  }
  T& operator [] (int index) {
    checkme ();
    assert (0 <= index);
    assert (index < used);
    return v[index];
  }
  // *dynavec is like &dynavec[0], except *dynavec is always
  // meaningful even if the vector has length 0.  This is good as an
  // argument to memcpy, memmove, etc., if the next step will be to
  // copy zero bytes.
  T& operator * () {
    return *v;
  }
  const T& operator * () const {
    return *v;
  }
  const T& operator [] (int index) const {
    checkme ();
    assert (0 <= index);
    assert (index < used);
    return v[index];
  }
  ~Dynavec () {
    checkme ();
    if (defaultArray != v) {
      delete [] v;
    }
    v = 0;
    used = 0;
    allocated = 0;
  }
};

#endif
