// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __AnyMouseTrackball_h__
#include "AnyMouseTrackball.h"
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __AnyMouseTrackballConfiguration_h__
#include "AnyMouseTrackballConfiguration.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __KeyboardMouseEvent_h__
#include "KeyboardMouseEvent.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __TransformConfiguration_h__
#include "TransformConfiguration.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

AnyMouseTrackball::AnyMouseTrackball ()
  : TypedFactory <AnyMouseTrackballConfiguration, Action> ("AnyMouseTrackball")
{}

SP<AnyMouseTrackballConfiguration> AnyMouseTrackball::typedDefaultConfiguration () const {
  return NEW (AnyMouseTrackballConfiguration ());
}

String AnyMouseTrackball::getPrettyName (const Configuration *c) const {
  SP<const AnyMouseTrackballConfiguration> conf =
    dynamic_cast <const AnyMouseTrackballConfiguration *> (c);
  assert (conf);
  return ("Use the mouse as a trackball to " +
	  conf->getTransformAction()->unParse());
}

SP<Action> AnyMouseTrackball::makeIt (AnyMouseTrackballConfiguration *conf)
  const
{
  SP<TopLevel> top = conf->getTopLevel ();
  SP<const KeyboardMouseEvent> e =
    dynamic_cast <const KeyboardMouseEvent *> (&*top->getEvent());
  assert (e);
  const int oldX = conf->getX();
  const int oldY = conf->getY();
  const int x = e->getX();
  const int y = e->getY();
  conf->setX (x);
  conf->setY (y);
  SP<Action> result = NEW (Action ());
  if (conf->isFirst ()) {
    if (e->getButton () == e->pointerMotionButton ()) {
      // The problem here is that this action looks at changes to the mouse
      // position to decide what to do.  We can't grab the mouse if we're
      // bound to pointer motion because then no other commands can run, and we
      // can't observe changes to the mouse position if we can't grab the
      // mouse.
      result->setProblem ("It does not make sense to bind AnyMouseTrackball "
			  "to pointer motion");
    } else {
      SelectionManager::setSelectionBlink (false);
      conf->grabIfFirst (this);
    }
  } else {
    if (conf->isLast ()) {
      SelectionManager::setSelectionBlink (true);
    }
    if (e->getButton() == e->pointerMotionButton ()) {
      const int dx = x - oldX;
      const int dy = y - oldY;
      SP<RecursiveSlotValue> rsv = conf->getTransformAction();
      SP<TransformConfiguration> tc =
	dynamic_cast <TransformConfiguration *> (&*rsv->getConfiguration());
      assert (tc);
      SP<Canvas> canvas = top->getCanvas ();
      Vec3 tbCenter =
	canvas->transformWorldToRHScreen
	(top->getSceneGraph ()->centerOfSomething ());
      tc->setCenter (tbCenter);
      const Float radius = conf->getRadius ();
      const Vec3 oldDiff = Vec3 (oldX, oldY, tbCenter [2]) - tbCenter;
      const Vec3 newDiff = Vec3 (x, y, tbCenter [2]) - tbCenter;
      // We don't want things to twist around suddenly if the mouse goes over
      // the center of rotation, so only do twisting if the mouse starts
      // and ends more than the radius from the center of rotation.
      const bool isFar =
	oldDiff.length () > radius && newDiff.length () > radius;
      Quaternion rollingQuat;
      {
	// rollingVec will be the motion vector we will consider for rolling
	Vec3 rollingVec;
	if (isFar) {
	  // We'll use the part that's a rotation about the center for
	  // twisting, below, so to keep people from going nuts due to
	  // non-orthognal controls, we should only use the radial part for
	  // rolling here.
	  Vec3 oldDiffUnit = oldDiff.makeUnit ();
	  rollingVec = (oldDiffUnit * Vec3 (dx, dy, 0)) * oldDiffUnit;
	} else {
	  rollingVec = Vec3 (dx, dy, 0);
	}
	Float angle = rollingVec.length () / radius;
	if (angle > 0) {
	  const Vec3 axis = Vec3 (0, 0, 1).cross (Vec3 (rollingVec [0],
							rollingVec [1],
							1));
	  rollingQuat = Quaternion::makeRotation (angle, axis);
	} else {
	  rollingQuat = Quaternion ();
	}
      }
      Quaternion twistingQuat;
      {
	if (isFar) {
	  Vec3 start = oldDiff.makeUnit ();
	  Vec3 end = newDiff.makeUnit ();
	  Vec3 axis = start.cross (end);
	  Float angle = FloatUtil::asin (axis.length ());
	  if (axis.length () > 0) {
	    twistingQuat = Quaternion::makeRotation (angle, axis);
	  } else {
	    twistingQuat = Quaternion ();
	  }
	}
      }
      // Took an arbitrary decision here about what order to multiply them.
      // They ought to be small and nearly commutative, anyway.
      tc->setRotation (twistingQuat * rollingQuat);
      tc->setTranslation (Vec3 (0,0,0));
      tc->setTopLevel (top);
      result = dynamic_cast <Action *> (&*rsv->makeIt());
      assert (result);
    }
  }
  if (result->isProblem ()) {
    SelectionManager::setSelectionBlink (true);
  }
  return result;
}

namespace {
  static const bool useless =
  (FactoryTable::store ("KeyboardMouseAction", NEW (AnyMouseTrackball ())),
   true);
}

SP<RecursiveSlotValue> AnyMouseTrackball::mkMouseTrackball
(RecursiveSlotValue *slot)
{
  SP<AnyMouseTrackballConfiguration> conf =
    NEW (AnyMouseTrackballConfiguration ());
  conf->setTransformAction (slot);
  SP<RecursiveSlotValue> rsv = NEW (RecursiveSlotValue ("KeyboardMouseAction",
							"AnyMouseTrackball"));
  rsv->setConfiguration (&*conf);
  return rsv;
}

SP<RecursiveSlotValue> AnyMouseTrackball::mkMouseTrackball (const String &name) {
  return mkMouseTrackball (NEW (RecursiveSlotValue ("TransformAction", name)));
}
