// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __TopLevel_h__
#define __TopLevel_h__

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __TopConfiguration_h__
#include "TopConfiguration.h"
#endif

class Canvas;
class Event;
class Vec3;
class SceneGraph;
class Configuration;
class ActionConfiguration;
class Action;
class InputDevice;

// One instance of this for X, maybe one for Windows
// someday.  Maybe one for raw VGA, too, since that ought to give us
// the best frame rate (assuming full screen is good).
class TopLevel 
  : public Configurable
{
  struct TopLevelData;
  TopLevelData *m_data;
public:
  TopLevel ();
  ~TopLevel ();
  // This will do all the work, doing physics and updating the
  // screen.  If a keyboard key is pressed, it will configure and
  // invoke an action.
  // mainLoop returns true if the user indicated that a new TopLevel
  // should be made and then reentered, or false if the main program
  // should exit.
  virtual bool mainLoop () = 0;
  // If this is called, then mainLoop will exit with the value false
  // next time it gets to its top level.  For the glut top level, this
  // will simply exit, since there isn't any other way for us to get
  // out of glut's top level.
  virtual void doExit () = 0;
  // If this is called, then mainLoop will exit with the value true
  // next time it gets to its top level.  For the glut top level, this
  // will abort, since we can't exit glut's top level in a civilized
  // way.
  virtual void newWindowSystem () = 0;

  // You can get and set the scene graph in the TopLevel.  
  virtual SP<SceneGraph> getSceneGraph () = 0;
  virtual SP<const SceneGraph> getSceneGraph () const = 0;
  // Next one is virtual because the TopLevel has some work to do
  // with the SceneGraph when it is set to plug it into the numerical
  // integration and rendering schemes.
  virtual void setSceneGraph (SP<SceneGraph> sg) = 0;

  // The TopLevel knows the top configuration of everything.  We need
  // this so it can invoke actions that edit or read the configuration.
  SP<TopConfiguration> getConfiguration ();
  SP<const TopConfiguration> getConfiguration () const;
  void setConfiguration (SP<TopConfiguration> configuration);

  // Start or stop time.
  virtual void setIntegrating (bool b) = 0;
  virtual bool getIntegrating () const = 0;

  // Is the mouse pointing at anything?  Actions can call this if they care.
  // If we found something, return true, otherwise return false.
  // If we found something, index is filled in with the index of the
  // object we found, and intersectionPoint is filled in with the
  // world coordinates of the point of intersection.
  // If we did not find anything, index and intersectionPoint are left
  // unchanged.
  // The TopLevel should remember the state of the mouse since this
  // function doesn't take that state as an argument.
  virtual bool findUnderMouse (int &index, Vec3 &intersectionPoint)
    const = 0;

  // Times are in milliseconds.
  struct Times {
    int lastDrawTime;    // Time to render last frame.
    int lastIntTime;     // Time of last integration time step
    int totalIntTime;    // Total time of all integration time steps
                         // for last frame. 
    int interFrameTime;  // Time between frames.
    Times ()
      : lastDrawTime (0),
	lastIntTime (0),
	totalIntTime (0),
	interFrameTime (0)
    {}
  };
  virtual Times getTimes () const = 0;
  // The last event received from the user.
  // FIXME this doesn't need to be virtual any more
  // It should always return m_getDevice->getEvent().
  virtual SP<const Event> getEvent () const = 0;
  // The last input device that received input from the user.
  virtual SP<const InputDevice> getDevice () const = 0;
  //
  // Grab input.  Here's the idea:
  // Normally, an Action just gets the event that started it, and that's all.
  // This frees simple Actions from having to pay attention to events.
  //
  // If an Action calls grab (...) on the top level, then the RSV it passes
  // into grab will get called again for all following
  // events, until the top level determines that it's time to release the grab.
  // The TopLevel will release the grab when the key that was most recently
  // pressed at the time that the grab was done is released, which is usually
  // what you want, so actions that grab will generally automatically have the
  // grab for the correct amount of time.  If your action wants to have the
  // grab for a different amount of time, it is prudent to just call grab again
  // for every event that is received, and then to call grab (0) when the grab
  // is done.  This way, the action can have what it wants without having to
  // anticipate when the TopLevel will automatically release its grab.
  // Grabbing is quite efficient, so don't worry about the compute power lost
  // in regrabbing.
  // 
  // The rsv should have a factory that will make an Action, and its
  // configuration should be an ActionConfiguration.
  //
  // If an action reports a problem, it will lose any grab it has.  It can also
  // deliberately lose the grab by calling grab with a null pointer.
  // 
  // FIXME This doesn't have to be virtual any more.
  // Should be the same as getDevice->grab (rsv).
  virtual void grab (SP<RecursiveSlotValue> rsv) = 0;

  // Get the canvas, so actions can do things like change the point of
  // view.
  virtual Canvas *getCanvas () = 0;
};

#endif
