// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "TopConfiguration.h"

#ifndef __VectorSceneConfiguration_h__
#include "VectorSceneConfiguration.h"
#endif

#ifndef __VectorSlotValue_h__
#include "VectorSlotValue.h"
#endif

#ifndef __TestSceneConfiguration_h__
#include "TestSceneConfiguration.h"
#endif

#ifndef __SceneLoader_h__
#include "SceneLoader.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __DiskConfiguration_h__
#include "DiskConfiguration.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __PositiveIntSlotValue_h__
#include "PositiveIntSlotValue.h"
#endif

#ifndef __ConstantRecursiveSlotValue_h__
#include "ConstantRecursiveSlotValue.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __ConstantRecursiveSlotValue_h__
#include "ConstantRecursiveSlotValue.h"
#endif

namespace {
  // Don't have a static variable with a string, it looks like a memory leak.
  // Instead use no-argument functions.
  const String sceneGraphName () {
    return "Scene Graph";
  }

  const String sceneName () {
    return "Scene Generator";
  }

  const String stepsPerFrameName () {
    return "Timesteps Per Frame";
  }

  const String toplevelName () {
    return "Top Level";
  }

  const String objectDrawerName () {
    return "Object Drawer";
  }

  SP<RecursiveSlotValue> makeTestConf (const String &className,
				       SP<Configuration> objConf,
				       const Vec3 &translation,
				       int repeat,
				       Float spacing)
  {
    SP<RecursiveSlotValue> result = NEW (RecursiveSlotValue ("SceneLoader",
							     "TestSceneLoader"));
    SP<TestSceneConfiguration> sceneConf =
      dynamic_cast <TestSceneConfiguration *> (result->getConfiguration());
    assert (sceneConf);
    SP<RecursiveSlotValue> rsv = NEW (RecursiveSlotValue ("PhysicsObject",
							  className));
    rsv->setConfiguration (objConf);
    sceneConf->setRecursiveSlotValue (rsv);
    sceneConf->setWhere (translation);
    sceneConf->setCubeSize (repeat);
    sceneConf->setCubeSpace (spacing);
    return &*result;
  }
  
  static SP<RecursiveSlotValue> diskAt (const Vec3 &where,
					const Vec3 &normal)
  {
    SP<DiskConfiguration> di = NEW (DiskConfiguration ());
    di->setDefaultNorm (normal);
    return makeTestConf ("Disk", &*di, where, 1, 1);
  }

  static SP<RecursiveSlotValue> cubeAt (const String &name,
					const Vec3 &where, int size,
					Float spacing)
  {
    SP<Factory> f = FactoryTable::load ("PhysicsObject", name);
    assert (f);
    return makeTestConf (name, f->defaultConfiguration(),
			 where, size, spacing);
  }

  static SP<RecursiveSlotValue> initialSceneLoader () {
    SP<RecursiveSlotValue> result =
      NEW (RecursiveSlotValue ("SceneLoader", "VectorSceneLoader"));
    SP<Configuration> conf = result->getConfiguration ();
    assert (conf);
    SP<VectorSceneConfiguration> vsc =
      dynamic_cast <VectorSceneConfiguration *> (&*conf);
    assert (vsc);
    SP<VectorSlotValue> vsv = vsc->getVectorSlotValue ();
    vsv->extendTo (6);
    vsv->setValue (0, &*diskAt (Vec3 ( 0, 5,0), Vec3 (0,-1,1)));
    vsv->setValue (1, &*diskAt (Vec3 ( 0,-5,0), Vec3 (0, 1,1)));
    vsv->setValue (2, &*diskAt (Vec3 ( 5, 0,0), Vec3 (-1,0,1)));
    vsv->setValue (3, &*diskAt (Vec3 (-5, 0,0), Vec3 ( 1,0,1)));
    vsv->setValue (4, &*cubeAt ("RedBouncingBall",Vec3(0,0,5),
				8 ,1));
    vsv->setValue (5, &*cubeAt ("BlueBouncingBall", Vec3(0,0,15), 3, 2));
    return result;
  }
} // namespace

TopConfiguration::TopConfiguration (const String &topLevelClass)
{
  initializeSlot
    (sceneGraphName (),
     NEW (RecursiveSlotValue ("SceneGraph", "SubsetStorageScene")));
  initializeSlot (sceneName (), &*(initialSceneLoader ()));
  initializeSlot (stepsPerFrameName (), NEW (PositiveIntSlotValue (5)));
  initializeSlot (toplevelName (),
		  NEW (RecursiveSlotValue ("TopLevel", topLevelClass)));
}

RecursiveSlotValue *
TopConfiguration::getSceneGraphSlot () {
  return RecursiveSlotValue::getSlotValue (this, sceneGraphName ());
}

RecursiveSlotValue *
TopConfiguration::getSceneLoaderSlot () {
  return RecursiveSlotValue::getSlotValue (this, sceneName ());
}

void TopConfiguration::setSceneLoaderSlot (RecursiveSlotValue *rsv) {
  RecursiveSlotValue::setSlotValue (this, sceneName (), rsv);
}

RecursiveSlotValue *TopConfiguration::getTopLevelSlot () {
  return RecursiveSlotValue::getSlotValue (this, toplevelName ());
}

int TopConfiguration::getStepsPerFrame () const {
  return IntSlotValue::getSlot (this, stepsPerFrameName ());
}

SP<Configuration> TopConfiguration::copy () const {
  return NEW (TopConfiguration (*this));
}

bool TopConfiguration::hasTheSameTypeAs(const Configuration *c) const {
  return 0 != dynamic_cast <const TopConfiguration *> (c);
}
