// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __String_h__
#define __String_h__

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Hashable_h__
#include "Hashable.h"
#endif

class ostream;
class istream;

class String : public Hashable {
  Dynavec <char> m_chars;
  inline void String::check () const {
    assert (m_chars.size() >= 1);
    assert ('\0' == m_chars[m_chars.size()-1]);
  }
public:
  String ();
  virtual ~String ();
  String (char c);
  String (const char *c);
  String (const String &s);
  String (int i);
  String (double i);
  // Chronic problem -- any pointer type converts to bool which converts to
  // string.  The "explicit" keyword on page 423 of the C++ ARM is supposed to
  // solve exactly this problem, and it does!  Yay!
  explicit String (bool i);
  String & operator= (const String &s);
  // On error, return 0.0 and set *error (if specified and nonzero) to a
  // nonempty string describing what went wrong.
  // On success, return the floating point number and leave *error
  // unmodified.  This way, error can be used to accumulate errors
  // from several calls.  If error is not specified, we'll abort with a message
  // if parsing fails.
  // A blank string is an error.  FIXME This is misimplemented now.
  // Fixing would require changes to the PDB reader and is not a high priority
  // right now.
  Float toFloat (int start = 0, String *error = 0) const;
  // chars is the maximum field width.
  Float toFloat (int start, int chars, String *error = 0) const;
  // Same, but for integers.
  int toInt (int start = 0, String *error = 0) const;
  int toInt (int start, int chars, String *error = 0) const;
  // Whether the specified substring is entirely whitespace.
  //
  // FIXME We really need a constant-time substring operation.
  //
  // All these string operations that really work on substrings are silly.
  // Array references into strings would have to return an array-reference
  // object, which would copy the destination string if it has constant-time
  // substrings pointing into it.  Or maybe we just don't allow mutable
  // references into strings and fix the code that wants it to do something
  // else.  Hmm, converting string to char* would have to be disallowed, since
  // we couldn't guarantee the null termination of the char* if it was a
  // substring reference to the middle of some longer string.
  //
  // FIXME isBlank isn't implemented at all.
  bool isBlank (int start, int chars) const;
  bool isBlank (int start = 0) const;
  // Convert a Float to ascii with the given width and insert it into the
  // string.  I would like to use printf for this, but the manual entry says
  // "In no case does a non-existent or small field width cause truncation of a
  // field; if the result of a conversion is wider than the field width, the
  // field is expanded to contain the conversion result."  PDB files really
  // have fixed-width fields, so direct use of printf isn't good enough.
  // Places is the preferred number of digits after the decimal point, although
  // if there isn't room you'll get less than that.  If the integer part can't
  // fit, you should get an error.  If the string has to be extended so it has
  // characters at the required positions, we extend with spaces.
  void fromFloat (Float val, int start, int chars, int places,
		  String *error = 0);
  // Likewise for integers.
  void fromInt (int val, int start, int chars, String *error = 0);
  // Copy the given string into a substring of this.  Left justify and pad with
  // spaces.  Get an error if it doesn't fit.
  void fromString (const String &val, int start, int chars, String *error = 0);
  // Like fromString, except right justify.
  void fromStringRight (const String &val, int start, int chars,
			String *error = 0);
  // Returns a substring of "this" starting at from, and extending to the end
  // of the string or for chars characters, whichever is shorter.
  String substring (int from, int chars) const;
  // x.matches (y, m) is equivalent to
  // x.substring (y, strlen (m)) == m
  // but faster because it avoids the strlen and more importantly it avoids
  // allocating a new string for the substring.
  bool matches (int position, const char *match);
  Hashable::HASH hash () const;
  // Next one is for the Hashable interface.
  String *copy () const;
  friend String operator+ (const String &s1, const String &s2);
  const String &operator+= (const String &s);
  const String &operator+= (char ch);
  bool operator== (const Hashable &s) const; 
  friend bool operator== (const String &s1, const String &s2);
  friend bool operator== (const String &s, const char *cp);
  friend bool operator== (const char *cp, const String &s);
  bool operator!= (const String &s) const;
  friend bool operator< (const String &s1, const String &s2);
  char &operator[] (int i);
  // *s is equivalent to s[0], except you're allowed to do *s even if
  // the string has length 0.  Usual idiom is &*s to convert to a
  // char * pointer.
  char &operator * ();
  const char &operator * () const;
  // Have const char & immediately below instead of just char because I want to
  // be able to take &foo[n] and get a char*.
  const char &operator[] (int i) const;
  // Number of characters in the string.  There's a trailing null
  // hidden at the end, but that's just a representation detail, so
  // it's none of the user's business and it isn't counted in the
  // size.  Putting it inline because it tends to be called repeatedly if the
  // nonemptyness of a string is used as an error indicator.
  int size () const {
    check ();
    return m_chars.size() - 1;
  }

  String upcase () const;
  friend ostream &operator<< (ostream &o, const String& s);
  // Next one reads the newline and discards it.
  friend istream &operator>> (istream &i, String &s);
};

#endif
