// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "String.h"

// For sscanf.
#ifndef __stdio_h__
#include <stdio.h>
#define __stdio_h__
#endif

// Defines ostream.
#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

// For toupper.
#ifndef __ctype_h__
#include <ctype.h>
#define __ctype_h__
#endif

#ifndef __string_h__
#include <string.h>
#define __string_h__
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __stdlib_h__
#include <stdlib.h>
#define __stdlib_h__
#endif

String::~String () {
  check ();
}

String::String () {
  m_chars.push ('\0');
  check ();
}

String::String (char ch) {
  m_chars.push (ch);
  m_chars.push ('\0');
  check ();
}

void copyFromCharStar (const char * const c, Dynavec <char> &m_chars) {
  assert (c);
  int len = strlen (c)+1;
  m_chars.extendTo (len);
  for (int i = 0; i < len; i++) {
    m_chars [i] = c [i];
  }
}

String::String (const char * const c) {
  copyFromCharStar (c, m_chars);
  check ();
}

String::String (const String &s) : Hashable (s) {
  s.check ();
  int len = s.m_chars.size();
  m_chars.extendTo (len);
  for (int i = 0; i < len; i++) {
    m_chars [i] = s.m_chars[i];
  }
  check ();
}

// For sprintf.
#ifndef __stdio_h__
#include <stdio.h>
#define __stdio_h__
#endif

String::String (int i) {
  char buf [50];
  sprintf (buf, "%d", i);
  copyFromCharStar (buf, m_chars);
  check ();
}

String::String (double i) {
  char buf [50];
  sprintf (buf, "%f", i);
  copyFromCharStar (buf, m_chars);
  check ();
}

String::String (bool i) {
  if (i) {
    copyFromCharStar ("true", m_chars);
  } else {
    copyFromCharStar ("false", m_chars);
  }
  check ();
}

String *String::copy () const {
  return NEW (String (*this));
}

String::HASH String::hash () const 
{
  check ();
  int result = 0;
  int s = m_chars.size();
  for (int i = 0; i < s; i++) {
    /* This worked well on Hp200lx... */
    result = (result << 3) ^ m_chars[i] ^ (result >> 11);
  }
  return result;
}

String operator+ (const String &s1, const String &s2) {
  s1.check ();
  s2.check();
  String result = s1;
  // Remove the trailing null.
  result.m_chars.pop ();
  for (int i = 0; i < s2.m_chars.size(); i++) {
    result.m_chars.push (s2.m_chars [i]);
  }
  result.check ();
  return result;
}

bool String::operator== (const Hashable &s) const {
  check ();
  const String *s1 = dynamic_cast <const String *>(&s);
  if (s1) {
    return (*this) == (*s1);
  } else {
    return false;
  }
}

bool operator== (const String &s, const char *cp) {
  s.check();
  // Note that this length includes the trailing null.
  const char *sptr = &*s.m_chars;
  const char *last = sptr + s.m_chars.size();
  for (; sptr < last; sptr++, cp++) {
    if (*sptr != *cp) {
      return false;
    }
  }
  return true;
}

bool operator== (const char *cp, const String &s) {
  return s == cp;
}

bool operator== (const String &s1, const String &s2) {
  // Don't just reinterpret one of them as char *'s and use the above
  // inline operator== to compare them, since that isn't 8 bit clean.
  int l = s1.m_chars.size();
  if (l != s2.m_chars.size ()) {
    return false;
  } else {
    const char *s1p = &*s1.m_chars;
    const char *s2p = &*s2.m_chars;
    for (; l > 0; l--, s1p++, s2p++) {
      if (*s1p != *s2p) return false;
    }
    return true;
  }
}

bool operator< (const String &s1, const String &s2) {
  int i = 0;
  for (;;) {
    if (i >= s1.size() && i >= s2.size()) return false; // not ("" < "")
    if (i >= s2.size()) return true;                    // "" < "a"
    if (i >= s1.size()) return false;                   // not ("a" < "")
    if (s1 [i] < s2 [i]) return true;
    if (s1 [i] > s2 [i]) return false;
    i++;
  }
}

char &String::operator[] (int i) {
  assert (i < size());
  return m_chars [i];
}

const char &String::operator[] (int i) const {
  assert (i < size());
  return m_chars [i];
}

bool String::operator!= (const String&s) const {
  return ! ((*this) == s);
}

String & String::operator= (const String &s) {
  s.check();
  // Don't use copyFromCharStar here, since it isn't 8-bit clean.
  m_chars = s.m_chars;
  return *this;
}

String String::upcase () const {
  String result = *this;
  for (int i = 0; i < result.size(); i++) {
    result [i] = toupper (result[i]);
  }
  return result;
}

ostream &operator<< (ostream &o, const String &s) {
  o << &(s.m_chars[0]);
  return o;
}

// Not 8-bit clean.  :-(.
istream &operator>> (istream &i, String &s) {
  static const int extension = 100;
  Dynavec <char> &vec = s.m_chars;
  vec.extendTo (0);
  int oldSize = 0;
  for (;;) {
    vec.extendTo (oldSize + extension);
    char *start = &*vec+oldSize;
    i.get (start, extension);
    oldSize = oldSize + strlen (start);
    if ('\0' == *start) break;
  }
  vec.extendTo (oldSize + 1);
  vec[oldSize] = '\0';
  if (!i.eof()) {
    char ch;
    i.get (ch);
    assert ('\n' == ch);
  }
  s.check ();
  return i;
}

const String &String::operator += (const String &s) {
  // Get rid of trailing null.
  m_chars.pop ();
  for (int i = 0; i < s.size(); i++) {
    m_chars.push (s[i]);
  }
  m_chars.push (0);
  check ();
  return *this;
}

const String &String::operator += (char c) {
  // Get rid of trailing null.
  m_chars.pop ();
  m_chars.push (c);
  m_chars.push (0);
  check ();
  return *this;
}

const char &String::operator* () const {
  return *m_chars;
}

char &String::operator* () {
  return *m_chars;
}

String String::substring (int from, int chars) const {
  check ();
  String result;
  result.m_chars.extendTo (0);
  int end = FloatUtil::min (from+chars, m_chars.size()-1);
  for (int i = from; i < end; i++) {
    result.m_chars.push (m_chars[i]);
  }
  result.m_chars.push (0);
  result.check ();
  return result;
}

bool String::matches (int position, const char *match) {
  const int s = m_chars.size ();
  for (int i = position; (i < s) & ('\0' != *match); i++, match++) {
    if (*match != m_chars[i]) {
      return false;
    }
  }
  if (*match) {
    return false;
  } else {
    return true;
  }
}

namespace {
  // Size of a fixed size buffer to copy things into when parsing.
  // The choices are:
  // 1. Dynamically allocate a variable-sized buffer.  Slow.
  // 2. Statically allocate a sometimes-varying sized buffer.  Non-reentrant.
  // 3. Dynamically allocate a fixed-size buffer on the stack.  This is what I
  // did.  Numbers can't really be more than 100 bytes, right?
#ifdef NDEBUG
  const int bufsize = 100;
#else
  // Pick a small enough value to test both big and small buffers.
  const int bufsize = 6;
#endif
}

Float String::toFloat (const int start, String *const error) const {
  return toFloat (start, size () - start, error);
}

namespace {
  // Copy the substring beginning with "start" and ending with "end" from
  // "from" into either buffer1 (if it's big) or buffer2 (if it fits), and
  // return a pointer to wherever we put it.
  const char *copyIntoBuf (const String &from,
			   String &buffer1,
			   // buffer2 must be char [bufsize], but declaring
			   // char buffer2 [bufsize] does pass-by-value, and
			   // declaring char &buffer2 [bufsize] gets an error
			   // because it's declaring an array of references.
			   char *buffer2,  
			   const int start,
			   const int end) {
    const char *result;
    if (end - start > bufsize - 1) {
      buffer1 = from.substring (start, end - start);
      result = &*buffer1;
    } else {
      result = buffer2;
      strncpy (buffer2, &from [start], end - start);
      buffer2 [end - start] = '\0';
    }
    return result;
  }
}

Float String::toFloat (const int start, const int chars, String *const error)
  const
{
  // Result can't be Float, since sscanf doesn't know whether Float is float or
  // double.
  double result = 0;
  if (start >= size ()) {
    const String e = String ("Failed to parse a Float starting at position ")
      + start + " past the end of this string: "+ *this;
    if (error) {
      *error = e;
    } else {
      die (e);
    }
    return 0;
  }
  int end = FloatUtil::min (size (), start + chars);
  if (end <= start) {
    const String e = String ("Failed to parse a Float from ")+(end-start)+
      " characters from the string: " + *this;
    if (error) {
      *error = e;
    } else {
      die (e);
    }
    return 0;
  }
  String buffer1;
  char buffer2 [bufsize];
  const char *buffer = copyIntoBuf (*this, buffer1, buffer2, start, end);
  if (0 == sscanf (buffer, "%lf", &result)) {
    const String e = String ("Could not parse a Float starting at position ")
      + start + "\nout of this string: " + *this;
    if (error) {
      *error = e;
    } else {
      die (e);
    }
    // Compiler doesn't know that die doesn't return.
    return 0;
  } else {
    return (Float) result;
  }
}

int String::toInt (const int start, String *const error) const {
  return toInt (start, size() - start, error);
}

int String::toInt (const int start, const int chars,
		   String *const error) const {
  int result = 0;
  if (start >= size ()) {
    const String e = String ("Failed to parse a Float starting at position ")
      + start + " past the end of this string: "+ *this;
    if (error) {
      *error = e;
    } else {
      die (e);
    }
    return 0;
  }
  int end = FloatUtil::min (size (), start + chars);
  if (end <= start) {
    const String e = String ("Failed to parse a Float from ")+(end-start)+
      " characters from the string: " + *this;
    if (error) {
      *error = e;
    } else {
      die (e);
    }
    return 0;
  }
  String buffer1;
  char buffer2 [bufsize];
  const char *buffer = copyIntoBuf (*this, buffer1, buffer2, start, end);
  // FIXME String("8x88").toInt (0, 4, 0) returns 8, instead of aborting.
  // Maybe the right solution is to give up on using sscanf and just parse it
  // myself? 
  if (0 == sscanf (buffer, "%d", &result)) {
    const String e =
      String("Could not parse an integer starting at position ") + start
      + "\nout of this string: " + *this;
    if (error) {
      *error = e;
    } else {
      die (e);
    }
    return 0;
  } else {
    return result;
  }
}

void String::fromFloat (const Float val, const int start, const int chars,
			const int places,
			String *const error = 0) {
  char fmt [60];
  // There's a useful function snprintf that prints into a limited-length
  // buffer in the Linux man entry for sprintf, but it isn't standard and it
  // behaves differently with different versions of glibc, so don't use it.
  sprintf (fmt, "%%%d.%df", chars, places);
  // I know this isn't bulletproof, but I can't imagine any scenarios where it
  // would actully happen, nor can I imagine the usual scenario where a cracker
  // tries to take us over by overflowing our buffers, since we don't have any
  // security against misbehaving plugins anyway.
  if (strlen (fmt) > sizeof (fmt)) {
    die ("Memory corrupted because the format overflowed the buffer in "
	 "fromFloat");
  }
  char result [100];
  if ((unsigned int) chars > sizeof (result)) {
    const String e =
      String ("Field width ")+chars+" too big in fromFloat";
    if (error) {
      *error = e;
    } else {
      die (e);
    }
  }
  sprintf (result, fmt, val);
  if (strlen (result) >= sizeof (result)) {
    die ("Memory corrupted by overflowing result buffer in fromFloat");
  }
  if (start+chars > size()) {
    // extendTo can shorten a string.  We don't want that here.
    m_chars.pop (); // Pop a trailing null.
    m_chars.extendTo (start + chars, ' ');
    m_chars.push ('\0');
  }
  bool foundDot = false;
  for (int i = 0; i < chars && result [i]; i++) {
    const char toCopy = result [i];
    if ('.' == toCopy) {
      foundDot = true;
    }
    m_chars [start + i] = toCopy;
  }
  if (!foundDot) {
    const String e =
      String ("The decimal point didn't fit in the field when printing ")
	 +val+" to "+chars+" places";
    if (error) {
      *error = e;
    } else {
      die (e);
    }
  }
}

void String::fromInt (const int val, const int start, const int chars,
		      String *const error = 0) {
  char fmt [60];
  sprintf (fmt, "%%%dd", chars);
  if (strlen (fmt) > sizeof (fmt)) {
    die ("Memory corrupted because the format overflowed the buffer in "
	 "fromFloat");
  }
  char result [100];
  if ((unsigned int) chars >= sizeof (result)) {
    const String e = String ("Field width ")+chars+" too big in fromInt";
    if (error) {
      *error = e;
    } else {
      die (e);
    }
  }
  sprintf (result, fmt, val);
  if (strlen (result) >= sizeof (result)) {
    die ("Memory corrupted by overflowing result buffer in fromInt");
  }
  if (start+chars > size()) {
    m_chars.pop (); // Pop a trailing null
    m_chars.extendTo (start+chars, ' ');
    m_chars.push ('\0');
  }
  if (strlen (result) > (unsigned int) chars) {
    const String e = String ("The integer ") + val +
      " overflowed the field width " + chars + " in fromInt";
    if (error) {
      *error = e;
    } else {
      die (e);
    }
  }
  int i;
  for (i = 0; i < chars && result [i]; i++) {
    m_chars [start+i] = result [i];
  }
}

void String::fromString (const String &val,
			 const int start,
			 const int chars,
			 String *const error = 0)
{
  if (val.size () > chars) {
    const String e =
      String ("Can't fit the string \""+val+"\" into the field width ")+chars;
    if (error) {
      *error = e;
    } else {
      die (e);
    }
    return;
  }
  if (start + chars > size()) {
    m_chars.pop();
    m_chars.extendTo (start+chars, ' ');
    m_chars.push ('\0');
  }
  for (int i = 0; i < val.size(); i++) {
    m_chars [start+i] = val [i];
  }
  for (int i = val.size (); i < chars; i++) {
    m_chars [start+i] = ' ';
  }
}

void String::fromStringRight (const String &val, int start, int chars,
			      String *error = 0)
{
  if (val.size () > chars) {
    const String e =
      String ("Can't fit the string \""+val+"\" into the field width ")+chars;
    if (error) {
      *error = e;
    } else {
      die (e);
    }
    return;
  }
  if (start + chars > size()) {
    m_chars.pop ();
    m_chars.extendTo (start+chars, ' ');
    m_chars.push ('\0');
  }
  const int startCopy = chars - val.size();
  for (int i = 0; i < startCopy; i++) {
    m_chars [start+i] = ' ';
  }
  for (int i = startCopy; i < chars; i++) {
    m_chars [start+i] = val [i - startCopy];
  }
}

