// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "StreamUtil.h"

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif
 
#ifndef __String_h__
#include "String.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

void StreamUtil::endLine () {
  // Throw away the rest of the current line on the given stream.
  if (cin.eof ()) {
    cin.clear();
  } else {
    for (;;) {
      cin.clear ();
      int ch = cin.get();
      if (EOF == ch) break;
      if ('\n' == ch) break;
    }
  }
}

void StreamUtil::promptRead (const char *prompt, int &result) {
  for (;;) {
    cout << prompt;
    cin.clear();
    cin >> result;
    if (cin.eof ()) {
      cout << "Got an EOF.  Try again." << endl;
      cin.clear ();
    } else if (cin.fail ()) {
      cout << "Failed to parse a number.  Try again." << endl;
      endLine ();
    } else {
      endLine ();
      break;
    }
  }
}

void StreamUtil::promptRead (const char *prompt, String &result) {
  result = "";
  cout << prompt;
  cin.clear ();
  char ch;
  for (;;) {
    cin.get (ch);
    if (cin.eof ()) {
      cout << "Got an EOF.  Try again." << endl;
      cin.clear ();
      result = "";
      cout << prompt;
    } else if ('\n' == ch) {
      break;
    } else {
      result += ch;
    }
  }
}

void StreamUtil::promptRead (const char *prompt, int &result,
			     int min, int max)
{
  for (;;) {
    StreamUtil::promptRead (prompt, result);
    if (result < min) {
      cout << "Please enter a number greater than or equal to "
	   << min<<".";
    } else if (result >= max) {
      cout << "Please enter a number less than "<<max<<"."<<endl;
    } else {
      break;
    }
  }
}

void StreamUtil::promptRead (const char *prompt, Float &result) {
  for (;;) {
    cout << prompt;
    cin.clear();
    cin >> result;
    if (cin.eof ()) {
      cout << "Got an EOF.  Try again." << endl;
      cin.clear ();
    } else if (cin.fail ()) {
      cout << "Failed to parse a number.  Try again." << endl;
      endLine ();
    } else {
      endLine ();
      break;
    }
  }
}

void StreamUtil::promptRead (const char *prompt, Vec3 &result) {
  if (prompt [0] != '\0') {
    cout << prompt << endl;
  }
  promptRead ("What is the x coordinate?", result[0]);
  promptRead ("What is the y coordinate?", result[1]);
  promptRead ("What is the z coordinate?", result[2]);
}

void StreamUtil::promptReadNew (Vec3 &result) {
  promptRead ("", result);
}

void StreamUtil::promptRead (const char *prompt, Quaternion &result) {
  if (prompt[0] != '\0') {
    cout << prompt << endl;
  }
  Float angle;
  promptRead ("What angle of rotation do you want, in degrees? ", angle);
  Vec3 axis;
  promptRead ("Please give the coordinates for the axis of rotation.", axis);
  result = Quaternion::makeRotation (angle * FloatUtil::PI / 180, axis);
}

void StreamUtil::promptReadNew (Quaternion &result) {
  promptRead ("", result);
}

// For tolower.
#ifndef __ctype_h__
#include <ctype.h>
#define __ctype_h__
#endif

char StreamUtil::readMenuCh () {
  int ch;
  for (;;) {
    cin.clear();
    ch = cin.get ();
    if (cin.eof()) {
      cout << "Got an EOF.  Try again." << endl;
    } else if (' ' == ch || '\t' == ch) {
      // Do nothing.
    } else {
      break;
    }
  }
  endLine ();
  return ch;
}

void StreamUtil::promptRead (const char *prompt, bool &result) {
  for (;;) {
    cout << prompt;
    char ch = tolower (readMenuCh ());
    if ('y' == ch || 't' == ch) {
      result = true;
      break;
    } else if ('n' == ch || 'f' == ch) {
      result = false;
      break;
    } else {
      cout << "Please answer yes or no or true or false.  Try again." << endl;
    }
  }
}

#ifndef __String_h__
#include "String.h"
#endif

// Print an array of lines, where each line is an array of strings,
// but line up the columns.
void StreamUtil::printTableau (ostream &o, Dynavec <Dynavec <String> > &t) {
  Dynavec <int> widths;
  for (int i = 0; i < t.size(); i++) {
    for (int j = 0; j < t[i].size(); j++) {
      if (widths.size() <= j) {
	widths.push (0);
      }
      assert (widths.size() > j);
      if (t[i][j].size() > widths[j]) {
	widths[j] = t[i][j].size();
      }
    }
  }
  for (int i = 0; i < t.size(); i++) {
    for (int j = 0; j < t[i].size(); j++) {
      o << t[i][j];
      if (j < t[i].size() - 1) {
	for (int k = 0; k < (widths [j] - t[i][j].size()); k++) {
	  o.put (' ');
	}
	o << "  ";
      }
    }
    o << endl;
  }
}

