// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __SelectFrustrum_h__
#include "SelectFrustrum.h"
#endif

#ifndef __float_h__
#include <float.h>
#define __float_h__
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __SimpleGrabbingActionConfiguration_h__
#include "SimpleGrabbingActionConfiguration.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __KeyboardMouseEvent_h__
#include "KeyboardMouseEvent.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __ObjectLink_h__
#include "ObjectLink.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

namespace {
  // Use a mouse drag to select a region of space.  Objects withing bounding
  // spheres that intersect the region are added to the selection, except if
  // they are all selected at the beginning then they are removed from the
  // selection.  You may want to macro this with a DeselectAll running before.
  // The left, right, top, and bottom sides of the cube are determined by the
  // rectangle dragged out by the mouse.  If doCube is true then the front is
  // the nearest object in the frustrum and the distance between the front and
  // the back is the greater of the width and the height.  If doCube is false,
  // then there is no front or back and the region of space is infinite in
  // those directions.
  class SelectFrustrumConfiguration
    : public SimpleGrabbingActionConfiguration
  {
  public:
    // The starting x and y.
    int m_x;
    int m_y;
    // Default and default copy constructors are fine.
    bool hasTheSameTypeAs (const Configuration *c) const {
      return 0 != dynamic_cast <const SelectFrustrumConfiguration *> (c);
    }
    SP<Configuration> copy () const {
      return NEW (SelectFrustrumConfiguration (*this));
    }
  };
}
SelectFrustrum::SelectFrustrum (bool doCube)
  : Factory (doCube?"SelectCube":"SelectFrustrum"), m_doCube (doCube)
{}

SP<Configuration> SelectFrustrum::defaultConfiguration () const {
  return NEW (SelectFrustrumConfiguration ());
}

SP<Configurable> SelectFrustrum::makeIt (SP<Configuration> c) const {
  const SP<SelectFrustrumConfiguration> sfc =
    dynamic_cast <SelectFrustrumConfiguration *> (&*c);
  assert (sfc);
  SP<TopLevel> top = sfc->getTopLevel ();
  SP<const KeyboardMouseEvent> e =
    dynamic_cast <const KeyboardMouseEvent *> (&*top->getEvent ());
  assert (e);
  if (sfc->isFirst ()) {
    // grabIfFirst makes a copy, so we need to set m_x and m_y before
    // calling grabIfFirst.
    sfc->m_x = e->getX();
    sfc->m_y = e->getY();
    sfc->grabIfFirst (this);
  } else if (sfc->isLast ()) {
    Float minX = FloatUtil::min (sfc->m_x, e->getX());
    Float maxX = FloatUtil::max (sfc->m_x, e->getX());
    Float minY = FloatUtil::min (sfc->m_y, e->getY());
    Float maxY = FloatUtil::max (sfc->m_y, e->getY());
    SP<SceneGraph> sg = top->getSceneGraph ();
    const int maxIndex = sg->maxIndex();
    SP<Canvas> canvas = top->getCanvas ();
    Dynavec <bool> objectsInBox;
    objectsInBox.extendTo (maxIndex, false);
    Float maxZ = -FLT_MAX;
    BoundingBox frustrum (Vec3 (minX, minY, -FLT_MAX),
			  // Maximum Z is 0 because we have a right handed
			  // coordinate system with Z increasing toward the
			  // eye, and the near clipping plane is at 0.
			  Vec3 (maxX, maxY, 0));
    for (int i = 0; i < maxIndex; i++) {
      ObjectLink ol (sg, i);
      if (ol.isSelectable ()) {
	BoundingSphere bs =
	  canvas->transformWorldToRHScreen (ol.selectionBoundingSphere());
	if (frustrum.intersects (bs)) {
	  if (m_doCube) {
	    maxZ = FloatUtil::max (maxZ,
				   // Use pythagoras to find the nearest point
				   // in the frustrum that's also on the
				   // bounding sphere.
				   bs.center()[2] +
				   FloatUtil::sqrt
				   // Call max to avoid negative square root on
				   // round off error.
				   (FloatUtil::max
				    ((Float)0.0, 
				     FloatUtil::sqr (bs.radius())
				     - 
				     (frustrum.nearest(bs.center())
				      - bs.center()).lengthSquared())));
	  } else {
	    objectsInBox [i] = true;
	  }
	}
      }
    }
    if (m_doCube && -FLT_MAX != maxZ) {
      const Float size = FloatUtil::max ((Float)1.0,
					 FloatUtil::max (maxX - minX,
							 maxY - minY));
      const Float minZ = maxZ - size;
      frustrum =
	BoundingBox (Vec3 (minX, minY, minZ), Vec3 (maxX, maxY, maxZ));
      for (int i = 0; i < maxIndex; i++) {
	ObjectLink ol (sg, i);
	if (ol.isSelectable ()) {
	  BoundingSphere bs =
	    canvas->transformWorldToRHScreen (ol.selectionBoundingSphere());
	  if (frustrum.intersects (bs)) {
	    objectsInBox [i] = true;
	  }
	}
      }
    }
    Dynavec <int> selection;
    SelectionManager::getSelection (selection);
    Dynavec <bool> selectionSet;
    selectionSet.extendTo (maxIndex, false);
    for (int i = 0; i < selection.size(); i++) {
      selectionSet[selection[i]] = true;
    }
    // Now selectionSet[i] is true if object i is selected.
    bool allObjectsInBoxAreSelected = true;
    for (int i = 0; i < maxIndex; i++) {
      if (objectsInBox [i] && ! selectionSet [i]) {
	allObjectsInBoxAreSelected = false;
	break;
      }
    }
    // Now the value of allObjectsInBoxAreSelected fits its name.
    // If the objects in the box are all already selected, then deselect them.
    // Otherwise select them.
    for (int i = 0; i < maxIndex; i++) {
      if (objectsInBox [i]) {
	selectionSet [i] = !allObjectsInBoxAreSelected;
      }
    }
    selection.extendTo (0);
    for (int i = 0; i < selectionSet.size(); i++) {
      if (selectionSet [i]) selection.push (i);
    }
    SelectionManager::setSelection (sg, selection);
  }
  return NEW (Action ());
}

namespace {
  static const bool useless =
  (FactoryTable::store ("KeyboardMouseAction", NEW (SelectFrustrum (false))),
   true);
}
