// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __ActionConfiguration_h__
#include "ActionConfiguration.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __ObjectLink_h__
#include "ObjectLink.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

namespace {
  class GrowSelectionNearby
    : public Factory
  {
  public:
    GrowSelectionNearby ()
      : Factory ("GrowSelectionNearby")
    {}
    SP<Configuration> defaultConfiguration () const {
      return NEW (ActionConfiguration ());
    }
    SP<Configurable> makeIt (const SP<Configuration> c) const {
      SP<ActionConfiguration> conf =
	dynamic_cast <ActionConfiguration *> (&*c);
      assert (conf);
      SP<TopLevel> top = conf->getTopLevel ();
      Dynavec <int> selection;
      SelectionManager::getSelection (selection);
      // FIXME We could do faster by using bins.
      // See DESIGN.NEAREST.  Or, just use any of the Storage classes and have
      // an a-priori maximum distance of a few angstroms.  But release the slow
      // implementation for now.
      SP<SceneGraph> sg = top->getSceneGraph ();
      // The set of objects selected.
      // somethingIn will eventually hold the index of an object that is
      // selected.
      ObjectLink somethingIn;
      Dynavec <bool> selectionSet;
      selectionSet.extendTo (sg->maxIndex(), false);
      for (int i = 0; i < selection.size(); i++) {
	const int s = selection [i];
	assert (-1 != s);
	selectionSet [s] = true;
	if (!somethingIn.isValid ()) {
	  somethingIn = ObjectLink (sg, s);
	  assert (somethingIn.isValid ());
	}
      }
      SP<Action> result = NEW (Action ());
      if (!somethingIn.isValid ()) {
	result->setProblem ("You must select something before growing "
			    "the selection.");
      }
      // somethingOut will eventually hold the index of an object that is not
      // selected.
      ObjectLink somethingOut;
      // How far the nearest thing in the scene graph is from something in the
      // set.
      // Compiler thinks nearDistance may be used uninitialized unless I
      // initialize it here.  It's wrong.  No big deal.
      Float nearestDistance = 0;
      if (!result->isProblem ()) {
	for (int i = 0; i < sg->maxIndex(); i++) {
	  if (!selectionSet [i] && sg->object (i)->isSelectable()) {
	    ObjectLink po (sg, i);
	    if (po.isValid () && po.isVisible ()) {
	      // Now we know po points to something not selected.
	      if (!somethingOut.isValid ()) {
		somethingOut = po;
		assert (somethingOut.isValid ());
		// Initialize nearestDistance to any distance between something
		// in the set and something out of the set.
		nearestDistance =
		  (somethingOut.boundingSphereCenter () -
		   somethingIn.boundingSphereCenter()).lengthSquared ();
	      }
	      for (int j = 0; j < selection.size(); j++) {
		ObjectLink thisSel (sg, selection[j]);
		if (thisSel.isVisible ()) {
		  nearestDistance =
		    FloatUtil::min (nearestDistance,
				    (po.boundingSphereCenter() -
				     thisSel.boundingSphereCenter())
				    .lengthSquared());
		}
	      }
	    }
	  }
	}
	if (!somethingOut.isValid ()) {
	  result->setProblem ("The entire scene is selected, so the selection "
			      "cannot grow any more.");
	}
      }
      if (!result->isProblem ()) {
	// Make sure nearestDistance increases more than roundoff error each
	// time.
	// FIXME The 1.01 should be a configuration constant.
	nearestDistance *= 1.01;
	message (String ("Adding everything closer than ") +
		 FloatUtil::sqrt (nearestDistance));
	for (;;) {
	  bool grown = false;
	  for (int i = 0; i < sg->maxIndex(); i++) {
	    if (!selectionSet [i]) {
	      ObjectLink po (sg, i);
	      if (po.isValid () && SelectionManager::isSelectable (po)) {
		for (int j = 0; j < selection.size (); j++) {
		  ObjectLink thisSel (sg, selection [j]);
		  if ((po.boundingSphereCenter()-
		       thisSel.boundingSphereCenter()).lengthSquared()
		      <= nearestDistance) {
		    selection.push (i);
		    selectionSet [i] = true;
		    grown = true;
		    break;
		  }
		}
	      }
	    }
	  }
	  if (!grown) break;
	}
	SelectionManager::setSelection (sg, selection);
	result->setMessage (String (selection.size())+ " objects selected");
      }
      return &*result;
    }
  };
  const bool useless =
  (FactoryTable::store ("Action", NEW (GrowSelectionNearby ())),
   true);
}
