// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __PhysicsObject_h__
#include "PhysicsObject.h"
#endif

#ifndef __SubsetStorage_h__
#include "SubsetStorage.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

// Memset, memcpy
#ifndef __string_h__
#include <string.h>
#define __string_h__
#endif

// cout
#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __BoundingSphere_h__
#include "BoundingSphere.h"
#endif

#ifndef __SceneGraphConfiguration_h__
#include "SceneGraphConfiguration.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __BadSlotValue_h__
#include "BadSlotValue.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __SubsetStorageConfiguration_h__
#include "SubsetStorageConfiguration.h"
#endif

#ifndef __Drawer_h__
#include "Drawer.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __PhysicsObjectInfo_h__
#include "PhysicsObjectInfo.h"
#endif

#ifndef __Evaluator_h__
#include "Evaluator.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

namespace {

  struct ObjectInfo {
    // The PhysicsObject for this object.
    SP<const PhysicsObject> m_object;
    // The position of this object in state vectors.
    int m_stateIndex;
    // The neighbors to pass to doPhysics.
    Dynavec <PhysicsObjectInfo> m_neighbors;
    // The links to pass to doPhysics.
    Dynavec <PhysicsObjectInfo> m_links;
    // The bounding box that store claims it actually stored it into.
    // Use this when we notice that an unbounded object has moved.
    BoundingBox m_approximateBox;
    // How many unbounded neighbors we have.  The unbounded ones go
    // first in the m_neighbors list, and the rest go later and are
    // maintained by the SubsetStorageScene playing its role as a
    // SubsetStorage::Maintainer.
    int m_unboundedNeighbors;

    ObjectInfo ()
      : m_stateIndex (-1),
	m_unboundedNeighbors (0)
    {}
    PhysicsObjectInfo poi () const {
      return PhysicsObjectInfo (m_object, m_stateIndex);
    }
  };

  struct UnboundedObjectInfo {
    // The index of the object in the main object list.
    int m_index;
    // The state of the object last time we computed neighbors for it.
    // If the state changes, we will need to compute neighbors for it
    // again.  Unbounded objects had better not change state
    // frequently, since this will be slow.
    Dynavec <Float> m_state;
  };

  class SubsetSceneConfiguration : public SceneGraphConfiguration {
    static String storageSlotName () {
      return "Storage";
    };
    static String drawerSlotName () {
      return "Drawer";
    }
  public:
    SubsetSceneConfiguration ()
    {
      initializeSlot (storageSlotName (),
		      NEW (RecursiveSlotValue ("SubsetStorage",
					       "SubsetStorage")));
      // Now that we aren't doing OGL any more, there is no benefit to
      // NeighborDrawer.  SimpleDrawer is 20 ms/frame faster for the
      // diffGear.pdb example.
      initializeSlot (drawerSlotName (),
		      NEW (RecursiveSlotValue ("Drawer",
					       "XDepthClippingDrawer"
					       )));
    }
    void setStorageFactory (Factory *f) {
      RecursiveSlotValue::setSlotValue
	(this, storageSlotName (),
	 NEW (RecursiveSlotValue ("SubsetStorage", SP<Factory> (f))));
    }
    const Factory *getStorageFactory () const {
      return RecursiveSlotValue::getSlotValue (this, storageSlotName ())
	->getFactory ();
    }
    SP<const SubsetStorageConfiguration> getStorageConfiguration ()
      const
    {
      SP<const SubsetStorageConfiguration> result =
	dynamic_cast <const SubsetStorageConfiguration*> 
	(RecursiveSlotValue::getSlotValue (this, storageSlotName ())->
	 getConfiguration ());
      assert (result);
      return result;
    }
    SP<SubsetStorageConfiguration> getStorageConfiguration () {
      SP<SubsetStorageConfiguration> result =
	dynamic_cast <SubsetStorageConfiguration*> 
	(RecursiveSlotValue::getSlotValue (SP<Configuration> (this),
					   storageSlotName ())->
	 getConfiguration ());
      assert (result);
      return result;
    }
    SP<Drawer> getDrawer () const {
      SP<Configurable> preDrawer =
	RecursiveSlotValue::getSlotValue (this, drawerSlotName ())
	-> makeIt ();
      Drawer *result = dynamic_cast <Drawer *> (&*preDrawer);
      assert (result);
      return result;
    }
    SubsetSceneConfiguration (const SubsetSceneConfiguration &s)
      : SceneGraphConfiguration (s)
    {}
    SP<Configuration> copy () const {
      return NEW (SubsetSceneConfiguration (*this));
    }
    bool hasTheSameTypeAs (const Configuration *c) const {
      return 0 != dynamic_cast <const SubsetSceneConfiguration *> (c);
    }
  };

  class SubsetStorageScene : public SceneGraph,
			     public Evaluator,
			     public SubsetStorage::Maintainer
  {
    const SP<SubsetStorage> m_storage;
    // The last observed value of the link manager's generation.  Use this to
    // decide when to recompute the cached links in m_info.
    int m_linkManagerGeneration;
    // The states of the objects in the scene.
    Dynavec<Float> m_states;
    // What we know about the neighbors of each object.
    Dynavec <ObjectInfo> m_info;
    // Information about the unbounded objects, so we can decide whether
    // to add them to a neighbor list when a bounded object moves.
    Dynavec <UnboundedObjectInfo> m_unbounded;
    // Next one is logically part of newBoundingBox, but I didn't want
    // to have an allocation in hot code.
    mutable Dynavec <PhysicsObjectInfo> m_newNeighbors;
    SP<Drawer> m_drawer;
    // Whether we have deleted any objects since last time validate() was
    // called.  Adding objects does not cause things to be invalid, since it
    // does not renumber any objects.  This will change if I ever decide to
    // sort the scene graph.
    bool m_valid;
    // Indices of the animations, that is, the objects for which
    // hasAnimationEver returns true.
    Dynavec <int> m_animations;
    
    // Scan the scene graph and update m_animations 
    void updateAnimations ();
    // Consistency check.
    inline void check ();
    void checkStorageAgainstState (const Float *state);
    bool updateNeighbors (const Float *state);
    // Squeeze out any null objects.
    void garbageCollectObj ();
    // Update the unbounded neighbors of all objects, restricting
    // attention to the ones listed in movedBounded.
    void updateUnboundedNeighbors (Dynavec <PhysicsObjectInfo> &movedBounded,
				   const Float *state);
    // Return a pointer to the beginning of the state information for the given
    // object.  
    inline Float *statePointer (const ObjectInfo &oi) {
      // This would simply be &(m_states [oi.m_stateIndex]), except that the
      // last object in the SceneGraph can have a stateSize of 0.  In that case
      // &(m_states [oi.m_stateIndex]) can get a bogus rangecheck error.
      assert (0 == oi.m_object->stateSize() ||
	      (0 <= oi.m_stateIndex && oi.m_stateIndex < m_states.size ()));
      return &*m_states + oi.m_stateIndex;
    }
    inline const Float *statePointer (const ObjectInfo &oi) const {
      // This would simply be &(m_states [oi.m_stateIndex]), except that the
      // last object in the SceneGraph can have a stateSize of 0.  In that case
      // &(m_states [oi.m_stateIndex]) can get a bogus rangecheck error.
      assert (0 == oi.m_object->stateSize() ||
	      (0 <= oi.m_stateIndex && oi.m_stateIndex < m_states.size ()));
      return &*m_states + oi.m_stateIndex;
    }
  public:
    SP<Evaluator> getEvaluator () {
      return this;
    }
    SP<const Evaluator> getEvaluator () const {
      return this;
    }
    inline int maxIndex () const {
      return m_info.size ();
    }
    ~SubsetStorageScene () {
      // If the code below actually did any smart pointer manipulation on the
      // SceneGraph, it might cause a recursive call to the destructor if we
      // didn't have the ref() and deref() calls below.  I do not believe any
      // such smart pointer manipulation happens, but that looks like a horrid
      // bug to track down if it ever creeps in, so I'll just prevent it now.
      ref();
      if (getLinkManager ()) {
	// We still own the LinkManager, so when this destructor finishes, the
	// next thing to happen will be destruction of the LinkManager.  Be
	// sure to make all the ObjectLinks invalid now, so nobody tries to
	// follow them as the objects in the SceneGraph are being destructed.
	// This code can't go in the SceneGraph base class because it needs to
	// be done before the objects in the scene graph are themselves
	// destructed, and that happens immediately after this code is run (as
	// part of destructing SubsetStorageScene's instance variables) but
	// before the destructor for the SceneGraph base class is called.
	Dynavec <int> oldToNew;
	oldToNew.extendTo (maxIndex(), -1);
	// Used to have a "0" instead of "this" in the next line.  That was
	// wrong; objects that destruct themselves while destructing the scene
	// graph still need to be able to deallocate their ObjectLinks before
	// the LinkManager is gone, and they need a valid SceneGraph pointer in
	// the object links if they are going to do this.
	getLinkManager()->updateAfterGC (oldToNew, this);
      }
      deref ();
    }
    SubsetStorageScene (SP<SubsetSceneConfiguration> conf);
    // Methods on SceneGraph
    int objectStateIndex (int index) const;
    const PhysicsObject *object (int index) const;
    void objectsOnLine (const Vec3 &v1, const Vec3 &v2,
			Dynavec <int> &indices,
			Dynavec <Vec3> &intersectionPoints)
      const;
    BoundingSphere graphicsBoundingSphere () const;
    int addObject (const PhysicsObject *obj,
		   const Float *initialState);
    void deleteObject (int index);
    void validate ();
    bool draw (SP<Canvas> vo);
    SP<Evaluator::OutOfBounds> eval (Float t, const Float *state,
				     Float *deriv);
    Dynavec <Float> &stateVec ();
    const Dynavec <Float> &stateVec () const;
    void animateTimeStep ();
    void animateFrame (Canvas *canvas);
    bool anyAnimations () const;

    // Methods on Maintainer
    bool addToSubset (int object, int toAdd);
    void deleteFromSubset (int object, int position);
    void newBoundingBox (int object, const BoundingBox &newBox);
    void checkNeighbors (int object, const Dynavec <int> &neighborList) const;
  };
  
  inline int SubsetStorageScene::objectStateIndex (int index) const {
    return m_info[index].m_stateIndex;
  }
  
  inline const PhysicsObject *SubsetStorageScene::object (int index) const {
    return m_info[index].m_object;
  }
  
  inline void SubsetStorageScene::check () {
#ifdef VERY_SLOW_DEBUG
    // All objects exist.
    for (int i = 0; i < maxIndex(); i++) {
      assert (m_info[i].m_object);
    }
    // Exempt objects have no neighbors.
    for (int i = 0; i < maxIndex(); i++) {
      if (m_info[i].m_object && m_info[i].m_object->exempt()) {
	assert (0 == m_info[i].m_neighbors.size());
      }
    }
    // m_unbounded includes all of the unbounded objects.
    for (int i = 0; i < maxIndex(); i++) {
      if (m_info[i].m_object) {
	if (!m_info[i].m_object->isPhysicsBounded ()) {
	  int index = -1;
	  for (int k = 0; k < m_unbounded.size(); k++) {
	    if (m_unbounded[k].m_index == i) {
	      assert (-1 == index);
	      index = k;
	    }
	  }
	  assert (-1 != index);
	}
      }
    }
    // All objects on m_unbounded are unbounded and not deleted.
    for (int i = 0; i < m_unbounded.size(); i++) {
      assert (m_info[m_unbounded[i].m_index].m_object);
      assert (!(m_info[m_unbounded[i].m_index].m_object->isPhysicsBounded()));
    }
    // Check that the first m_unboundedNeighbors objects in each
    // neighbor vector are actually unbounded.
    for (int i = 0; i < maxIndex(); i++) {
      if (m_info[i].m_object) {
	assert (m_info[i].m_neighbors.size() >=
		m_info[i].m_unboundedNeighbors);
	for (int j = 0; j < m_info [i].m_unboundedNeighbors; j++) {
	  assert (!m_info[i].m_neighbors[j].m_object->isPhysicsBounded ());
	}
      }
    }
    // Check that the objects and indices in each neighbor list are
    // consistent with each other.
    for (int i = 0; i < maxIndex(); i++) {
      if (m_info[i].m_object) {
	for (int j = 0; j < m_info[i].m_neighbors.size(); j++) {
	  // Find the object in the state list.
	  int index = -1;
	  for (int k = 0; k < m_info.size(); k++) {
	    if (m_info[k].m_stateIndex == m_info[i].m_neighbors[j].m_index) {
	      // It should be there exactly once.
	      assert (-1 == index);
	      index = k;
	    }
	  }
	  assert (-1 != index);
	  assert (m_info[index].m_object == m_info[i].m_neighbors[j].m_object);
	}
      }
    }
    // The unbounded neighbors of each object should intersect the
    // approximateBox of that object.
    for (int i = 0; i < maxIndex(); i++) {
      for (int j = 0; j < m_info [i].m_unboundedNeighbors; j++) {
	// Find the object in the unbounded list.
	int index = -1;
	for (int k = 0; k < m_unbounded.size(); k++) {
	  if (objectStateIndex (m_unbounded[k].m_index) ==
	      m_info[i].m_neighbors[j].m_index) {
	    // It should be there exactly once.
	    assert (-1 == index);
	    index = k;
	  }
	}
	assert (index != -1);
	assert (m_info[i].m_neighbors[j].m_object
		->physicsIntersect (&(m_unbounded[index].m_state[0]),
				    m_info[i].m_approximateBox));
      }
    }
    assert (!VecUtil::hasDuplicates (m_animations));
    for (int i = 0; i < maxIndex (); i++) {
      if (m_info [i].m_object && m_info[i].m_object->hasAnimationEver ()) {
	assert (VecUtil::find (i, m_animations));
      }
    }
    for (int i = 0; i < m_animations.size(); i++) {
      assert (m_info[m_animations[i]].m_object->hasAnimationEver ());
    }
#endif
  }

  inline void SubsetStorageScene::checkStorageAgainstState (const Float *state)
  {
    (void) state;
#ifdef VERY_SLOW_DEBUG
    // The saved state of unbounded objects is equal to their current state.
    for (int i = 0; i < m_unbounded.size(); i++) {
      // The saved state has the right size.
      assert (m_unbounded[i].m_state.size() ==
	      m_info[m_unbounded[i].m_index].m_object->stateSize());
      for (int j = 0; j < m_unbounded[i].m_state.size(); j++) {
	assert (m_unbounded[i].m_state[j] ==
		m_states[m_info[m_unbounded[i].m_index].m_stateIndex+j]);
      }
    }
    // The neighbor relation is consistent with actual intersections
    // of objects.
    for (int i = 0; i < maxIndex(); i++) {
      if (m_info[i].m_object && ! m_info[i].m_object->exempt()) {
	for (int j = 0; j < maxIndex (); j++) {
	  // If the physics intersects...
	  if (j != i && m_info [j].m_object &&
	      m_info[j].m_object->physicsIntersect
	      (state + m_info[j].m_stateIndex,
	       m_info[i].m_approximateBox))
	    {
	      // Then it had better be a neighbor.
	      int index = -1;
	      for (int k = 0; k < m_info[i].m_neighbors.size(); k++) {
		if (m_info[i].m_neighbors[k].m_index ==
		    m_info[j].m_stateIndex) {
		  assert (-1 == index);
		  index = k;
		}
	      }
	      assert (-1 != index);
	    }
	}
      }
    }
#endif
  }

  void SubsetStorageScene::garbageCollectObj () {
    didGarbageCollection ();
    // Need to know the mapping between old object indices and new object
    // indices so we can tell the LinkManager.  Since we may retry the garbage
    // collection, we have to know the mapping both ways, since we have to tell
    // the LinkManager the mapping between the original objects and the final
    // ones, not just the objects before the last iteration and the final
    // objects.  Hmm, having a Storage that can't store everything definitely
    // adds complexity; maybe I should go back to OctTree's.
    Dynavec <int> oldToNew;
    Dynavec <int> newToOld;
#ifndef NDEBUG
    // The physics object in each old position, so we can verify the
    // mapping at the end.
    Dynavec <SP <const PhysicsObject> > pos;
#endif
    for (int i = 0; i < m_info.size(); i++) {
      oldToNew.push (i);
      newToOld.push (i);
#ifndef NDEBUG
      pos.push (m_info[i].m_object);
#endif
    }
    for (;;) {
      // Loop until we don't have to retry.  We have to retry if we
      // find objects that we cannot store.
      bool retry = false;
      int toState = 0;
      int toIndex = 0;
      for (int fromIndex = 0; fromIndex < m_info.size(); fromIndex++) {
	ObjectInfo &fromOi = m_info[fromIndex];
	if (fromOi.m_object) {
	  ObjectInfo &toOi = m_info[toIndex];
	  toOi.m_object = fromOi.m_object;
	  toOi.m_stateIndex = toState;
	  toOi.m_neighbors.extendTo (0);
	  toOi.m_unboundedNeighbors = 0;
	  const int fromState = fromOi.m_stateIndex;
	  for (int i = 0; i < toOi.m_object->stateSize(); i++) {
	    m_states [toState+i] = m_states [fromState + i];
	  }
	  // The object that was at fromIndex before is now at toIndex.
	  // Need to compute the original index of the object that moved
	  const int origIndex = newToOld [fromIndex];
	  oldToNew [origIndex] = toIndex;
	  newToOld [toIndex] = origIndex;
	  toIndex++;
	  toState += toOi.m_object->stateSize();
	} else {
	  oldToNew [newToOld [fromIndex]] = -1;
	}
      }
      m_info.extendTo (toIndex);
      newToOld.extendTo (toIndex);
      m_states.extendTo (toState);
      m_storage->empty();
      m_unbounded.extendTo (0);
      Dynavec <PhysicsObjectInfo> allUnbounded;
      for (int i = 0; i < m_info.size(); i++) {
	ObjectInfo &oi = m_info[i];
	const PhysicsObject *po = oi.m_object;
	assert (po);
	if (po->isPhysicsBounded()) {
	  bool canStore =
	    m_storage->store
	    (i,
	     po->boundingSphereCenter(statePointer (oi)),
	     po->physicsBoundingSphereRadius ());
	  if (!canStore) {
	    if (!retry) {
	      cerr << "Found out-of-bounds object during garbage collection."
		   << endl
		   << "Restarting garbage collection." << endl;
	    }
	    oi.m_object = 0;
#ifndef NDEBUG
	    pos [newToOld [i]] = 0;
#endif
	    retry = true;
	  }
	} else {
	  int newUnboundedPos = m_unbounded.size();
	  m_unbounded.push ();
	  UnboundedObjectInfo &uoi = m_unbounded[newUnboundedPos];
	  uoi.m_index = i;
	  // Update the saved state vector.
	  uoi.m_state.extendTo (oi.m_object->stateSize());
	  for (int i = 0; i < uoi.m_state.size(); i++) {
	    uoi.m_state[i] = m_states[oi.m_stateIndex+i];
	  }
	  allUnbounded.push (oi.poi());
	}
      }
      updateUnboundedNeighbors (allUnbounded, &*m_states);
      if (!retry) break;
    }
    // Have to updateAnimations() before the check(), otherwise check() fails.
    updateAnimations();
    check ();
    checkStorageAgainstState (&*m_states);
#ifndef NDEBUG
    for (int i = 0; i < pos.size(); i++) {
      // This isn't a complete check, since we're just verifying that the
      // physics objects match and we are not verifying that the state
      // information matches.
      assert ((0 == pos [i] && -1 == oldToNew [i]) ||
	      (pos [i] == m_info[oldToNew [i]].m_object));
    }
#endif
    getLinkManager ()->updateAfterGC (oldToNew, this);
  }

  static SP<SubsetStorage> buildStorage (SubsetStorageScene *scene,
					 SubsetSceneConfiguration *conf)
  {
    SP<SubsetStorageConfiguration> storageConf =
      conf->getStorageConfiguration();
    storageConf->setMaintainer(scene);
    SP<SubsetStorage> s =
	dynamic_cast <SubsetStorage *>
      (&*conf->getStorageFactory()->makeIt(&*storageConf));
    assert (s);
    return s;
  }

  SubsetStorageScene::SubsetStorageScene
  (SP<SubsetSceneConfiguration> conf)
    : SceneGraph (conf->getSceneGraph()?
		  conf->getSceneGraph()->getLinkManager():
		  SP<LinkManager> (NEW (LinkManager ()))),
      m_storage (buildStorage (this, conf)),
      m_linkManagerGeneration (-1), m_drawer (conf->getDrawer()),
		    m_valid (true)
  {
    ref ();
    SP<SceneGraph> sg = conf->getSceneGraph();
    if (sg) {
      // The old scene graph no longer owns the LinkManager.  Instead, this new
      // one does.
      clearLinkManager (sg);
      // Copy from the old SceneGraph to the new.
      Dynavec <int> oldToNew;
      int maxI = sg->maxIndex ();
      oldToNew.extendTo (maxI, -1);
      for (int i = 0; i < maxI; i++) {
	const PhysicsObject *ob = sg->object (i);
	if (ob) {
	  // Next one does the right thing if we failed to add the object
	  // because it is out of bounds: put a -1 in oldToNew at that
	  // position.
	  oldToNew [i] = addObject (ob, sg->objectState (i));
	} else {
	  oldToNew [i] = -1;
	}
      }
      getLinkManager()->updateAfterGC (oldToNew, this);
    }
    deref ();
  }

  void SubsetStorageScene::updateUnboundedNeighbors
  (Dynavec <PhysicsObjectInfo> &movedBounded,
   const Float *state)
  {
    // The following code does not need to be particularly fast, because the
    // unbounded neighbors won't move most of the time.  We can get called two
    // ways -- either movedBounded is all unbounded objects, or movedBounded is
    // some objects that moved in space but have not had their index changed
    // since last time we called isRelevantNeighbor for them.  Either way, it's
    // correct to reuse the unbounded objects that aren't on movedBounded:
    // either there aren't any, or their index hasn't changed.
    Dynavec <PhysicsObjectInfo> newNeighbors;
    for (int i = 0; i < m_info.size(); i++) {
      ObjectInfo &oi = m_info[i];
      // We can get called during garbage collection, so it's okay if
      // oi.m_object is null.
      if (oi.m_object) {
	Dynavec <PhysicsObjectInfo> &oldNeighbors = oi.m_neighbors;
	newNeighbors.extendTo (0);
	bool changed = false;
	// Push the unbounded neighbors who didn't move into newNeighbors.
	for (int i = 0; i < oi.m_unboundedNeighbors; i++) {
	  if (VecUtil::find (oldNeighbors[i], movedBounded)) {
	    changed = true;
	  } else {
	    newNeighbors.push (oldNeighbors[i]);
	  }
	}
	// Push all of the guys that moved that now intersect the
	// object into newNeighbors.
	for (int i = 0; i < movedBounded.size(); i++) {
	  PhysicsObjectInfo &mbpoi = movedBounded [i];
	  if (mbpoi.m_object->physicsIntersect (state+mbpoi.m_index,
						oi.m_approximateBox)
	      &&
	      oi.m_object->isRelevantNeighbor (mbpoi.m_object, oi.m_stateIndex,
					       mbpoi.m_index)) {
	    changed = true;
	    newNeighbors.push (mbpoi);
	  }
	}
	// Most objects won't intersect the guy who just moved, so
	// they won't change.  Take a shortcut in the common case.
	if (changed) {
	  int newUnboundedNeighbors = newNeighbors.size();
	  // Push all of the rest of the neighbors onto newNeighbors.
	  for (int i = oi.m_unboundedNeighbors; i < oldNeighbors.size(); i++) {
	    newNeighbors.push (oldNeighbors[i]);
	  }
	  oldNeighbors = newNeighbors;
	  oi.m_unboundedNeighbors = newUnboundedNeighbors;
	}
      }
    }
  }

  // This method has to be fast.
  SP<Evaluator::OutOfBounds> SubsetStorageScene::eval (Float t,
						       const Float *state,
						       Float *deriv)
  {
    // Ignore t; the laws of physics don't care what time it is.
    (void) t; // Suppress warning
    memset ((void *) deriv, 0, m_states.size () * sizeof (Float));
    // Move everybody to their new positions.
    for (int i = 0; i < m_info.size(); i++) {
      ObjectInfo &oi = m_info[i];
      const PhysicsObject *const obj = oi.m_object;
      // FIXME Would be better to sort by the object type
      // Then we wouldn't have to call isPhysicsBounded for each one
      // every time.
      if (!obj) {
	assert (!m_valid);
      } else if (obj->isPhysicsBounded ()) {
	if (!m_storage->moveTo
	    (i, obj->boundingSphereCenter (state+oi.m_stateIndex))) {
	  m_valid = false;
	  oi.m_object = 0;
	}
      }
    }
    if (!m_valid) {
      validate ();
      return NEW (Evaluator::OutOfBounds ());
    };
    // Do the neighbor computations for the unbounded objects
    // ourselves.
    // There usually will not be any moving bounded objects, and
    // Dynavec's don't call new if they're empty, so it's okay to
    // have this Dynavec as a local variable in a hot loop.
    Dynavec <PhysicsObjectInfo> movedBounded;
    for (int i = 0; i < m_unbounded.size(); i++) {
      UnboundedObjectInfo &uoi = m_unbounded [i];
      bool moved = false;
      {
	const Float *stateStart = state + m_info[uoi.m_index].m_stateIndex;
	const Float *uoiStart = &uoi.m_state[0];
	const Float *const uoiEnd = uoiStart + uoi.m_state.size();
	while (uoiStart < uoiEnd) {
	  if (*(uoiStart++) != *(stateStart++)) {
	    moved = true;
	    break;
	  }
	}
      }
      if (moved) {
	movedBounded.push (m_info[uoi.m_index].poi());
	// Update the saved state vector.
	for (int i = 0; i < uoi.m_state.size(); i++) {
	  uoi.m_state[i] = state[m_info[uoi.m_index].m_stateIndex+i];
	}
      }
    }
    if (movedBounded.size()) {
#ifndef NDEBUG
      cout << "One or more unbounded objects moved!" << endl;
#endif
      updateUnboundedNeighbors (movedBounded, state);
    }
    if (getLinkManager()->getLinkGeneration () != m_linkManagerGeneration) {
      SP<LinkManager> lm = getLinkManager();
      m_linkManagerGeneration = lm->getLinkGeneration ();
      for (int i = 0; i < maxIndex(); i++) {
	int count;
	const int *links;
	lm->getLinks (i, count, links);
	ObjectInfo &oi = m_info [i];
	oi.m_links.extendTo (0);
	for (int j = 0; j < count; j++) {
	  const int l = links [j];
	  if (-1 == l) {
	    oi.m_links.push (PhysicsObjectInfo (0, -1));
	  } else {
	    oi.m_links.push (PhysicsObjectInfo (m_info [l].m_object,
						m_info[l].m_stateIndex));
	  }
	}
      }
    }
    // Do the physics.
    const ObjectInfo *oi = &*m_info;
    const ObjectInfo *const lastOi = oi+m_info.size();
    // Much work has gone into making the following loop trivial.
    for (; oi < lastOi; oi++) {
      oi->m_object->doPhysics (state, deriv,
			       oi->m_stateIndex, oi->m_neighbors,
			       oi->m_links);
    }
    return 0;
  }

  // Add the objects on the line to the indices and intersectionPoints vectors.
  void SubsetStorageScene::objectsOnLine
  (const Vec3 &v1, const Vec3 &v2,
   Dynavec <int> &indices,
   Dynavec <Vec3> &intersectionPoints)
    const
  {
    for (int i = 0; i < m_info.size (); i++) {
      const PhysicsObject *const o = m_info[i].m_object;
      assert (o);
      Vec3 where;
      if (o->getObjectDrawer()->intersect (statePointer (m_info[i]),
					   v1, v2, &where, i, this)) {
	indices.push (i);
	intersectionPoints.push (where);
      }
    }
  }

  BoundingSphere SubsetStorageScene::graphicsBoundingSphere () const {
    BoundingSphere result = BoundingSphere ();
    // FIXME is this taking a significant amount of time?
    // Do we ever call it?  Should it go away?
    // We needed it to set the near and far planes in the OpenGL version, but
    // not any more.
    for (int i = 0; i < m_info.size (); i++) {
      const PhysicsObject *o = m_info [i].m_object;
      assert (o);
      const ObjectDrawer *od = o->getObjectDrawer ();
      if (od->isVisible ()) {
	result.merge (od->
		      graphicsBoundingSphere (statePointer (m_info [i]),
					      i, this));
      }
    }
    return result;
  }

  int SubsetStorageScene::addObject (const PhysicsObject *obj,
				     const Float *initialState)
  {
    // If they are copying from the scene graph into the scene graph, they can
    // lose because extending the scene graph can make the pointer dangle.
    assert (! (initialState >= &*m_states &&
	       initialState < &*m_states+m_states.size()));
    check();
    checkStorageAgainstState (&*m_states);
    int newIndex = m_info.size();
    m_info.push (ObjectInfo ());
    ObjectInfo &oi = m_info[newIndex];
    assert (obj);
    oi.m_object = obj;
    // Now we know that obj has a positive reference count.
    assert (0 < obj->count ());
    oi.m_stateIndex = m_states.size ();
    // Need this to be initialized so the manager interface won't go
    // haywire when we call store.
    oi.m_unboundedNeighbors = 0;
    assert (0 == obj->stateSize() || initialState);
    for (int i = 0; i < obj->stateSize(); i++) {
      m_states.push (initialState [i]);
    }
    if (obj->isPhysicsBounded ()) {
      if (!m_storage->store (newIndex,
			     obj->boundingSphereCenter (initialState),
			     obj->physicsBoundingSphereRadius())) {
	// Oops, can't store it.  Un-grow the arrays, and return.
#ifndef NDEBUG
	cerr << "Ignoring attempt to add out-of-bounds object." << endl;
#endif
	m_states.extendTo (oi.m_stateIndex);
	m_info.extendTo (newIndex);
	newIndex = -1;
      }
    } else {
      // FIXME can't do non-exempt, non-bounded neighbors yet.
      // It isn't difficult, it just isn't important because right now
      // we don't have any.  To do non-exempt, non-bounded neighbors right,
      // ObjectInfo needs to have a vector saying where the object is in
      // all of the non-exempt, non-bounded objects that it is a
      // neighbor to, and we need to scan through and update that vector
      // every time the object moves to a new approximateBoundingBox.
      assert (obj->exempt());
      int unboundedIndex = m_unbounded.size();
      m_unbounded.push (UnboundedObjectInfo ());
      UnboundedObjectInfo &uoi = m_unbounded[unboundedIndex];
      uoi.m_index = newIndex;
      uoi.m_state.extendTo (0);
      for (int i = 0; i < obj->stateSize(); i++) {
	uoi.m_state.push (initialState[i]);
      }
      Dynavec <PhysicsObjectInfo> moved;
      moved.push (m_info[uoi.m_index].poi());
      updateUnboundedNeighbors (moved, &m_states[0]);
    }
    if (-1 != newIndex && obj->hasAnimationEver ()) {
      m_animations.push (newIndex);
    }
    check ();
    checkStorageAgainstState(&*m_states);
    return newIndex;
  }

  void SubsetStorageScene::deleteObject (int index) {
    ObjectInfo &oi = m_info[index];
    assert (oi.m_object);
    if (oi.m_object->hasAnimationEver ()) {
      int where;
      bool found = VecUtil::find (index, m_animations, &where);
      (void) found;
      assert (found);
      m_animations [where] = -1;
      // We set m_valid to false below, so next time we call validate the
      // animations array will be updated and the -1 that we just inserted
      // won't be there any more.  We want to have it now just in case a method
      // like hasAnimationNow chooses to delete the object from the scene graph
      // while we're looping over the m_animations array.  We used to have code
      // here that moved elements in the m_animations array around when an
      // animation is deleted, and that's a bad idea if we're looping over the
      // array at the time.
    }
    if (oi.m_object->isPhysicsBounded ()) {
      m_storage->remove (index);
    } else {
      int index = -1;
      for (int i = 0; i < m_unbounded.size(); i++) {
	if (m_unbounded [i].m_index == i) {
	  index = i;
	  break;
	}
      }
      assert (-1 != index);
      PhysicsObjectInfo poiToRemove = m_info[m_unbounded[index].m_index].poi();
      // Remove it from m_unbounded.
      m_unbounded[index] = m_unbounded[m_unbounded.size()-1];
      m_unbounded.pop();
      // Remove it from all of the neighbor lists.
      for (int i = 0; i < m_info.size(); i++) {
	ObjectInfo &n_oi = m_info[i];
	int pos;
	if (VecUtil::find (poiToRemove, n_oi.m_neighbors, &pos)) {
	  assert (pos < n_oi.m_unboundedNeighbors);
	  VecUtil::removeNoDups (poiToRemove, n_oi.m_neighbors);
	  n_oi.m_unboundedNeighbors--;
	}
      }
    }
    oi.m_object = 0;
    m_valid = false;
    check ();
  }

  void SubsetStorageScene::validate () {
    if (! m_valid) {
      garbageCollectObj ();
      m_valid = true;
    }
  }

  bool SubsetStorageScene::draw (SP<Canvas> vo) {
    validate ();
    return m_drawer->draw (this, &*vo);
  }

  Dynavec <Float> &SubsetStorageScene::stateVec () {
    return m_states;
  }

  const Dynavec <Float> &SubsetStorageScene::stateVec () const {
    return m_states;
  }

  void SubsetStorageScene::updateAnimations () {
    m_animations.extendTo (0);
    for (int i = 0; i < maxIndex (); i++) {
      if (m_info[i].m_object && m_info[i].m_object->hasAnimationEver ()) {
	m_animations.push (i);
      }
    }
  }

  void SubsetStorageScene::animateTimeStep () {
    for (int i = 0; i < m_animations.size(); i++) {
      const int objNo = m_animations[i];
      if (-1 != objNo) {
	// When an animation is deleted, a -1 should have been inserted into
	// the m_animations array, so m_info[objNo].m_object should not be null
	// when we observe that the element in m_animations isn't -1.
	assert (m_info[objNo].m_object);
	m_info[objNo].m_object->animateTimeStep (this, objNo);
      }
    }
  }

  void SubsetStorageScene::animateFrame (Canvas *canvas) {
    for (int i = 0; i < m_animations.size(); i++) {
      const int objNo = m_animations[i];
      if (-1 != objNo) {
	assert (m_info[objNo].m_object);
	m_info[objNo].m_object->animateFrame (canvas, this, objNo);
      }
    }
  }

  bool SubsetStorageScene::anyAnimations () const {
    for (int i = 0; i < m_animations.size(); i++) {
      const int objNo = m_animations[i];
      if (-1 != objNo) {
	assert (m_info[objNo].m_object);
	if (m_info[objNo].m_object->hasAnimationNow (this, objNo)) {
	  return true;
	}
      }
    }
    return false;
  }

  // Methods on Maintainer
  bool SubsetStorageScene::addToSubset (int object, int toAdd) {
    if (m_info[object].m_object->isRelevantNeighbor (m_info[toAdd].m_object,
						     m_info[object].m_stateIndex,
						     m_info[toAdd].m_stateIndex)) {
      m_info[object].m_neighbors.push (m_info[toAdd].poi());
      return true;
    } else {
      return false;
    }
  }
  void SubsetStorageScene::deleteFromSubset (int object, int position)
  {
    ObjectInfo &oi = m_info[object];
    Dynavec <PhysicsObjectInfo> &neighbors = oi.m_neighbors;
    neighbors[oi.m_unboundedNeighbors+position] =
      neighbors [neighbors.size()-1];
    neighbors.pop();
  }
  
  // This one must be fast.
  inline void SubsetStorageScene::newBoundingBox (int object,
						  const BoundingBox &newBox)
  {
    ObjectInfo &oi = m_info[object];
    oi.m_approximateBox = newBox;
    m_newNeighbors.extendTo (0);
    bool changed = false;
    for (int i = 0; i < m_unbounded.size(); i++) {
      UnboundedObjectInfo &uoi = m_unbounded [i];
      ObjectInfo &u_oi = m_info[uoi.m_index];
      if (u_oi.m_object->physicsIntersect (&uoi.m_state[0], newBox)
	  &&
	  oi.m_object->isRelevantNeighbor (u_oi.m_object, oi.m_stateIndex,
					   u_oi.m_stateIndex)) {
	PhysicsObjectInfo poi = u_oi.poi();
	const int newSize = m_newNeighbors.size();
	changed = changed ||
	  newSize >= oi.m_unboundedNeighbors ||
	  poi != oi.m_neighbors[newSize];
	m_newNeighbors.push (poi);
      }
    }
    // Usually there won't be any change in the unbounded neighbors.
    const int oldUnboundedNeighbors = oi.m_unboundedNeighbors;
    const int newUnboundedNeighbors = m_newNeighbors.size();
    if ((newUnboundedNeighbors != oldUnboundedNeighbors) || changed) {
      const int boundedNeighbors =
	oi.m_neighbors.size() - oldUnboundedNeighbors;
      if (newUnboundedNeighbors > oldUnboundedNeighbors) {
	oi.m_neighbors.extendTo (boundedNeighbors+newUnboundedNeighbors);
      }
      PhysicsObjectInfo *start = &*(oi.m_neighbors);
      memmove ((void *)(start + newUnboundedNeighbors),
	       (void *)(start + oldUnboundedNeighbors),
	       boundedNeighbors * sizeof (PhysicsObjectInfo));
      memcpy ((void *)start, (void *)&*m_newNeighbors,
	      newUnboundedNeighbors * sizeof (PhysicsObjectInfo));
      // Shrink to fit.
      oi.m_neighbors.extendTo (boundedNeighbors + newUnboundedNeighbors);
      oi.m_unboundedNeighbors = newUnboundedNeighbors;
    }
  }

  void SubsetStorageScene::checkNeighbors (int object,
					   const Dynavec <int> &neighborList)
    const
  {
    // Next two declarations avoid a warning when compiling with -DNDEBUG.
    (void) object;
    (void) neighborList;
    assert (neighborList.size() ==
	    (m_info[object].m_neighbors.size() -
	     m_info[object].m_unboundedNeighbors));
    for (int i = 0; i < neighborList.size(); i++) {
      assert (m_info[neighborList [i]].poi() ==
	      m_info[object].m_neighbors[i +
					 m_info[object].m_unboundedNeighbors]);
    }
  }

  class SubsetSceneFactory : public Factory {
  public:
    SubsetSceneFactory () : Factory ("SubsetStorageScene") {
    }
    SP<Configuration> defaultConfiguration () const {
      return NEW (SubsetSceneConfiguration ());
    }
    SP<Configurable> makeIt (SP<Configuration> c) const {
      SubsetSceneConfiguration *conf =
	dynamic_cast <SubsetSceneConfiguration *> (&*c);
      assert (conf);
      return NEW (SubsetStorageScene (conf));
    }
  };

  static const bool useless =
  (FactoryTable::store ("SceneGraph", NEW (SubsetSceneFactory ())),
   true);

} // namespace

