// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __SubsetStorage_h__
#define __SubsetStorage_h__

// Scene graphs typically need to maintain a filtered list of
// approximate neighbors to each object.  This class tries to do an
// optimal job of incrementally maintaining such lists.

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

// Why a SubsetStorage is not an IncrementalStorage or a Storage:
//
// Our store and moveTo methods take a BoundingSphere as an argument
// instead of a BoundingBox, so we can shrink the box so it bounds the
// center of the object instead of bounding the bounding box of the object.
// This way we can determine in half the time whether the object has
// moved out of the box.  Because of this change, a SubsetStorage is
// not an IncrementalStorage or a Storage.  Also, the empty() method
// is obsolete, since it's better just to destruct the whole thing.
// So define this new type of storage from scratch.

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

class BoundingSphere;
class BoundingBox;
class Vec3;

class SubsetStorage : public Configurable {
public:
  // The scene graph has to implement the following interface to play
  // its role.  We assume that the scene graph's approximate neighbor
  // list is represented as a vector, so we use this interface to
  // issue orders to add and remove elements from the vector.
  //
  // To avoid reference cycles, SubsetStorage points to Maintainer, but it
  // presumes that Maintainer also points at SubsetStorage, so it
  // doesn't count its reference to Maintainer.  If this isn't the
  // case, the guy who has a reference to a SubsetStorage had better
  // maintain a reference to the Maintainer too, otherwise
  // SubsetStorage is likely to suffer from a dangling pointer.
  class Maintainer {
  public:
    // toAdd is an object index, just like object is.
    // Return false if we can't add it to the subset.  Typically we will return
    // false if the information that toAdd is near object is not worth
    // remembering.
    virtual bool addToSubset (int object, int toAdd) = 0;
    // Delete the subset entry at the given positition in the subset
    // for the given object.
    virtual void deleteFromSubset (int object, int position) = 0;
    // When an object moves to a new place, the Maintainer needs to
    // know because the Maintainer probably has it's own bounding-box
    // dependent stuff stashed away.  (For SceneGraph's, the nearby
    // unbounded objects will typically be such stuff.)
    // We never need to know the old bounding box, so we call the
    // following method whether the new bounding box is for a new
    // object, or an old object that moved.
    virtual void newBoundingBox (int object, const BoundingBox &newBox) = 0;
    // When debugging is enabled, the next method will be called with
    // what we think the neighbor list should be, and it should abort
    // if that disagrees with the state maintained by the scene graph.
    // We don't have an #ifndef NDEBUG here because it's best for the
    // interfaces to be constant whether or not we are compiling with
    // debugging.  This way I could load in a debug version of
    // something if I wanted.
    virtual void checkNeighbors (int object, const Dynavec <int> &neighborList)
      const = 0;
  };
  // The constructor for the concrete class should definitely take a
  // Maintainer pointer as an argument.
  virtual ~SubsetStorage () {};
  // Say that the object moved to the given location from wherever it
  // was before.  Returns true if the movement was a success.
  // Otherwise the object fell off of the edge of the world and has
  // been deleted from the SubsetStorage.
  virtual bool moveTo (int value, const Vec3 &where) = 0;
  // Specify a new object.  where and radius define the bounding
  // sphere.  We do not support bounding spheres with a changing radius.
  virtual bool store (int object, const Vec3 &where, const Float radius) = 0;
  // Delete the value from the storage.  
  // Beware -- if an attempt was made to store the object, and the
  // attempt returned "false" because the object was out of bounds,
  // then it is not valid to delete the object.
  virtual void remove (int value) = 0;
  // Empty out the scene graph.  Don't bother telling the maintainer
  // about this, since it has faster ways to empty itself out
  // than by using the Maintainer interface. 
  virtual void empty () = 0;
  // The "fetch" method from storage has gone away, since it is
  // replaced by the calls that happen on the Maintainer object.
};

#if 0
// The following method on Maintainer would do everything we need done
// with one virtual method call instead of a bunch, but the
// programming is too complex for me right now.  Let's do it the easy
// way and tighten it up later, maybe.  It may be the case that the
// simpler interfaces allows passing more stuff in registers and is
// actually fater than this one.
    virtual void updateNeighbors
    (// The index of the object that moved.
     int object,
     // Its new approximate bounding box.
     const BoundingBox &b,
     // The positions, in the subset for object, of the neighbors to
     // delete.  The manager must do the deletions in order starting
     // at the beginning.
     const Dynavec <int> &objectOldGoneNeighborPositions;
     // The object numbers for the neighbors of the object that is now
     // gone, if those neighbors had object in the subset.
     const Dynavec <int> &otherOldGoneNeighbors;
     // The indices into the subset of this object for each of the
     // corresponding elements of otherOldGoneNeighbors.
     const Dynavec <int> &otherOldGoneNeighborPositions;
     // The new neighbors of the object that moved.
     const Dynavec <int> &newNeighbors,
     // The returned positions of each of the new neighbors in the
     // subset for object, or -1 if the new neighbor didn't belong in
     // the subset.
     Dynavec <int> &newNeigborSubsetPositions,
     // The returned positions of the object in each of the subsets of
     // the neighbors in newNeighbors, or -1 if the object didn't
     // belong in the new neighbor's subset.
     Dynavec <int> &newObjectSubsetPositions);
#endif
#endif
