// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "SubsetStorage.h"

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __IncrementalStorage_h__
#include "IncrementalStorage.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __BoundingSphere_h__
#include "BoundingSphere.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __SubsetStorageConfiguration_h__
#include "SubsetStorageConfiguration.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

namespace {

  // FIXME use a boolean array to do fast set operations to only
  // update the guys who change when we move to a new approximateBox.
  // Two steps, first figure out the guys who really went away, then
  // figure out the guys who are really new.
  // (Deferring because I know I don't need to fix it for the release.  This
  // might already be fixed; think about it later.)
  // FIXME m_subset could be a list of int *'s, not ints.
  // Small performance gain in any case.  Maybe drop this one.
  // FIXME m_myPos could be an int *, not an int.
  // Small performance gain in any case.  Maybe drop this one too.

  // To keep abstractness from obfuscating this, let's admit that we
  // know the normal case, which is that a SceneGraph is using a
  // SubsetStorage and is also the Manager that the SubsetStorage
  // points at.

  // Then a node is the index of an object in the SceneGraph.  The
  // name of a node is an integer, which is its position 
  // in m_neighbors, which will also be its index in the SceneGraph.

  // What a node knows about each of its neighbors.
  struct NeighborNeighborInfo {
    // The position of the neighbor in m_objects.
    int m_name;
    // The position of this node (that is, the one that the
    // ObjectNeighborInfo is about) in the neighbor list of this
    // neighbor.
    int m_myPos;
    // The position of this neighbor in the subset of nodes that the
    // Manager is maintaining information about.  Negative if this
    // node is not in the set.  Has to be an integer, not a direct
    // pointer to the destination, because we pass it to
    // deleteNeighborAt. 
    int m_subsetPos;
    NeighborNeighborInfo () {};
    NeighborNeighborInfo (int name, int myPos, int subsetPos)
      : m_name (name), m_myPos (myPos), m_subsetPos (subsetPos)
    {}
  };

  // Everything we know about a node.
  struct ObjectNeighborInfo {
    // The approximate neighbors of the given node.  Unordered.
    Dynavec <NeighborNeighborInfo> m_list;
    // The managed subset, given as a list of positions of elements in
    // m_list.
    Dynavec <int> m_subset;
    // The radius that the bounding sphere had when we stored
    // originally.  If the object is not present, this is negative. 
    Float m_radius;
    // The bounding box that we used to store this index.  The object
    // may have moved since then.
    BoundingBox m_box;
    // The bounding box that store claims it actually stored it into.
    BoundingBox m_approximateBox;
    // Same as m_approximateBox, except this one is shrunk so we only
    // have to compare the center of the bounding sphere against it.
    BoundingBox m_shrunkenApproximateBox;
    ObjectNeighborInfo () : m_radius (-1)
#ifndef NDEBUG
      // Need to initialize the data that will be passed to fetch for
      // nonexistent nodes in checkNeighborsAgainstStorage.
      , m_box (Vec3 (0, 0, 0), Vec3 (1, 1, 1))
#endif
    {}
    inline bool nodeExists () const {
      return m_radius >= 0;
    }
  };

  class SubsetStorageImpl : public SubsetStorage {
    Dynavec <ObjectNeighborInfo> m_objects;
    // m_maintainer deliberately is not a smart pointer; see comments
    // in SubsetStorage.h.
    Maintainer * const m_maintainer;
    SP<IncrementalStorage> m_storage;
    // Next one is logically local of doStore, but we don't want to
    // reallocate it again and again.
    mutable Dynavec <int> m_fetchResult;
    // Used for set operations on neighbors.
    Dynavec <bool> m_flag;
    // The configuration, so we can increment the moved and notMoved counters.
    SP<SubsetStorageConfiguration> m_conf;

    // Verify that the information in m_neighbors is consistent with the
    // information in m_storage.

    // Jargon in the variable names below: 
    // o = object = the node we're working on
    // n = neighbor = a neighbor of the node we're working on.
    // s = stranger = a neighbor of n
    inline void checkNeighborsAgainstStorage () {
#ifdef VERY_SLOW_DEBUG
      for (int i = 0; i < m_objects.size(); i++) {
	if (m_objects[i].nodeExists ()) {
	  Dynavec <int> fetched;
	  m_storage->fetch (m_objects [i].m_box, i, fetched);
	  Dynavec <int> now;
	  for (int j = 0; j < m_objects[i].m_list.size(); j++) {
	    now.push (m_objects[i].m_list[j].m_name);
	  }
	  assert (VecUtil::setEqual (fetched, now));
	}
      }
#endif
    }

    inline void checkSubset (int object) {
      (void) object;
#ifdef VERY_SLOW_DEBUG
      ObjectNeighborInfo &o_oni = m_objects[object];
      Dynavec <int> subsetNames;
      for (int j = 0; j < o_oni.m_list.size(); j++) {
	const int n_sPos = o_oni.m_list[j].m_subsetPos;
	if (n_sPos >= 0) {
	  if (n_sPos >= subsetNames.size()) {
	    subsetNames.extendTo (n_sPos + 1, -1);
	  }
	  subsetNames[n_sPos] = o_oni.m_list[j].m_name;
	  assert (j == o_oni.m_subset[n_sPos]);
	} else {
	  assert (! VecUtil::find (n_sPos, o_oni.m_subset));
	}
      }
      assert (o_oni.m_subset.size() == subsetNames.size());
      m_maintainer->checkNeighbors (object, subsetNames);
#endif
    }

    inline void checkFlagsCleared () {
#ifdef VERY_SLOW_DEBUG
      for (int i = 0; i < m_objects.size(); i++) {
	assert (! m_flag[i]);
      }
#endif
    }

    inline void check () {
#ifdef VERY_SLOW_DEBUG
      Dynavec <int> testNeighborList;
      for (int i = 0; i < m_objects.size(); i++) {
	ObjectNeighborInfo &o_oni = m_objects[i];
	if (o_oni.nodeExists ()) {
	  // No duplicates on neighbor list.
	  for (int j = 0; j < o_oni.m_list.size(); j++) {
	    for (int k = 0; k < j; k++) {
	      assert (o_oni.m_list[j].m_name != o_oni.m_list[k].m_name);
	    }
	  }
	  // The m_myPos variables are consistent, and also
	  // check that I am a neighbor of each of my neighbors.
	  for (int j = 0; j < o_oni.m_list.size(); j++) {
	    NeighborNeighborInfo &n_nni = o_oni.m_list[j];
	    assert (i == m_objects[n_nni.m_name].m_list[n_nni.m_myPos].m_name);
	  }
	  checkSubset (i);
	}
      }
#endif
    }

    // Every reference into m_objects costs us an integer multiply and
    // add.  I'm passing around references to avoid that.

    // Given an object and a position of one of its neighbors, delete
    // that neighbor from the subset we're maintaining, if it's
    // there.
    // We only need to take object as an argument so we can pass it to
    // m_maintainer->deleteFromSubset.
    inline void deleteFromSubsetOneWay (ObjectNeighborInfo &o_oni,
					const int object, const int pos)
    {
      checkSubset (object);
      assert (&o_oni == &m_objects [object]);
      int &subsetPosLoc = o_oni.m_list[pos].m_subsetPos;
      const int subsetPos = subsetPosLoc;
      if (subsetPos >= 0) {
	Dynavec <int> &o_subset = o_oni.m_subset;
	const int last = o_subset.size() - 1;
	if (subsetPos < last ) {
	  const int toMove = o_subset [last];
	  o_subset [subsetPos] = toMove;
	  // Update the back pointer for the last element in the subset.
	  o_oni.m_list[toMove].m_subsetPos = subsetPos;
	}
	o_subset.pop();
	m_maintainer->deleteFromSubset (object, subsetPos);
	subsetPosLoc = -1;
      }
      checkSubset (object);
    }

    // Move the neighbor at position from in the object to the
    // position to, updating from's neighbor back pointer and subset
    // back pointer.
    inline void moveNeighbor (ObjectNeighborInfo &o_oni,
			      Dynavec <NeighborNeighborInfo> &o_list,
			      const int to,
			      const int from)
    {
      const NeighborNeighborInfo &o_nni = o_list [from];
      o_list [to] = o_nni;
      // Update the backpointer in the neighbor's m_myPos.
      m_objects[o_nni.m_name].m_list[o_nni.m_myPos].m_myPos = to;
      // Update the backpointer in m_subset.
      const int subsetPos = o_nni.m_subsetPos;
      if (0 <= subsetPos) {
	o_oni.m_subset [subsetPos] = to;
      }
    }

    // Given an object and a position of one of its neighbors, delete
    // that neighbor from this object's list and update the back pointer.
    inline void deleteFromNeighborsOneWay (ObjectNeighborInfo &o_oni,
					   const int pos)
    {
      Dynavec <NeighborNeighborInfo> &o_list = o_oni.m_list;
      const int o_last = o_list.size()-1;
      if (pos < o_last) {
	moveNeighbor (o_oni, o_list, pos, o_last);
      }
      o_list.pop();
    }

    // This routine is the only deletexxxx inline that's called from
    // two places.  Could plausibly declare this non-inline; cost is
    // two subroutine calls per loop and presumably impaired
    // variable-to-register allocation, benefit is smaller code size.
    // It isn't a whole lot of code, so leave it inline.
    // This needs to take object only because deleteFromSubsetOneWay
    // has to pass it to m_maintainer->deleteFromSubset.
    inline void deleteFromNeighborsAndSubsetOneWay (ObjectNeighborInfo &o_oni,
						    int object, int pos)
    {
      assert (&o_oni == &m_objects[object]);
      deleteFromSubsetOneWay (o_oni, object, pos);
      deleteFromNeighborsOneWay (o_oni, pos);
    }

    // This needs to take object only because object ultimately has to
    // be passed to m_maintainer->deleteFromSubset.
    inline void deleteFromNeighborsAndSubsetTwoWay (ObjectNeighborInfo &o_oni,
						    int object, int pos)
    {
      check ();
      assert (&o_oni == &m_objects[object]);
      NeighborNeighborInfo &o_nni = o_oni.m_list[pos];
      const int n_name = o_nni.m_name;
      deleteFromNeighborsAndSubsetOneWay (m_objects[n_name],
					  n_name,
					  o_nni.m_myPos);
      // At this point, the backpointers should be consistent, except
      // for my pointer to my deleted position in the neighbor.
      // Can't do this in the other order, since
      // deleteFromNeighborsAndSubsetOneWay(object,pos)
      // makes me forget who my neighbor was.
      deleteFromNeighborsAndSubsetOneWay (o_oni, object, pos);
      check ();
    }

    // Delete the neighbors of the node that aren't on the "keep"
    // list.  
    inline void deleteFromNeighbors (int object, const Dynavec <int> &keep)
    {
      checkFlagsCleared ();
      for (int i = 0; i < keep.size(); i++) {
	assert (! m_flag[keep[i]]);
	m_flag[keep[i]] = true;
      }
      ObjectNeighborInfo &o_oni = m_objects[object];
      Dynavec <NeighborNeighborInfo> &o_list = o_oni.m_list;
      int i = 0;
      for (;;) {
	// Beware that the next loop can either increase i or decrase
	// o_list.size().
	if (i >= o_list.size()) break;
	if (m_flag[o_list[i].m_name]) {
	  // The neighbor is in the keep list, so look at the next neighbor.
	  i++;
	} else {
	  deleteFromNeighborsAndSubsetTwoWay (o_oni, object, i);
	}
      }
      for (int i = 0; i < keep.size(); i++) {
	assert (m_flag[keep[i]]);
	m_flag[keep[i]] = false;
      }
      checkFlagsCleared ();
    }

    inline void addToNeighborsAndSubsetOneWay
    (ObjectNeighborInfo &o_oni,
     Dynavec <NeighborNeighborInfo> &o_list,
     const int object,
     // The position of the neighbor in m_objects.
     const int neighbor,
     // The eventual position of the neighbor in m_objects[object].m_list.
     const int neighborPos,
     // The eventual position of object in m_objects[neighbor].m_list.
     const int objectPos
     )
    {
      assert (&o_oni == &m_objects[object]);
      assert (&o_list == &o_oni.m_list);
      const bool subsetP = m_maintainer->addToSubset (object, neighbor);
      int subsetPos;
      Dynavec <int> &o_subset = o_oni.m_subset;
      if (subsetP) {
	subsetPos = o_subset.size();
	o_subset.push (neighborPos);
      } else {
	subsetPos = -1;
      }
      o_list.push (NeighborNeighborInfo (neighbor, objectPos, subsetPos));
    }

    inline void addToNeighborsAndSubsetTwoWay (ObjectNeighborInfo &o_oni,
					       const int object,
					       const int neighbor)
    {
      check ();
      Dynavec <NeighborNeighborInfo> &o_list = o_oni.m_list;
      const int neighborPos = o_list.size();
      ObjectNeighborInfo &n_oni = m_objects[neighbor];
      Dynavec <NeighborNeighborInfo> &n_list = n_oni.m_list;
      const int objectPos = n_list.size();
      addToNeighborsAndSubsetOneWay (o_oni, o_list, object,
				     neighbor, neighborPos,
				     objectPos);
      addToNeighborsAndSubsetOneWay (n_oni, n_list, neighbor,
				     object, objectPos,
				     neighborPos);
      check ();
    }

    // Add the neighbors "newNeighbors" to object, unless they are
    // already neighbors.
    inline void addToNeighbors (const int object,
				const Dynavec <int> &newNeighbors)
    {
      checkFlagsCleared ();
      ObjectNeighborInfo &o_oni = m_objects [object];
      Dynavec <NeighborNeighborInfo> o_list = o_oni.m_list;
      const int o_listSizeAtStart = o_list.size();
      for (int i = 0; i < o_listSizeAtStart; i++) {
	assert (! m_flag[o_list[i].m_name]);
	m_flag[o_list[i].m_name] = true;
      }
      for (int i = 0; i < newNeighbors.size(); i++) {
	const int newNeighbor = newNeighbors [i];
	if (! m_flag[newNeighbor]) {
	  addToNeighborsAndSubsetTwoWay (o_oni, object, newNeighbor);
	}
      }
      for (int i = 0; i < o_listSizeAtStart; i++) {
	assert (m_flag[o_list[i].m_name]);
	m_flag[o_list[i].m_name] = false;
      }
      checkFlagsCleared ();
    }
  public:
    SubsetStorageImpl (SP<IncrementalStorage> is, Maintainer *m,
		       SP<SubsetStorageConfiguration> ssc)
      : m_maintainer (m), m_storage (is), m_conf (ssc)
    {
      assert (m);
    }
    // The goal is to make this method fast.
    bool moveTo (int object, const Vec3 &where) {
#ifndef NDEBUG
      // Used to call checkNeighborsAgainstStorage every time, but
      // that was too slow for much debugging to happen.  Now call it
      // only when the object number decreases. 
#ifdef VERY_SLOW_DEBUG
      bool needCheck = true;
#else
      static int lastObject = -1;
      bool needCheck = object < lastObject;
      lastObject = object;
#endif
      if (needCheck) {
	checkNeighborsAgainstStorage ();
	check();
      }
#endif
      ObjectNeighborInfo &o_oni = m_objects [object];
      assert (o_oni.nodeExists ());
      if (o_oni.m_shrunkenApproximateBox.contains (where)) {
	m_conf->incNotMoved ();
	return true;
      } else {
	m_conf->incMoved ();
	// FIXME suboptimal; add move to IncrementalStorage.h
	// and ArrayStorage.cpp.  See comments in IncrementalStorage.h.
	m_storage->remove (object, o_oni.m_box);
	// FIXME suboptimal; could construct BoundingBox directly
	// from radius and where.
	o_oni.m_box = BoundingBox (BoundingSphere (o_oni.m_radius, where));
	BoundingBox newApproximateBox;
	if (m_storage->store (object, o_oni.m_box, newApproximateBox)) {
	  m_maintainer->newBoundingBox (object, newApproximateBox);
	  o_oni.m_approximateBox = newApproximateBox;
	  o_oni.m_shrunkenApproximateBox =
	    newApproximateBox.shrink (o_oni.m_radius);
	  m_storage->fetch (o_oni.m_box, object, m_fetchResult);
	  deleteFromNeighbors (object, m_fetchResult);
	  addToNeighbors (object, m_fetchResult);
#ifndef NDEBUG
	  if (needCheck) {
	    checkNeighborsAgainstStorage ();
	    check();
	  }
#endif
	  return true;
	} else {
	  // Mark the node as nonexistent.
	  o_oni.m_radius = -1;
	  m_fetchResult.extendTo (0);
	  deleteFromNeighbors (object, m_fetchResult);
#ifndef NDEBUG
	  if (needCheck) {
	    checkNeighborsAgainstStorage ();
	    check();
	  }
#endif
	  return false;
	}
      }
    }
    bool store (int object, const Vec3 &where, Float radius) {
      checkNeighborsAgainstStorage ();
      assert (radius >= 0);
      if (object >= m_objects.size()) {
	m_objects.extendTo (object + 1, ObjectNeighborInfo ());
	m_flag.extendTo (object + 1, false);
      }
      ObjectNeighborInfo &o_oni = m_objects [object];
      assert (!o_oni.nodeExists ());
      o_oni.m_radius = radius;
      // FIXME suboptimal; could construct BoundingBox directly
      // from radius and where.
      o_oni.m_box = BoundingBox (BoundingSphere (radius, where));
      // FIXME suboptimal; add storeAndFetch to IncrementalStorage.h
      // and ArrayStorage.cpp.  See comments in IncrementalStorage.h.
      // Except maybe don't bother because store can be slow.
      if (m_storage->store (object, o_oni.m_box, o_oni.m_approximateBox)) {
	m_maintainer->newBoundingBox (object, o_oni.m_approximateBox);
	o_oni.m_shrunkenApproximateBox =
	  o_oni.m_approximateBox.shrink (radius);
	m_storage->fetch (o_oni.m_box, object, m_fetchResult);
	addToNeighbors (object, m_fetchResult);
	checkNeighborsAgainstStorage ();
	return true;
      } else {
	o_oni.m_radius = -1;
	checkNeighborsAgainstStorage ();
	return false;
      }
    }
    void remove (int object) {
      checkNeighborsAgainstStorage ();
      ObjectNeighborInfo &o_oni = m_objects [object];
      assert (o_oni.nodeExists ());
      // Empty dynavec doesn't need to call "new", so don't bother
      // promoting this to an instance variable.
      Dynavec <int> empty;
      deleteFromNeighbors (object, empty);
      m_storage->remove (object, o_oni.m_box);
      // Mark the node as nonexistent.
      o_oni.m_radius = -1;
      checkNeighborsAgainstStorage ();
    }
    void empty () {
      checkNeighborsAgainstStorage ();
      m_objects.extendTo (0);
      m_storage->empty();
      m_fetchResult.extendTo (0);
      m_flag.extendTo (0);
      checkNeighborsAgainstStorage ();
    }
  };
  
  class SubsetStorageFactory : public Factory {
  public:
    SubsetStorageFactory () : Factory ("SubsetStorage")
    {}
    SP<Configuration> defaultConfiguration () const {
      return NEW (SubsetStorageConfiguration ());
    }
    SP<Configurable> makeIt (SP<Configuration> c) const {
      SP<SubsetStorageConfiguration> conf =
	dynamic_cast <SubsetStorageConfiguration *> (&*c);
      assert (conf);
      SP<Configurable> isc = conf->getIncrementalStorageSlot()->makeIt();
      SP<IncrementalStorage> is = dynamic_cast <IncrementalStorage *> (&*isc);
      assert (is);
      return NEW (SubsetStorageImpl (is, conf->getMaintainer (), conf));
    }
  };

  static const bool useless =
  (FactoryTable::store ("SubsetStorage", NEW (SubsetStorageFactory ())),
   true);
} // namespace
