// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __SelectionManager_h__
#define __SelectionManager_h__

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

class PhysicsObject;
class SceneGraph;
class Matrix4;
class Quaternion;
class ObjectLink;
class SelectionManager
{
public:
  // Get and set the current selection, as a list of integer object indices.
  //
  // Next one filters out any -1's for deleted objects.  It also calls
  // validate, so you won't see pointers to deleted objects that haven't been
  // garbage collected.
  static void getSelection (Dynavec <int> &selection);
  // Call this getSelectionLinks instead of overloading getSelection because if
  // we overload getSelection, we apparently need to include ObjectLink.h to
  // resolve the overloading, and that would give us cyclic header includes.
  static void getSelectionLinks (Dynavec <ObjectLink> &selection);
  // When setting the selection, we need to know the SceneGraph.  If we are
  // adding the first selection, and it is nonempty, we will need to insert an
  // object into the scene graph and an animation, so the selection will blink.
  // If we had a nontrivial selection and we are making it empty, we need to
  // remove the animation, so that the top level can recognize that it has no
  // animations and therefore it might be idle and want to relinquish the CPU.
  //
  // -1's and repeated object indices are filtered out of the selection list
  // that is actually stored.
  static void setSelection (SceneGraph *sg, const Dynavec <int> &selection);
  static void setSelectionLinks (SceneGraph *sg,
				 const Dynavec <ObjectLink> &selection);
  // Either the selection blinks, or the selection is indistinguishable from
  // the rest of the scene.  You don't want the selection to blink when you're
  // moving the selected objects.
  static void setSelectionBlink (bool blink);
  static bool getSelectionBlink ();
  // The invisible objects are like the selection, except instead of blinking,
  // they are always invisible.  They all should be selectable objects.
  static void getInvisible (Dynavec <int> &invisible);
  static void setInvisible (SceneGraph *sg, const Dynavec <int> &invisible);
  // Whether an object is selectable.  Returns true if the object considers
  // itself selectable and it isn't currently on the invisibility list.  This
  // caches enough information to be reasonably efficient; it updates the cache
  // when the garbageCollectionCount on the SceneGraph increases or the set of
  // invisible objects changes.
  // If someone digs the InvisibilityMarker out of the scene and manipulates it
  // directly, then the cache used by isSelctable will become wrong without its
  // knowledge so it will start to return incorrect results.
  static bool isSelectable (const ObjectLink &ol);
  // Apply the matrix m to the selection.  m ought to be a translation and a
  // scale and the rotation q in some order.  Rotate the objects in the
  // selection in place by q, so the whole selection moves rigidly.
  // 
  // Strictly speaking, q is redundant, since it should be implied by m, but I
  // do need to rotate the objects in place and I don't have code right now
  // that can extract q from m.
  static void transformSelection (const Matrix4 &m, const Quaternion &q);
};
#endif
