// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __OL_h__
#include "OL.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __InvisibilityMarker_h__
#include "InvisibilityMarker.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __ObjectLink_h__
#include "ObjectLink.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __Matrix4_h__
#include "Matrix4.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

// Next three are for code that limits the number of frames per second.
#ifndef __sys_timeb_h__
#include <sys/timeb.h>
#define __sys_timeb_h__
#endif

// For usleep.
#ifndef __unistd_h__
#include <unistd.h>
#define __unistd_h__
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

namespace {
  bool s_doBlink = true;
  class SelectionInvisibilityMarker
    : public InvisibilityMarker
  {
  public:
    SelectionInvisibilityMarker () {
    }

    bool hasAnimationEver () const {
      return true;
    }

    bool hasAnimationNow (const SceneGraph *sg, int index) const {
      if (!s_doBlink) return false;
      int count;
      const int *links;
      sg->getLinkManager()->getLinks (index, count, links);
      bool result = false;
      for (int i = 0; i < count; i++) {
	if (-1 != links[i]) {
	  result = true;
	  break;
	}
      }
#ifndef NDEBUG
      for (int i = 0; i < count; i++) {
	if (-1 != links [i] && sg->object (links [i])) {
	  assert (sg->object (links[i])->isSelectable ());
	}
      }
#endif
      return result;
    }

    // Don't try to inline this since it has a static variable.
    void animateFrame (Canvas *canvas, SceneGraph *sg, int index) const;

    static const char *const staticClassId = "SelectionInvisibilityManager";

    const char *classId () const {
      return staticClassId;
    }

    bool isInstanceOf (const char *classId) const {
      return (staticClassId == classId) ||
	InvisibilityMarker::isInstanceOf (classId);
    }
  };

  void SelectionInvisibilityMarker::animateFrame (Canvas *canvas, SceneGraph *sg, int index) const {
    (void) canvas; (void) sg; (void) index;
    // Blink on and off with each frame.
    if (s_doBlink) {
      // Minimum number of milliseconds between frames
      const int minblink = 100;
      static struct timeb lastblink;
      static bool lastblink_init = false;
      if (!lastblink_init) {
	lastblink_init = true;
	ftime (&lastblink);
      }
      struct timeb now;
      ftime (&now);
      const int millis =
	(now.time - lastblink.time) * 1000 + now.millitm - lastblink.millitm;
      // millis could conceivably be negative if lastblink is very old
      // and we've had an integer overflow when computing millis.
      if (millis < minblink && millis >= 0) {
	usleep ((minblink - millis) * 1000);
      }
      ftime (&now);
      lastblink = now;
      setVisible (!getVisible ());
    } else {
      setVisible (true);
    }
  }

  // s_selectionMarker will always be a pointer to a
  // SelectionInvisibilityMarker, but for code reuse we give it a less
  // specific type.
  OL<InvisibilityMarker> s_selectionMarker;
  OL<InvisibilityMarker> s_invisibilityMarker;
  void newSelectionMarkers (SceneGraph *sg) {
    assert (sg);
    if (!s_selectionMarker.isValid ()) {
      s_selectionMarker =
	OL<InvisibilityMarker> (sg, NEW (SelectionInvisibilityMarker ()), 0);
      assert (s_selectionMarker.isValid ());
    }
    if (!s_invisibilityMarker.isValid ()) {
      s_invisibilityMarker =
	OL<InvisibilityMarker> (sg, NEW (InvisibilityMarker ()), 0);
      assert (s_invisibilityMarker.isValid ());
    }
    s_invisibilityMarker->setVisible (false);
  }

  void getList (Dynavec <int> &selection, OL<InvisibilityMarker> &mark) {
    selection.extendTo (0);
    if (mark.isValid ()) {
      mark.getSceneGraph ()->validate ();
      int count;
      const int *links;
      mark.getLinks (count, links);
      for (int i = 0; i < count; i++) {
	const int thisLink = links [i];
	if (-1 != thisLink) {
	  selection.push (thisLink);
	}
      }
    } else {
      // Selection is empty, so we're done.
    }
  }

  // A set of objects, used to eliminate duplicates from the selection and
  // invisibility lists.
  Dynavec <bool> &s_objectSet = *(NEW (Dynavec <bool> ()));
  Dynavec <int> &s_tmpList = *(NEW (Dynavec <int> ()));
  void objectSetDeallocator () {
    delete &s_objectSet;
    delete &s_tmpList;
  }
  bool objectSetUseless =
  (MemoryUtil::registerDeallocator (objectSetDeallocator), true);

  void setList (SceneGraph *sg,
		const Dynavec <int> &selection, OL<InvisibilityMarker> &mark)
  {
    assert (sg);
    if (!mark.isValid ()) {
      newSelectionMarkers (sg);
    } else {
      assert (sg == mark.getSceneGraph ());
    }
    assert (mark.isValid ());
    s_objectSet.extendTo (sg->maxIndex(), false);
#ifndef NDEBUG
    for (int i = 0; i < s_objectSet.size(); i++) {
      assert (!s_objectSet [i]);
    }
#endif
    s_tmpList.extendTo (0);
    for (int i = 0; i < selection.size(); i++) {
      const int s = selection [i];
      if (s != -1 && !s_objectSet [s]) {
	s_tmpList.push (s);
	s_objectSet [s] = true;
      }
    }
    mark.setLinks (s_tmpList);
    // Clear out the elements of s_objectSet that we used, so we can use them
    // again next time.
    for (int i = 0; i < s_tmpList.size(); i++) {
      s_objectSet [s_tmpList[i]] = false;
    }
  }
};

void SelectionManager::getSelection (Dynavec <int> &selection) {
  getList (selection, s_selectionMarker);
}

void SelectionManager::getSelectionLinks (Dynavec <ObjectLink> &selection) {
  Dynavec <int> selectionInt;
  getSelection (selectionInt);
  selection.extendTo (0);
  for (int i = 0; i < selectionInt.size(); i++) {
    selection.push (ObjectLink (s_selectionMarker.getSceneGraph(),
				selectionInt [i]));
  }
}

void SelectionManager::setSelection (SceneGraph *sg,
				     const Dynavec <int> &selection)
{
  setList (sg, selection, s_selectionMarker);
}

void SelectionManager::setSelectionLinks
(SceneGraph *sg, const Dynavec <ObjectLink> &selection)
{
  Dynavec <int> selectionInt;
  for (int i = 0; i < selection.size(); i++) {
    selectionInt.push (selection [i].getIndex ());
  }
  setSelection (sg, selectionInt);
}

void SelectionManager::setSelectionBlink (bool blink) {
  s_doBlink = blink;
}

bool SelectionManager::getSelectionBlink () {
  return s_doBlink;
}

void SelectionManager::getInvisible (Dynavec <int> &selection) {
  getList (selection, s_invisibilityMarker);
}

namespace {
  // I thought for a moment that I could have a simple Dynavec <bool> here, and
  // then call the destructor explicitly in the deallocator function to avoid
  // the memory leak report.  But then the destructor would get called again
  // later as the program was exiting, and destructing things twice is bad.  (I
  // could fudge the Dynavec destructor so it can deal with being called twice,
  // but that's ugly.)
  Dynavec <bool> *s_visible = NEW (Dynavec <bool> ());
  void deallocator () {
    delete s_visible;
    s_visible = 0;
  }
  int s_gcCount = -1;
  bool useless = (MemoryUtil::registerDeallocator (deallocator),
		  true);
}

void SelectionManager::setInvisible (SceneGraph *sg,
				     const Dynavec <int> &selection)
{
  setList (sg, selection, s_invisibilityMarker);
  // Force recomputation of *s_visible.
  s_gcCount = -1;
}

bool SelectionManager::isSelectable (const ObjectLink &ol)
{
  if (!ol.isValid() || !ol->isSelectable() || !ol.isVisible()) return false;
  if (!s_invisibilityMarker.isValid()) return true;
  if (SceneGraph::garbageCollectionCount () != s_gcCount) {
    s_gcCount = SceneGraph::garbageCollectionCount ();
    SP<SceneGraph> sg = ol.getSceneGraph ();
    const int maxIndex = sg->maxIndex ();
    (*s_visible).extendTo (0);
    (*s_visible).extendTo (maxIndex, true);
    int count;
    const int *links;
    s_invisibilityMarker.getLinks (count, links);
    for (int i = 0; i < count; i++) {
      const int l = links [i];
      if (-1 != l) (*s_visible) [l] = false;
    }
  }
  // Objects added to the end of the scene graph must be visible.
  if (ol.getIndex () >= (*s_visible).size()) {
    return true;
  } else {
    return (*s_visible)[ol.getIndex ()];
  }
}

void SelectionManager::transformSelection (const Matrix4 &m,
					   const Quaternion &q)
{
  if (s_selectionMarker.isValid ()) {
    Dynavec <int> selection;
    getSelection (selection);
    SP<SceneGraph> sg = s_selectionMarker.getSceneGraph ();
    for (int i = 0; i < selection.size(); i++) {
      if (-1 != selection[i]) {
	ObjectLink ol (sg, selection[i]);
	Vec3 v1 = ol.boundingSphereCenter();
	Vec3 v2 = m * v1;
	ol.translate (v2 - v1);
	ol.rotate (q);
      }
    }
  } else {
    // No selection, so nothing to do.
  }
}
