// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __SceneGraph_h__
#define __SceneGraph_h__

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

class BoundingSphere;
class PhysicsObject;
class Vec3;
class PhysicsObjectInfo;
class Evaluator;
class Canvas;
class LinkManager;

class SceneGraph
  : public Configurable
{
  struct SceneGraphData;
  SceneGraphData *m_data;
  // Next one is deliberately unimplemented.
  SceneGraph (const SceneGraph &sg);
public:
  virtual SP<Evaluator> getEvaluator () = 0;
  virtual SP<const Evaluator> getEvaluator () const = 0;
  // You may specify the starting LinkManager when creating a SceneGraph.  This
  // is done when transplanting the LinkManager from some old SceneGraph.  For
  // the first SceneGraph, use NEW (LinkManager) for the LinkManager.  I am not
  // creating a default constructor that does this since I want to catch places
  // that ought to be passing in a LinkManager but aren't.
  SceneGraph (SP<LinkManager> lm);
  // Destructing a SceneGraph is tricky.  The objects in the SceneGraph are
  // allowed to include within themselves ObjectLinks that point to other
  // objects in the SceneGraph.  CursorBall's are an example, movies are
  // another.  Such an object is likely to want to delete the objects it points
  // to when the object itself is deleted.  This would cause problems if this
  // happens when the SceneGraph is being deleted, since the SceneGraph has
  // been partially destructed at that point and isn't in any position to cope
  // with objects being deleted from it.  Thus, the destructor for a SceneGraph
  // subclass should use the LinkManager::updateAfterGC method to map all links
  // to -1; this will cause the ObjectLink's embedded within a PhysicsObject to
  // be invalid, which for a robust object will cause it not to try to delete
  // the objects pointed to by the links.  When a SceneGraph is copied, the new
  // scene graph should take the linkmanager from the old one and set the old
  // one's link manager to null, so the destructor shouldn't call
  // LinkManager::updateAfterGC if the linkmanager is null.
  //
  // The outcome of this hackery is that if an object observes a ObjectLink
  // that is valid, it can always manipulate the object that the link points
  // to.
  //
  // When an object is being destructed, it doesn't know that the SceneGraph
  // still exists, and it does know that it is not referenced by the
  // SceneGraph.  If it wants to manipulate other objects at this time, it must
  // use ObjectLink's to find them, and be graceful if those ObjectLink's have
  // been marked invalid.  It isn't wise to retain a pointer to the SceneGraph
  // because it may be destructed by then, if the pointer is not
  // reference-counted, or you're creating a cycle if the pointer is
  // reference-counted.  It isn't wise to retain a pointer to the LinkManager
  // because theoretically a new LinkManager could have been created that makes
  // the old one irrelevant.  The current implementation never does this, but 
  // I don't want to commit to that.
  //
  // See SubsetStorageScene for an example of implementing the destructor, and
  // see CursorBall or PlayBrennerMovie for exmples of deleting PhysicsObjects
  // from a PhysicsObject destructor without getting into trouble.
  virtual ~SceneGraph ();
  // Maximum index.  The objects have indices between 0 and
  // maxIndex-1.  If object (the index) is null, then the index is the
  // index of a deleted object and should not be used. 
  virtual int maxIndex () const = 0;
  // The index into the state vector for the object at the given index.
  virtual int objectStateIndex (int index) const = 0;
  // Beware -- once you call addObject, the object state may be overwritten
  // because the dynavec can get extended.  So don't use the result from
  // objectState as an argument to addObject; instead, copy it into a vector
  // first.
  Float *objectState (int index);
  const Float *objectState (int index) const;
  // The PhysicsObject for the object at the given index.
  virtual const PhysicsObject *object (int index) const = 0;
  // Add the objects on the line to the indices and intersectionPoints vectors.
  // Beware that if the vectors are nonempty, their old contents will
  // still be there when this returns.
  virtual void objectsOnLine (const Vec3 &v1, const Vec3 &v2,
			      Dynavec <int> &indices,
			      Dynavec <Vec3> &intersectionPoints)
    const = 0;
  // The sphere bounding the scene.
  virtual BoundingSphere graphicsBoundingSphere () const = 0;
  // The initialState is copied when we add an object, so the initial
  // state can be reused if you like.
  // If the initial position is out of bounds, the object may
  // immedately be deleted, or it may be deleted when we try later to
  // run the evaluator.
  // The index for the new object is returned, or -1 is returned if the object
  // could not be added because it was out of bounds.
  // 
  // If the object has a stateSize of zero, and it's methods that take a state
  // don't mind getting a null pointer, then it's okay to pass in a null
  // state pointer.
  virtual int addObject (const PhysicsObject *obj,
			  const Float *initialState) = 0;
  int addObject (const PhysicsObject *obj, Dynavec <Float> &state);
  virtual void deleteObject (int index) = 0;
  // Returns true if we had to delete some objects during drawing.
  virtual bool draw (SP<Canvas> vo) = 0;
  // Tell how many times we have made changes to the scene graph that
  // would invalidate information cached about each object index.
  // This count starts at zero, so if you want the initial count to be
  // different, a value of -1 is good.  We have to increment
  // continuously even if someone makes a new SceneGraph, so this is static.
  static int garbageCollectionCount ();
  // Close up gaps so that there are no deleted objects in the scene graph, and
  // the scene graph is sorted (if it wants to maintain itself sorted).  
  // The goals here are:
  // 1. free most code from having to worry about encountering deleted objects,
  // 2. allow a bunch of objects to be deleted at once fairly efficiently,
  // 3. also allow a garbage collection procedure that takes a few
  // milliseconds without impacting interactivity in the normal case when
  // objects are not being added or deleted.
  // 4. Object numbers should stay stable while you're deleting, so you can
  // delete a bunch of objects without getting confused.
  // 
  // This is called by:
  // 1. the evaluator before beginning work
  // 2. draw before beginning work (draw may also delete things that are out of
  // bounds, so draw may renumber objects at any time).
  // 3. It can be called by any other code that wants to make sure that all
  // deleted objects are gone from the SceneGraph.
  // 
  // Validate may renumber the objects if any objects have been added or
  // deleted since the previous time validate was called.  If you have deleted
  // objects, and validate hasn't been called since you deleted the objects,
  // then the object (index) method will return null for the indices of the
  // deleted objects.
  virtual void validate () = 0;
  SP<const LinkManager> getLinkManager () const;
  SP<LinkManager> getLinkManager ();
  // The TopLevel should call this once per timestep, before doing the
  // timestep.
  virtual void animateTimeStep () = 0;
  // The TopLevel should call this once per frame, before drawing the frame.
  virtual void animateFrame (Canvas *canvas) = 0;
  // Whether there are any live animations going right now.  If there aren't,
  // then perhaps we'll relinquish the CPU waiting for user input.
  virtual bool anyAnimations () const = 0;
protected:
  // Increment the garbageCollectionCount, above.
  static void didGarbageCollection ();
  // If we've transplanted the LinkManager to another SceneGraph, then we have
  // to notify this SceneGraph so it doesn't try to manipulate the LinkManager
  // as this SceneGraph is being destructed.
  void clearLinkManager (SceneGraph *oldSG);
public:
  PhysicsObjectInfo poi (int index) const;
  // Return the center of the selection (this would be the center of gravity of
  // the selection, if each object had equal mass) except if there isn't
  // anything selected, return junk and set error to true.
  Vec3 centerOfSelection (bool &error) const;
  // Similarly, return the center of the scene and set error if there is not
  // hing in the scene.
  Vec3 centerOfScene (bool &error) const;
  // First try centerOfSelection; if nothing is selected then try
  // centerOfScene; if there are no objects in the scene then return 0,0,0; but
  // in no case return an error.
  Vec3 centerOfSomething () const;
  // Convenience method.
  Vec3 boundingSphereCenter (const int index) const;
};

#endif
