// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "SceneGraph.h"

#ifndef __Evaluator_h__
#include "Evaluator.h"
#endif

#ifndef __PhysicsObjectInfo_h__
#include "PhysicsObjectInfo.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __PhysicsObject_h__
#include "PhysicsObject.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

PhysicsObjectInfo SceneGraph::poi (int index) const {
  return PhysicsObjectInfo (object(index), objectStateIndex(index));
}

Float *SceneGraph::objectState (int index) {
  // We can have an object with state size 0 at the end of stateVec, so have to
  // skip an implicit Dyanvec bounds check here.
  assert (0 == object(index)->stateSize() ||
	  &(getEvaluator()->stateVec ()[objectStateIndex (index)]));
  return (&*(getEvaluator()->stateVec ())) + objectStateIndex (index);
};

const Float *SceneGraph::objectState (int index) const {
  return &(getEvaluator ()->stateVec ()[objectStateIndex (index)]);
};

int SceneGraph::addObject (const PhysicsObject *obj, Dynavec <Float> &state) {
  return addObject (obj, &*state);
}

namespace {
  int s_garbageCollections = 0;
}

int SceneGraph::garbageCollectionCount () {
  return s_garbageCollections;
}

void SceneGraph::didGarbageCollection () {
  s_garbageCollections++;
}

struct SceneGraph::SceneGraphData {
  SP<LinkManager> m_lm;
};

SceneGraph::SceneGraph (SP<LinkManager> lm)
  : m_data (NEW (SceneGraphData ()))
{
  m_data->m_lm = lm;
  didGarbageCollection ();
}

SceneGraph::~SceneGraph ()
{
  assert (m_data);
  delete m_data;
  m_data = 0;
}

SP<const LinkManager> SceneGraph::getLinkManager () const {
  return &*m_data->m_lm;
}

void SceneGraph::clearLinkManager (SceneGraph *oldSG) {
  oldSG->m_data->m_lm = 0;
}

SP<LinkManager> SceneGraph::getLinkManager () {
  return m_data->m_lm;
}

Vec3 SceneGraph::centerOfSelection (bool &error) const {
  Dynavec <int> selection;
  int objects = 0;
  Vec3 center (0,0,0);
  SelectionManager::getSelection (selection);
  for (int i = 0; i < selection.size(); i++) {
    const int s = selection [i];
    SP<const PhysicsObject> po = object (s);
    center += po->boundingSphereCenter (objectState (s));
    objects++;
  }
  if (0 == objects) {
    error = true;
    return Vec3 (0,0,0);
  } else {
    return center / objects;
  }
}

Vec3 SceneGraph::centerOfScene (bool &error) const {
  int objects = 0;
  Vec3 center (0,0,0);
  for (int i = 0; i < maxIndex(); i++) {
    SP<const PhysicsObject> po = object (i);
    // Note that we're including invisible objects here.
    if (po && po->isSelectable ()) {
      center += po->boundingSphereCenter (objectState (i));
      objects++;
    }
  }
  if (0 == objects) {
    error = true;
    return Vec3 (0,0,0);
  } else {
    return center / objects;
  }
}

Vec3 SceneGraph::centerOfSomething () const {
  bool error = false;
  Vec3 result = centerOfSelection (error);
  if (error) {
    error = false;
    result = centerOfScene (error);
    if (error) {
      result = Vec3 (0,0,0);
    }
  }
  return result;
}

Vec3 SceneGraph::boundingSphereCenter (const int index) const {
  return object (index)->boundingSphereCenter(objectState (index));
}
