// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

// Sample from the document at
// http://www.rcsb.org/pdb/docs/format/pdbguide2.2/guide2.2_frame.html:
//          1         2         3         4         5         6         7         8
// 12345678901234567890123456789012345678901234567890123456789012345678901234567890
// HETATM 1357 MG    MG   168       4.669  34.118  19.123  1.00  3.16          MG2+
// HETATM 3835 FE   HEM     1      17.140   3.115  15.066  1.00 14.14          FE3+
//
// Columns 55-60 are the occupancy.  I don't know what that means; I'll use
// 1.
//
// Colums 61-66 are the temperature factor.  I don't know that means; I'll use
// 0.
// 
// A line from diffGear.pdb.
// HETATM 2131 SI   NONE    1       3.867   0.841 -52.781  1.00  0.00
// 
// What I want to produce
// HETATM 2131 SI   XXX     1       3.867   0.841 -52.781  1.00  0.00          SI
// 
// Column 21 from diffGear.pdb has an "E" and is not described in the standard.
// I will have "XXX " instead in columns 18-21 so there's a more normal
// character " " in column 21.  I'll also put the element name at column 77.
// Note that the element name is supposed to be right justified in columns
// 77-78, as well as in columns 13-14.

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __SceneWriterConfiguration_h__
#include "SceneWriterConfiguration.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Ball_h__
#include "Ball.h"
#endif

#ifndef __fstream_h__
#include <fstream.h>
#define __fstream_h__
#endif

// For strerror.
#ifndef  __string_h__
#include <string.h>
#define __string_h__
#endif

// For errno.
#ifndef __errno_h__
#include <errno.h>
#define __errno_h__
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __BoringAtom_h__
#include "BoringAtom.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __AtomInfo_h__
#include "AtomInfo.h"
#endif

namespace {
  class PDBSceneWriter
    : public TypedFactory <SceneWriterConfiguration, Action>
  {
  public:
    PDBSceneWriter()
      : TypedFactory <SceneWriterConfiguration, Action> ("PDBSceneWriter")
    {}
    SP<SceneWriterConfiguration> typedDefaultConfiguration () const {
      return NEW (SceneWriterConfiguration ());
    }
    SP<Action> makeIt (SceneWriterConfiguration *conf) const {
      SP<Action> result = NEW (Action ());
      SP<SceneGraph> sg = conf->getTopLevel()->getSceneGraph ();
      // The object indices we're going to write out, in order by serial
      // number.  We use zero-based addressing for serialOrder but we write out
      // serial numbers starting with one, so element zero of this array will
      // be the atom with serial number 1.
      Dynavec <int> serialOrder;
      if (conf->getSelectionOnly ()) {
	SelectionManager::getSelection (serialOrder);
      } else {
	for (int i = 0; i < sg->maxIndex(); i++) {
	  serialOrder.push (i);
	}
      }
      // Filter out everything but the atoms.
      {
	int to = 0;
	for (int from = 0; from < serialOrder.size(); from++) {
	  const int s = serialOrder [from];
	  if (sg->object(s)->isInstanceOf (BoringAtom::staticClassId)) {
	    serialOrder [to] = s;
	    to++;
	  }
	}
	serialOrder.extendTo (to);
      }
      ofstream out (&*(conf->getFileName()));
      if (!out.is_open()) {
	result->setProblem (String ("Failed to open output file ")+
			    conf->getFileName()+": "+strerror (errno));
      }
      String line;
      String error = "";
      if (!result->isProblem ()) {
	line = "HETATM 2131 SI   XXX     1       "
	  "3.867   0.841 -52.781  1.00  0.00          SI";
	for (int i = 0; i < serialOrder.size() && !error.size(); i++) {
	  line.fromInt (i+1, 6, 5, &error);
	  const int objNo = serialOrder [i];
	  SP<const BoringAtom> ba =
	    dynamic_cast <const BoringAtom *> (&*sg->object (objNo));
	  assert (ba);
	  // Our name is the element symbol.  diffGear.pdb has two digits in
	  // the name after the names; the two digits are the first two digits
	  // of the serial number.  This doesn't seem useful to me, since most
	  // of the serial numbers are more than two digits.
	  line.fromStringRight
	    (AtomInfo::toSymbol (ba->getSymbolNumber()), 12, 2, &error);
	  Vec3 where = Ball::center (sg->objectState (objNo));
	  line.fromFloat (where[0], 30, 8, 3, &error);
	  line.fromFloat (where[1], 38, 8, 3, &error);
	  line.fromFloat (where[2], 46, 8, 3, &error);
	  line.fromStringRight (AtomInfo::toSymbol (ba->getSymbolNumber()),
				76, 2, &error);
	  if (error.size()) {
	    result->setProblem (String ("Could not format line for atom ")+
				AtomInfo::toSymbol (ba->getSymbolNumber())+
				" with serial number "+(i+1)+
				" at "+where+"\n"+error);
	  }
	  // Put the line out there even if there was an error.  Might give
	  // them some hints when they're trying to figure out why.
	  out << line << endl;
	  if (out.fail()) {
	    result->setProblem ("I/O error on "+conf->getFileName()+
				": "+strerror (errno));
	  }
	}
      }
      if (!result->isProblem ()) {
	line = "TER    8298      XXX     1 ";
	line.fromInt (serialOrder.size()+1, 6, 5, &error);
	if (error.size()) {
	  result->setProblem ("Could not put serial number into "
			      "TER Line: "+error);
	}
	out << line << endl;
	if (out.fail()) {
	  result->setProblem ("I/O error on "+conf->getFileName()+
			      ": "+strerror (errno));
	}
      }
      if (!result->isProblem()) {
	// A mapping from object indices to serial numbers.
	Dynavec <int> objNoToSerial;
	objNoToSerial.extendTo (sg->maxIndex(), -1);
	for (int i = 0; i < serialOrder.size(); i++) {
	  objNoToSerial [serialOrder [i]] = i;
	}
	// Now generate CONECT lines.
	// connections is out of the loop only to avoid repeated
	// allocation.  It's a list of the serial numbers of the connections,
	// with the 1's already added.
	Dynavec <int> connections;
	for (int i = 0; i < serialOrder.size() && !error.size(); i++) {
	  int count;
	  const int *links;
	  sg->getLinkManager()->getLinks (serialOrder [i], count, links);
	  connections.extendTo (0);
	  for (int j = 0; j < count; j++) {
	    if (-1 != links [j]) {
	      const int serial = objNoToSerial [links[j]];
	      if (-1 != serial) {
		connections.push (serial + 1);
	      }
	    }
	  }
	  int j = 0;
	  while (j < connections.size() && !error.size()) {
	    line = "CONECT";
	    line.fromInt (i+1, 6, 5, &error);
	    line.fromInt (connections [j], 11, 5, &error);
	    j++;
	    if (j < connections.size()) {
	      line.fromInt (connections [j], 16, 5, &error);
	      j++;
	    }
	    if (j < connections.size()) {
	      line.fromInt (connections [j], 21, 5, &error);
	      j++;
	    }
	    if (j < connections.size()) {
	      line.fromInt (connections [j], 26, 5, &error);
	      j++;
	    }
	    if (error.size()) {
	      result->setProblem (String("Could not format CONECT line for "
					 "atom with serial number ")+(i+1)
				  +": "+error);
	    }
	    out << line << endl;
	    if (out.fail()) {
	      result->setProblem ("I/O error on "+conf->getFileName()+
				  ": "+strerror (errno));
	    }
	  }
	}
      }
      if (!result->isProblem ()) {
	// Want the file to be obviously incomplete if an error occurred while
	// generating it. 
	out << "END" << endl;
	if (out.fail()) {
	  result->setProblem ("I/O error on "+conf->getFileName()+
			      ": "+strerror (errno));
	}
      }
      out.close();
      if (!result->isProblem ()) {
	// Put this check after the close because disk full errors are commonly
	// reported during the close. 
	if (out.fail()) {
	  result->setProblem ("I/O error while closing "+conf->getFileName()+
			      ": "+strerror (errno));
	}
      }
      return result;
    }
  };
  static const bool useless =
  (FactoryTable::store ("SceneWriter", NEW (PDBSceneWriter ())),
   FactoryTable::registerSubclass ("SceneWriter", "Action"),
   true);
}
