// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Float_h__
#include "Float.h"
#endif

// FIXME Do I  want to generalize this to 8-to-the-n-trees?
// FIXME Improve performance by doing only one computed goto per level

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

class BoundingBox;

#ifndef __BoundingBoxInt_h__
#include "BoundingBoxInt.h"
#endif

#ifndef __Storage_h__
#include "Storage.h"
#endif

class OctTreeImpl : public Storage {
  struct OctTreeNode {
#ifndef NDEBUG
    // If NDEBUG is defined, as we search the tree, we check that the
    // box for each node is equal to what we expect it to be.
    BoundingBoxInt box;
#endif
    Dynavec <int> indices;
    OctTreeNode * subnodes [2] [2] [2];
    public:
    void clearLinks ();
#ifndef NDEBUG
    static OctTreeNode *theGarbageLink ();
    static void checkGarbage (const OctTreeNode *o);
    void garbageLinks ();
    OctTreeNode () {
      garbageLinks ();
    }
#endif
  };
  // The OctTreeImpl is centered at (0,0,0) world coordinates.
  // Minimum size of an OctTreeImpl node.
  const Float m_minSize;
  // The size of the entire OctTreeImpl, measured in
  // units of minSize.  Must be a power of 2.
  int m_outerSize;
  OctTreeNode *m_theNode;
  // A list of OctTreeNodes that we might reuse someday.
  Dynavec <OctTreeNode *> m_freeList;
  // The things we allocated, so we can deallocate later.
  Dynavec <OctTreeNode *> m_blockList;
  // Allocate this many OctTreeNode's at once.
  static const int blockSize = 100;
  // Minimum coordinate represented, in x, y, and z.
  int minExtent () const;
  // Allocate a new node.  
  OctTreeNode *newNode ();
  void freeNode (OctTreeNode *);
  // Make the root bigger.
  void growTheRoot ();
  // Don't allow assignment or copy.
  OctTreeImpl (const OctTreeImpl &); // unimplemented
  OctTreeImpl &operator= (const OctTreeImpl &); // unimplemented
  void storeRecur (const BoundingBoxInt &box,
		   const int boxSize,
		   const int x, const int y, const int z,
		   OctTreeNode *&node,
		   int logSize,
		   int value);
  void emptyRecur (OctTreeNode *node, int outerSize);
  // Only OctTreeStorageFactory can call the constructor.
  friend class OctTreeStorageFactory;
  OctTreeImpl (Float minSize);
 public:
  ~OctTreeImpl ();
  bool store (int value, const BoundingBox &box);
  // The returned value list should not have any duplicates.
  // Any old stuff in the list is emptied out.
  void fetch (const BoundingBox &box, int notme, Dynavec <int> &result) const;
  // Empty out the OctTreeImpl, returning all nodes to m_freeList.
  void empty ();
  // Maybe include functions for tracing rays through the OctTreeImpl,
  // or enumerating objects that overlap a half-space, or computing a
  // global bounding box or bounding sphere.
  // FIXME Need better unbounded object support in PhysicsObject.h protocol
  // Problem is that a half-plane-like object has no way to query the
  // OctTreeImpl for all objects in the relevant half-plane.
  // FIXME because we don't use this code in the release anyway.
};

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

#ifndef __BoundingBoxInt_h__
#include "BoundingBoxInt.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Vec3i_h__
#include "Vec3i.h"
#endif

typedef OctTreeImpl::OctTreeNode OctTreeNode;

void OctTreeNode::clearLinks () {
  subnodes [0] [0] [0] = 0;
  subnodes [0] [0] [1] = 0;
  subnodes [0] [1] [0] = 0;
  subnodes [0] [1] [1] = 0;
  subnodes [1] [0] [0] = 0;
  subnodes [1] [0] [1] = 0;
  subnodes [1] [1] [0] = 0;
  subnodes [1] [1] [1] = 0;
}

#ifndef NDEBUG
OctTreeNode *OctTreeNode::theGarbageLink () {
  return (OctTreeNode *) -1;
}

void OctTreeNode::checkGarbage (const OctTreeNode *o) {
  assert (theGarbageLink () != o);
}

void OctTreeNode::garbageLinks () {
  subnodes [0] [0] [0] = theGarbageLink ();
  subnodes [0] [0] [1] = theGarbageLink ();
  subnodes [0] [1] [0] = theGarbageLink ();
  subnodes [0] [1] [1] = theGarbageLink ();
  subnodes [1] [0] [0] = theGarbageLink ();
  subnodes [1] [0] [1] = theGarbageLink ();
  subnodes [1] [1] [0] = theGarbageLink ();
  subnodes [1] [1] [1] = theGarbageLink ();
}

#endif

OctTreeImpl::OctTreeImpl (Float minSize) : m_minSize (minSize), m_theNode (0) {
  empty ();
}

OctTreeImpl::~OctTreeImpl () {
  m_theNode = 0;
  for (int i = 0; i < m_blockList.size(); i++) {
    assert (m_blockList [i]);
    delete [] m_blockList [i];
    m_blockList [i] = 0;
  }
}

OctTreeNode* OctTreeImpl::newNode () {
  OctTreeNode *result;
  if (!m_freeList.size()) {
    OctTreeNode *block = NEW (OctTreeNode [blockSize]);
    m_blockList.push (block);
    for (int i = 0; i < blockSize; i++) {
      m_freeList.push (&block [i]);
    }
  }
  assert (m_freeList.size());
  result = m_freeList.pop();
  result->indices.extendTo (0);
#ifndef NDEBUG
  result->garbageLinks ();
#endif
  return result;
}

static void fetchRecur (const BoundingBoxInt &toFetch,
			const int x, const int y, const int z,
			const OctTreeNode *theNode,
			const int size,
			Dynavec <int> &result) {
  assert (size >= 1);
#ifndef NDEBUG
  OctTreeNode::checkGarbage (theNode);
#endif
  if (theNode) {
    const BoundingBoxInt outerBox = BoundingBoxInt (Vec3i (x, y, z),
						    size);
    assert (outerBox == theNode->box);
    if (outerBox.overlaps (toFetch)) {
      {
	const int isize = theNode->indices.size ();
	for (int i = 0; i < isize; i++) {
	  result.push (theNode->indices[i]);
	}
      }
      // Most nodes are leaves.  Avoid looking at all the nulls on a
      // leaf.  
      if (size > 1) {
	const int subSize = size >> 1;
	const int xn = x + subSize;
	const int yn = y + subSize;
	const int zn = z + subSize;
	// FIXME could do better for small objects here.
	// When the object fits entirely within a subnode, we could do
	// three comparisons here instead of 8, at the cost of bigger
	// code and maybe some debugging time.  Use a loop if I do
	// this, since recursion would have a redundant call to "overlaps"
	// at the top of findRecur.
	//
	// This is a special case of the trick of improving performance by
	// doing only one computed goto per level.
	fetchRecur (toFetch, x, y, z, theNode->subnodes [0] [0] [0],
		    subSize, result);
	fetchRecur (toFetch, xn, y, z, theNode->subnodes [1] [0] [0],
		    subSize, result);
	fetchRecur (toFetch, x, yn, z, theNode->subnodes [0] [1] [0],
		    subSize, result);
	fetchRecur (toFetch, xn, yn, z, theNode->subnodes [1] [1] [0],
		    subSize, result);
	fetchRecur (toFetch, x, y, zn, theNode->subnodes [0] [0] [1],
		    subSize, result);
	fetchRecur (toFetch, xn, y, zn, theNode->subnodes [1] [0] [1],
		    subSize, result);
	fetchRecur (toFetch, x, yn, zn, theNode->subnodes [0] [1] [1],
		    subSize, result);
	fetchRecur (toFetch, xn, yn, zn, theNode->subnodes [1] [1] [1],
		    subSize, result);
      }
    }
  }
}

int OctTreeImpl::minExtent () const {
  // Dividing by 2 afterwards, instead of subtracting 1 from the log,
  // does something reasonable in the special case when logOuterSize
  // is 0.
  return - (m_outerSize >> 1);
}

void OctTreeImpl::fetch (const BoundingBox &box,
			 const int notme,
			 Dynavec <int> &result) const {
  assert (0 < m_minSize);
  const BoundingBoxInt intBox (box, m_minSize);
  Dynavec <int> resultWithDuplicates;
  fetchRecur (intBox, minExtent (), minExtent (), minExtent (),
	      m_theNode, m_outerSize, resultWithDuplicates);
  result.extendTo (0);
  // Copy resultWithDuplicates to result, removing the duplicates.
  {
    int rdSize = resultWithDuplicates.size();
    for (int i = 0; i < rdSize; i++) {
      const int thisResult = resultWithDuplicates [i];
      if (thisResult != notme) {
	bool isNew = true;
	const int rSize = result.size ();
	for (int j = 0; j < rSize; j++) {
	  if (result [j] == thisResult) {
	    isNew = false;
	    break;
	  }
	}
	if (isNew) {
	  result.push (thisResult);
	}
      }
    }
  }
}

void OctTreeImpl::freeNode (OctTreeNode *n) {
  m_freeList.push (n);
#ifndef NDEBUG
  n->garbageLinks();
#endif
}

// Double the extent of the root of the tree, multiplying its volume
// by 8.
void OctTreeImpl::growTheRoot () {
  // Push all of the root's indices into its subnodes, since the old
  // root is going to go away, and we have to keep those indices
  // somewhere.
  // For this to work, we have to ensure that each subnode of the root
  // exists.  Hair, hair, hair.
  // I could push the hair somewhere else by stipulating that the root
  // node always has all of its immediate subnodes.  However, that doesn't
  // decrease the code bulk, and it increases the number of invariants
  // that the data structure has to obey to stay correct, so it
  // increases the total hair.
  {
    for (int i = 0; i < 2; i++) {
      for (int j = 0; j < 2; j++) {
	for (int k = 0; k < 2; k++) {
	  OctTreeNode * &theSubnode = m_theNode->subnodes [i][j][k];
	  if (! theSubnode) {
	    theSubnode = newNode();
#ifndef NDEBUG
	    const int newSize = m_outerSize >> 1;
	    int m = minExtent ();
	    theSubnode->box =
	      BoundingBoxInt (Vec3i (m+newSize*i, m+newSize*j, m+newSize*k),
			      newSize);
#endif
	  }
	  for (int n = 0; n < m_theNode->indices.size(); n++) {
	    assert (theSubnode);
	    theSubnode->indices.push (m_theNode->indices[n]);
	  }
	}
      }
    }
  }
  OctTreeNode *newRoot = newNode();
  m_outerSize = m_outerSize << 1;
  assert (m_outerSize > 1);
#ifndef NDEBUG
  {
    int m = minExtent ();
    newRoot->box = BoundingBoxInt (Vec3i (m, m, m), m_outerSize);
  }
#endif
  {
    for (int i = 0; i < 2; i++) {
      for (int j = 0; j < 2; j++) {
	for (int k = 0; k < 2; k++) {
	  OctTreeNode *theNew = newNode ();
	  theNew->clearLinks ();
	  newRoot->subnodes [i] [j] [k] = theNew;
	  theNew->subnodes [1-i] [1-j] [1-k] =
	    m_theNode->subnodes [i] [j] [k];
#ifndef NDEBUG
	  {
	    const int newSize = m_outerSize >> 1;
	    int m = minExtent ();
	    theNew->box =
	      BoundingBoxInt (Vec3i (m+newSize*i, m+newSize*j, m+newSize*k),
			      newSize);
	  }
#endif
	}
      }
    }
  }
  freeNode (m_theNode);
  m_theNode = newRoot;
}

void OctTreeImpl::storeRecur (// The region that value should be associated with.
			  const BoundingBoxInt &box,
			  // The size of the region, so we don't have to
			  // recompute it.
			  const int boxSize,
			  // Coordinates of the region for this node.
			  const int x, const int y, const int z,
			  // The node we are contemplating storing into.
			  // Pass a reference so that we can fill it in if it's
			  // null.
			  OctTreeNode *&node,
			  // Log of the size of the bounding box for this node.
			  int size,
			  // The value to store.
			  int value)
{
#ifndef NDEBUG
  OctTreeNode::checkGarbage (node);
#endif
  assert (size >= 1);
  const BoundingBoxInt thisBox = BoundingBoxInt (Vec3i (x, y, z), size);
  if (thisBox.overlaps (box)) {
    if (! node) {
      node = newNode ();
      if (1 < size) {
	node->clearLinks();
      }
#ifndef NDEBUG
      node->box = thisBox;
#endif
    }
    assert (node->box == thisBox);
    if ((size == 1) ||
	// The rationale for the next one is given in the comments in
	// OctTreeNode.h.
	(boxSize > (size >> 1))) {
      node->indices.push (value);
    } else {
      const int subSize = size >> 1;
      const int xn = x + subSize;
      const int yn = y + subSize;
      const int zn = z + subSize;
      storeRecur (box, boxSize, x, y, z, node->subnodes [0] [0] [0],
		  subSize, value);
      storeRecur (box, boxSize, xn, y, z, node->subnodes [1] [0] [0],
		  subSize, value);
      storeRecur (box, boxSize, x, yn, z, node->subnodes [0] [1] [0],
		  subSize, value);
      storeRecur (box, boxSize, xn, yn, z, node->subnodes [1] [1] [0],
		  subSize, value);
      storeRecur (box, boxSize, x, y, zn, node->subnodes [0] [0] [1],
		  subSize, value);
      storeRecur (box, boxSize, xn, y, zn, node->subnodes [1] [0] [1],
		  subSize, value);
      storeRecur (box, boxSize, x, yn, zn, node->subnodes [0] [1] [1],
		  subSize, value);
      storeRecur (box, boxSize, xn, yn, zn, node->subnodes [1] [1] [1],
		  subSize, value);
    }
  }
}

bool OctTreeImpl::store (int value, const BoundingBox &box) {
  assert (0 < m_minSize);
  const BoundingBoxInt intBox (box, m_minSize);
  for (;;) {
    const int i = minExtent();
    if (BoundingBoxInt(Vec3i (i, i, i), m_outerSize)
	.contains (intBox)) break;
    growTheRoot ();
  }
  // Now the root node contains the given box.
  assert (m_theNode->box.contains (intBox));
  Vec3i intBoxDiagonal = intBox.max () - intBox.min ();
  int intBoxSize = intBoxDiagonal [0];
  for (int i = 1; i < 3; i++) {
    assert (intBoxSize > 0);
    if (intBoxDiagonal [i] > intBoxSize) {
      intBoxSize = intBoxDiagonal [i];
    }
  }
  {
    const int m = minExtent ();
    storeRecur (intBox, intBoxSize, m, m, m, m_theNode, m_outerSize, value);
  }
  return true;
}

void OctTreeImpl::emptyRecur (OctTreeNode *node, const int outerSize) {
  if (node) {
    if (outerSize > 1) {
#ifndef NDEBUG
      OctTreeNode::checkGarbage (node);
#endif
      const int smallerSize = outerSize >> 1;
      emptyRecur (node->subnodes [0] [0] [0], smallerSize);
      emptyRecur (node->subnodes [0] [0] [1], smallerSize);
      emptyRecur (node->subnodes [0] [1] [0], smallerSize);
      emptyRecur (node->subnodes [0] [1] [1], smallerSize);
      emptyRecur (node->subnodes [1] [0] [0], smallerSize);
      emptyRecur (node->subnodes [1] [0] [1], smallerSize);
      emptyRecur (node->subnodes [1] [1] [0], smallerSize);
      emptyRecur (node->subnodes [1] [1] [1], smallerSize);
    }
#ifndef NDEBUG
    node->garbageLinks();
#endif
    freeNode (node);
  }
}

void OctTreeImpl::empty () {
  emptyRecur (m_theNode, m_outerSize);
  m_theNode = newNode ();
  m_theNode->clearLinks();
  m_outerSize = 2;
#ifndef NDEBUG
  m_theNode->box = BoundingBoxInt (Vec3i (-1, -1, -1), 2);
#endif
}

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __OctTreeStorageConfiguration_h__
#include "OctTreeStorageConfiguration.h"
#endif

class OctTreeStorageFactory : public Factory {
public:
  OctTreeStorageFactory () : Factory ("OctTreeStorage") {}
  SP<Configuration> defaultConfiguration () const {
    return NEW (OctTreeStorageConfiguration ());
  }
  SP<Configurable> makeIt (const SP<Configuration> c) const {
    OctTreeStorageConfiguration *conf =
      dynamic_cast <OctTreeStorageConfiguration *> (&*c);
    assert (conf);
    return NEW (OctTreeImpl (conf->getCharacteristicSize()));
  }
};

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

static const bool useless =
(FactoryTable::store ("Storage", NEW (OctTreeStorageFactory ())),
 true);
