// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __ObjectLink_h__
#define __ObjectLink_h__

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __PhysicsObject_h__
#include "PhysicsObject.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __BoundingSphere_h__
#include "BoundingSphere.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

class SceneGraph;
class PhysicsObject;

// An ObjectLink is generated by the makeLink method of LinkManager.  It
// represents a pointer from C++ code to an object in the scene.  It is fairly
// lightweight, only three words, so we use it as freely as a SP.  You'll
// probably want to use the OL<...> template instead of this class; this class
// is a non-template subclass of that one so we can have uniform, separately
// compiled code in LinkManager that applies to all instances of the template.
//
// If objects are deleted and change their numbers, the LinkManager will update
// the ObjectLink's to have appropriate new numbers.
//
// If the object that the ObjectLink points to is deleted, and the SceneGraph
// has had validate called on it since then, then the ObjectLink will be
// marked invalid.  If validate has not been called on the SceneGraph, then the
// ObjectLink will be valid, but if ol is the ObjectLink and sg is the
// SceneGraph then sg->object(ol->getIndex()) will be 0.
//
// The LinkManager retains a pointer to the ObjectLink, but it does not count
// it as a reference (hence no reference count here).  The destructor for
// ObjectLink takes it out of the list that LinkManager maintains.
//
// If the LinkManager is deleted, it will update all of its ObjectLinks to be
// invalid.
//
// If a LinkManager is copied, it will grab all of the ObjectLinks from the
// old link manager and manage them itself.  This is reasonable if we assume
// that the old LinkManager will not be used after the copy is done.
class ObjectLink
{
  // The scene graph we are in.  This is a pointer, not a smart pointer,
  // to avoid reference cycles.
  // We are managed by the LinkManager if and only if m_sg is non-null.
  // Invariants: 
  // This ObjectLink is managed by a LinkManager if and only if it's m_sg field
  // is non-null, and in that case the LinkManager that is managing it is
  // m_sg->getLinkManager().
  // The LinkManager only stops managing an ObjectLink that it is managing when
  // either the ObjectLink is destructed, or the LinkManager is destructed.
  SceneGraph *m_sg;
  // Position of this object in the scene graph.  -1 if we are invalidated.
  int m_objIndex;
  // Position of this link in the LinkManager's list of ManagedLinks.
  int m_linkIndex;

  friend class LinkManager;
  // A ObjectLink knows its position in the vector of ObjectLinks that the
  // LinkManager manages, so we can efficiently delete it when it is time.
  int getObjectLinkIndex () const {
    return m_linkIndex;
  }
  // If the LinkManager is being destructed, it will set the link index to -1.
  // This allows us to survive destructing an ObjectLink after the LinkManager
  // is gone, when the program is exiting.
  void setObjectLinkIndex (int newIndex) {
    m_linkIndex = newIndex;
  }
  void setSceneGraph (SceneGraph *sg) {
    // We can set the scene graph from non-null to null if we're being called
    // from the LinkManager's destructor, and it's hard to know if that's
    // happening, so we can't check the normal case where it is not permitted
    // to set the scene graph from non-null to null.  So we can't do this:
    // assert (sg || !m_sg); 
    m_sg = sg;
  }
  inline void check () const {
#ifndef NDEBUG
    m_sg->getLinkManager()->checkObjectLink (this);
#endif
  }
  // The LinkManager needs to be able to get the index of a possibly invalid
  // ObjectLink during garbage collections.
  int getRawIndex () const {
    return m_objIndex;
  }
public:
  // Need a default constructor for ObjectLink so we can make a Dynavec of
  // them, for instance.  It will be invalid, and you can't do much with it
  // other than destruct it or assign another ObjectLink into it.
  ObjectLink () {
    m_objIndex = -1;
    // Initialize the scene graph pointer to zero, so we can destruct this
    // object later without crashing.
    m_sg = 0;
    // Next one is neatness.  m_sg being 0 should actually say all we care
    // about, because it causes isValid to return false.
    m_linkIndex = -1;
  }
  // Make a new ObjectLink that points to the object at the given index.
  ObjectLink (SceneGraph *sg, int index) {
    m_sg = sg;
    m_objIndex = index;
    sg->getLinkManager()->addObjectLink (this);
    check ();
  }
  ObjectLink (SceneGraph *sg, const PhysicsObject *obj,
	      const Float *initialState) {
    // Here we take advantage of the fact that we represent a bad pointer with
    // a m_objectIndex of -1, and sg->addObject returns -1 if the object cannot
    // be added.
    m_sg = sg;
    m_objIndex = m_sg->addObject (obj, initialState);
    m_sg->getLinkManager()->addObjectLink (this);
    check ();
  }
  ObjectLink (const ObjectLink &ml) {
    m_sg = ml.m_sg;
    m_objIndex = ml.m_objIndex;
    m_sg->getLinkManager()->addObjectLink (this);
    check ();
  }
  ~ObjectLink () {
    // Do something reasonable when m_sg null so we can destruct the results of
    // the default constructor.
    if (m_sg) {
      check ();
      m_sg->getLinkManager()->deleteObjectLink (this);
    }
  }
  ObjectLink &operator= (const ObjectLink &ol) {
    SceneGraph *oldsg = m_sg;
    if (ol.m_sg) {
      m_sg = ol.m_sg;
    }
    m_objIndex = ol.m_objIndex;
    // If this guy isn't already managed, and we have enough information to
    // manage it now, tell the LinkManager to manage it.
    if (!oldsg && m_sg) {
      m_sg->getLinkManager()->addObjectLink (this);
    }
    check ();
    return *this;
  }
  // You can get a non-const SceneGraph from an ObjectLink.  This makes sense
  // because the SceneGraph is not logically part of the ObjectLink.
  SceneGraph *getSceneGraph () const {
    return m_sg;
  }
  // Check that the link points to an existing object in the scene graph.
  // Can return false if the link has been marked invalid (because the object
  // was deleted and then garbage collected) or because the object has been
  // deleted but not yet garbage collected (we check that SceneGraph::object
  // returns non-null).
  bool isValid () const {
    return
      m_sg &&
      -1 != m_objIndex &&
      m_objIndex < m_sg->maxIndex() &&
      m_sg->object (m_objIndex);
  }
  int getIndex () const {
    // Aren't allowed to get the index of an invalid link.
    assert (isValid ());
    return m_objIndex;
  }
  // You can change the object that a ObjectLink points to.
  // The new object ought to exist, though.
  void setIndex (int index) {
    m_objIndex = index;
  }
  // Called by the LinkManager when it knows that the object pointed to by this
  // ObjectLink has been deleted.
  void invalidate () {
    m_objIndex = -1;
  }
  // Convenience methods:
  // Returns the PhysicsObject, asserting that the pointer is valid.
  const PhysicsObject &operator* () const {
    assert (isValid ());
    check ();
    return *(m_sg->object (m_objIndex));
  }
  const PhysicsObject *operator-> () const {
    return &(operator*());
  }
  // Returns a pointer to the beginning of the object state, asserting that the
  // pointer is valid.
  Float *objectState () {
    assert (isValid ());
    check ();
    return m_sg->objectState (m_objIndex);
  }
  const Float *objectState () const {
    assert (isValid ());
    check ();
    return m_sg->objectState (m_objIndex);
  }
  // Get the links, asserting that the pointer is valid.
  void getLinks (int &count, const int *&links) const {
    getSceneGraph ()->getLinkManager()->getLinks (getIndex (), count, links);
  }
  // Get the links, asserting that the pointer is valid.  Leave in any -1's for
  // invalid objects.
  void getLinks (Dynavec <int> &dynalinks) const {
    dynalinks.extendTo (0);
    int count;
    const int *links;
    getLinks (count, links);
    for (int i = 0; i < count; i++) {
      dynalinks.push (links [i]);
    }
  }
  void setLinks (const Dynavec <int> &links) {
    getSceneGraph ()->getLinkManager()->setLinks (getIndex (), links);
  }
  // After you delete the object from an objectlink, the scene graph slot still
  // has the old value, and you can use that value.
  void deleteObject () {
    assert (isValid ());
    // It is not safe to do any computation after calling deleteObject, because
    // it may have caused this to be deleted.  So twist things around a bit so
    // we call invalidate before calling deleteObject.
    const int oldIndex = m_objIndex;
    invalidate ();
    m_sg->deleteObject (oldIndex);
  }
  Vec3 boundingSphereCenter () {
    assert (isValid ());
    return operator *().boundingSphereCenter (objectState ());
  }
  void translate (const Vec3 &v) {
    assert (isValid ());
    operator*().translate (objectState (), v);
  }
  void rotate (const Quaternion &q) {
    assert (isValid ());
    operator*().rotate (objectState(), q);
  }
  bool isVisible () const {
    assert (isValid ());
    return operator*().getObjectDrawer()->isVisible ();
  }
  bool isSelectable () const {
    assert (isValid ());
    return SelectionManager::isSelectable (*this);
  }
  BoundingSphere graphicsBoundingSphere () const {
    return operator*().getObjectDrawer()->graphicsBoundingSphere(objectState(),
								 m_objIndex,
								 m_sg);
  }
  BoundingSphere selectionBoundingSphere () const {
    return operator*().getObjectDrawer()->
      selectionBoundingSphere(objectState(),
			      m_objIndex,
			      m_sg);
  }
  bool operator== (const ObjectLink &ol) const {
    check ();
    ol.check ();
    return getIndex () == ol.getIndex ();
  }
  bool operator!= (const ObjectLink &ol) const {
    check ();
    ol.check ();
    return getIndex () != ol.getIndex ();
  }
};
#endif
