// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __OL_h__
#define __OL_h__

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __ObjectLink_h__
#include "ObjectLink.h"
#endif

// A template wrapper for ObjectLink that will avoid some downcasts.
// ObjectType should be a subclass of PhysicsObject.
template <class ObjectType> class OL
  : public ObjectLink
{
  void check () const {
    assert (dynamic_cast <const ObjectType *> (&(ObjectLink::operator* ())));
  }
public:
  // Need a default constructor for OL so we can have Dynavec's of them, for
  // instance.
  OL ()
    : ObjectLink ()
  {}
  OL (SceneGraph *sg, int index)
    : ObjectLink (sg, index)
  {
    check ();
  }
  OL (SceneGraph *sg, const ObjectType *obj, const Float *initialState)
    : ObjectLink (sg, obj, initialState)
  {
    // If the object failed to be added because it is out of bounds, then the
    // link may be invalid at this point, so it is not appropriate to call
    // check().
  }
  OL (const OL &ol)
    : ObjectLink (ol)
  {
    check ();
  }
  OL &operator= (const OL &ol)
  {
    ObjectLink::operator= (ol);
    check ();
    return *this;
  }
  // This version of operator* does the down cast in a relatively type-safe
  // way.
  const ObjectType &operator * () const {
    check ();
    const PhysicsObject &result = ObjectLink::operator* ();
    // Avoid dynamic_cast in production code, since it's very slow.
    return static_cast <const ObjectType &> (result);
  }
  const ObjectType *operator-> () const {
    assert (isValid ());
    check ();
    const PhysicsObject &o = ObjectLink::operator* ();
    const PhysicsObject *op = &o;
    assert (dynamic_cast <const ObjectType *> (op));
    return static_cast <const ObjectType *> (op);
  }
  // setIndex where the new index points at a deleted object is illegal.
  void setIndex (int index) {
    ObjectLink::setIndex (index);
    check ();
  }
};

#endif
