// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __LinkManager_h__
#define __LinkManager_h__

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

class SceneGraph;
class ObjectLink;

class LinkManager
  : public Refcount
{
  struct LinkManagerData;
  LinkManagerData *m_data;
  // Next one is unimplemented.  When we copy a scene graph, we transplant the
  // old link manager, instead of copying it.
  LinkManager (const LinkManager &lm);
  friend class ObjectLink;
  // Delete l from the list of links we manage.  This is only called from
  // ManageLink's destructor.
  void deleteObjectLink (ObjectLink *l);
  // Add an existing ObjectLink to the list of links we manage.  The link
  // should not already be on the list of links we manage.
  void addObjectLink (ObjectLink *l);
#ifndef NDEBUG
  // Verify that the object link is at the right poisition in our list of
  // object links.
  void checkObjectLink (const ObjectLink *l);
#endif
public:
  LinkManager ();
  // Delete a LinkManager.  This has to update all of the ObjectLink's created
  // by makeLink so they are invalid.
  ~LinkManager ();
  // A Scene Graph has a pointer to a LinkManager, but a LinkManager does not
  // need to have pointers to a scene graph.
  //
  // There are two kinds of links -- cheap ones that are between objects
  // (simply represented as integers that will be updated when the objects move
  // around in the scene graph), and expensive ones (ObjectLink's) that are
  // from C++ code outside of the scene graph and point into the scene graph
  // (and are updated in place when the objects move around in the scene
  // graph).  If C++ code needs to have a link to many things, it's best to
  // insert one object into the scene graph, have a managed link to that
  // object, and have links between that object and the many things that the
  // C++ code really wants to point to.
  //
  // None of the links are required to be bidirectional.
  // If the object at the other end of the link is deleted, then the link
  // logically disappears, for both kinds of link.
  // For the numerical links, we leave the number pointing to the vacant
  // PhysicsObject until the next garbage collection, at which time we delete
  // it from the list.  When we're reporting the links going out of an object
  // we filter out the links to the deleted objects.
  // For the managed links, when the destination object is deleted we will
  // arrange for the "valid" method to return false.
  //
  // If the scene graph is replaced with a difference scene graph, then we'll
  // copy the old one into the new one.  Thus we'll copy the LinkManager.  If
  // the new one changes the object numbering, it should call updateAfterGC
  // when copying to indicate the mapping from the old numbers to the new
  // numbers.

  // getLinks gets the numerical links leading away from the given object.
  // These link numbers are valid until the next time validate is called and
  // needs to do work.  The order of the numbers is the same as the order that
  // the numbers were given to setLinks, although the numbers themselves might
  // be different if there have been calls to updateAfterGC between then and
  // now.  The intent here is that a physics object can have a table mapping
  // the position of a link to some property of the relationship between this
  // object and the object at the other end of the link, and that table will
  // retain its meaning because the links won't move around.
  // 
  // The number of links is returned, as well as a pointer to the first link.
  // If there are no links, then don't use the returned value of links; it may
  // be null.
  //
  // If one of the objects in the links list has since been deleted and garbage
  // collected away, its object index in the links list will be replaced by -1.
  // If it has been deleted but no garbage collection has happened, then its
  // value in the links list will be unchanged but if you look the index up in
  // the SceneGraph you'll get a null object.  (No choice on this last one,
  // since the LinkManager is not notified when things are deleted from the
  // SceneGraph.)
  void getLinks (int index, int &count, const int *&links) const;

  // setLinks replaces all of the links from the given object with the given
  // list of links.  The given list may have duplicates, and it may have
  // pointers to deleted objects.  If you want to preserve some of 
  // the links, then call getLinks, work with the list of links, and then call
  // setLinks.
  // 
  // This method tries to be fast (it doesn't call NEW very often) and doesn't
  // worry much about minimizing storage.  Each time updateAfterGC is called,
  // we'll free all the wasted memory.
  // 
  // Beware that if you call setLinks, then any integer pointers returned by
  // getLinks may become invalid.  If you still want to have those links
  // returned by getLinks, better make a copy somewhere if you call setLinks.
  //
  // You can have a link value of -1, which means that a -1 will still be there
  // when someone eventually calls getLinks.
  void setLinks (int index, const Dynavec <int> &links);
  // Search the links from index, and return true if we find toFind.  
  bool findLink (int index, int toFind) const;
  // Add the given link to the list of links coming from index, unless it is
  // already there.  This allocates a Dynavec so it isn't very efficient.
  // Assumes that the links coming out of index are just a bag of links; if
  // there is some more interesting interpretation of the order of the links,
  // then don't use this.
  void addNewLink (int index, int newLink);
  // Remove the outgoing link from index.  Allocates a Dynavec, so this is not
  // very fast.
  void removeLink (int index, int toDelete);
  // Return a list of all links from the given object that aren't -1.
  Dynavec <int> simpleLinks (int index) const;
  // Find all object links that point at from, and make them point at to
  // instead.
  void redirectObjectLinks (int from, int to);
  
  // Update the links after a garbage collection.  For each old index i,
  // oldToNew [i] is the new index for the object, or is -1 if the
  // object has been deleted.  The scene graph has to be passed in just in case
  // the scene graph has been changed.  All of the ObjectLinks will be updated
  // to point to the new SceneGraph.
  void updateAfterGC (Dynavec <int> &oldToNew, SceneGraph *sg);
  // The value returned by getLinkGeneration increments every time we change
  // the links of an object, so it can be used to decide when to flush cached
  // information that depends on the links.  The initial value is 0, so it's
  // safe to use -1 as a value that is different from any value that will ever
  // be returned by getLinkGeneration.
  int getLinkGeneration () const;
};

#endif
