// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __ObjectLink_h__
#include "ObjectLink.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

namespace {
  struct ObjInfo {
    // Position of the first link.
    int m_first;
    // Number of links extending from the object.
    int m_numLinks;
    // Amount of space in LinkManagerData::m_links that is allocated for the
    // links from the object.  The actual number of links can grow to this much
    // before we have to move them.
    int m_allocatedLinks;
    ObjInfo ()
      : m_first (0),
	m_numLinks (0),
	m_allocatedLinks (0)
    {}
  };
};

struct LinkManager::LinkManagerData {
  // All the links are stored in here.
  Dynavec <int> m_links;
  // Where to find the links for each object.
  Dynavec <ObjInfo> m_info;
  // The managed links that we have to maintain.  Note that we have an ordinary
  // pointer here, not a smart pointer.  The ObjectLink destructor knows to
  // update this list when a ObjectLink is deleted.
  Dynavec <ObjectLink *> m_objectLinks;
  // The generation count, so code that caches information about links can know
  // when to recompute the cache. 
  int m_generation;
  LinkManagerData ()
    : m_generation (0)
  {}
};

LinkManager::LinkManager ()
  : m_data (NEW (LinkManagerData ()))
{
  // Nothing here.
}

LinkManager::~LinkManager () {
  // Any remaining object links are no longer our responsibility.  Set their
  // SceneGraph pointers to zero so they can't find us, since we won't exist
  // when they are destructed.
  for (int i = 0; i < m_data->m_objectLinks.size(); i++) {
    m_data->m_objectLinks[i]->setSceneGraph (0);
  }
  delete m_data;
  m_data = 0;
}

void LinkManager::getLinks (int index, int &count, const int *&links) const {
  if (index >= m_data->m_info.size ()) {
    count = 0;
    links = 0;
  } else {
    const ObjInfo &oi = m_data->m_info [index];
    count = oi.m_numLinks;
    // One possibility here is that oi.m_first is zero and m_data->m_links is
    // an array of size zero.  &* on a Dynavec is allowed when the dynavec has
    // size zero, so this code can work in all cases if we use pointer
    // arithmetic to skip the bounds check and do our own bounds check instead.
    assert (oi.m_first + oi.m_numLinks <= m_data->m_links.size ());
    links = (&*m_data->m_links) + oi.m_first;
  }
}

void LinkManager::setLinks (int index, const Dynavec <int> &links) {
  m_data->m_generation ++;
  if (index >= m_data->m_info.size ()) {
    // Default constructor for ObjInfo will initialize this properly for us.
    m_data->m_info.extendTo (index+1);
  }
  ObjInfo &oi = m_data->m_info [index];
  Dynavec <int> &l = m_data->m_links;
  if (links.size () > oi.m_allocatedLinks) {
    // If the old storage is too small, abandon it and allocate more at the end
    // of the vector.
    oi.m_first = m_data->m_links.size();
    // Arbitrary constant 2 is meant to avoid reallocation if only one or two
    // more links are added.  Double oi.m_allocatedLinks so that a gradually
    // growing set of links doesn't lead to quadratic behavior copying the old
    // links.
    oi.m_allocatedLinks =
      FloatUtil::max (links.size() + 2, 2 * oi.m_allocatedLinks);
    l.extendTo (oi.m_first + oi.m_allocatedLinks);
  }
  oi.m_numLinks = links.size ();
  for (int i = 0; i < links.size(); i++) {
    l [oi.m_first + i] = links [i];
  }
}

void LinkManager::updateAfterGC (Dynavec <int> &oldToNew, SceneGraph *sg) {
  LinkManagerData *newData = NEW (LinkManagerData ());
  newData->m_generation = m_data->m_generation + 1;
  // Update newData->m_info and newData->m_links.
  assert (oldToNew.size() >= m_data->m_info.size());
  newData->m_links.extendTo (m_data->m_links.size ());
  for (int i = 0; i < m_data->m_info.size(); i++) {
    const int newI = oldToNew [i];
    if (newI != -1) {
      if (newI >= newData->m_info.size ()) {
	newData->m_info.extendTo (newI+1);
      }
      ObjInfo &oi = m_data->m_info [i];
      newData->m_info [newI] = oi;
      for (int j = 0; j < oi.m_numLinks; j++) {
	const int pos = oi.m_first+j;
	const int oldIndex = m_data->m_links [pos];
	if (-1 == oldIndex) {
	  newData->m_links[pos] = -1;
	} else {
	  newData->m_links[pos] = oldToNew [oldIndex];
	}
      }
    }
  }
  // Update newData->m_objectLinks.
  newData->m_objectLinks = m_data->m_objectLinks;
  for (int i = 0; i < newData->m_objectLinks.size(); i++) {
    // Can't garbage collect invalid ObjectLinks here because the comments in
    // the ObjectLink.h header file say that the SceneGraph is kept current in
    // ObjectLink's, even invalid ones.
    // The invalid ObjectLinks will go away eventually when
    // they get destructed, and that will have to be good enough.
    // Can't call isValid before renumbering the objects, since isValid checks
    // that the object is non-null, and it can thus reference an array out of
    // bounds.
    const int oldIndex = newData->m_objectLinks [i]->getRawIndex();
    if (-1 != oldIndex) {
      const int newIndex = oldToNew [oldIndex];
      if (-1 == newIndex) {
	newData->m_objectLinks [i]->invalidate ();	
      } else {
	newData->m_objectLinks [i]->setIndex (newIndex);
	assert (sg->object (newIndex));
      }
      newData->m_objectLinks [i]->setSceneGraph (sg);
    }
  }
#ifndef NDEBUG
  // Gripe about being unfinished.
  static bool griped = false;
  if (!griped) {
    griped = true;
    message ("Consider making LinkManager garbage collect");
  }
#endif
  delete m_data;
  m_data = newData;
}

void LinkManager::deleteObjectLink (ObjectLink *const l) {
  Dynavec <ObjectLink *> &dml = m_data->m_objectLinks;
  const int pos = l->getObjectLinkIndex();
  assert (l == dml[pos]);
  if (pos < dml.size()-1) {
    dml[pos] = dml [dml.size()-1];
    dml[pos]->setObjectLinkIndex (pos);
  }
  dml.pop();
#ifndef NDEBUG
  for (int i = 0; i < dml.size(); i++) {
    assert (dml[i]->getObjectLinkIndex() == i);
  }
  if (dml.size () > 0) {
    SceneGraph *firstSg = dml[0]->getSceneGraph ();
    assert (firstSg);
    for (int i = 1; i < dml.size(); i++) {
      assert (dml[i]->getSceneGraph () == firstSg);
    }
  }
#endif
}

void LinkManager::addObjectLink (ObjectLink *l) {
#ifndef NDEBUG
  assert (!VecUtil::find (&*l, m_data->m_objectLinks));
#endif
  l->setObjectLinkIndex (m_data->m_objectLinks.size());
  m_data->m_objectLinks.push (l);
}

#ifndef NDEBUG
void LinkManager::checkObjectLink (const ObjectLink *const l) {
  assert (m_data->m_objectLinks[l->m_linkIndex] == l);
}
#endif

int LinkManager::getLinkGeneration () const {
  return m_data->m_generation;
}

bool LinkManager::findLink (const int index, const int toFind) const {
  int count;
  const int *links;
  getLinks (index, count, links);
  for (int i = 0; i < count; i++) {
    if (links [i] == toFind) {
      return true;
    }
  }
  return false;
}

void LinkManager::addNewLink (int index, int newLink) {
  if (!findLink (index, newLink)) {
    Dynavec <int> oldLinks;
    int count;
    const int *links;
    getLinks (index, count, links);
    for (int i = 0; i < count; i++) {
      oldLinks.push (links [i]);
    }
    oldLinks.push (newLink);
    setLinks (index, oldLinks);
  }
}

void LinkManager::removeLink (const int index, const int toDelete) {
  Dynavec <int> oldLinks;
  int count;
  const int *links;
  getLinks (index, count, links);
  for (int i = 0; i < count; i++) {
    if (links [i] != toDelete) {
      oldLinks.push (links [i]);
    }
  }
  setLinks (index, oldLinks);
}

// Return a list of all links from the given object that aren't -1.
Dynavec <int> LinkManager::simpleLinks (int index) const {
  Dynavec <int> result;
  int count;
  const int *l;
  getLinks (index, count, l);
  for (int i = 0; i < count; i++) {
    const int li = l [i];
    if (li != -1) {
      result.push (li);
    }
  }
  return result;
}

void LinkManager::redirectObjectLinks (const int from, const int to) {
  Dynavec <ObjectLink *> &links = m_data->m_objectLinks;
  const int s = links.size ();
  for (int i = 0; i < s; i++) {
    int &objIndex = links [i]->m_objIndex;
    if (from == objIndex) {
      objIndex = to;
    }
  }
}
