// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

// Write a .d file for input into Brenner's MD Fortran code.
// Format is described in Structures/READMD in the tar file for Brenner's MD
// package.  What this file doesn't say is:
//
// All text past the header is read with format "*", so I don't have to get any
// columns in any particular places.
// 
// I will write all of the derivatives as 0 for now.  
// FIXME Write velocities once I have atoms with velocities,

#ifndef __SceneWriterConfiguration_h__
#include "SceneWriterConfiguration.h"
#endif

#ifndef __fstream_h__
#include "fstream.h"
#define __fstream_h__
#endif

// For strerror.
#ifndef  __string_h__
#include <string.h>
#define __string_h__
#endif

// For errno.
#ifndef __errno_h__
#include <errno.h>
#define __errno_h__
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __FloatSlotValue_h__
#include "FloatSlotValue.h"
#endif

#ifndef __Vec3SlotValue_h__
#include "Vec3SlotValue.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __PositiveFloatSlotValue_h__
#include "PositiveFloatSlotValue.h"
#endif

#ifndef __OL_h__
#include "OL.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __BoringAtom_h__
#include "BoringAtom.h"
#endif

#ifndef __AtomInfo_h__
#include "AtomInfo.h"
#endif

#ifndef __BrennerAtom_h__
#include "BrennerAtom.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

namespace {
  String timeName () {
    return "Start time (Femtoseconds)";
  }

  String timeStepName () {
    return "Time step (Femtoseconds)";
  }

  String cubeName () {
    return "Periodicity of space (Angstroms)";
  }

  class DSceneWriterConfiguration
    : public SceneWriterConfiguration
  {
  public:
    DSceneWriterConfiguration () {
      initializeSlot (timeName(), NEW (FloatSlotValue (0)));
      initializeSlot (timeStepName (), NEW (PositiveFloatSlotValue (0.5)));
      initializeSlot (cubeName (),
		      NEW (Vec3SlotValue (Vec3 (1000, 1000, 1000))));
    }
    Float getTime () const {
      return FloatSlotValue::getSlot (this, timeName ());
    }
    void setTime (Float val) {
      FloatSlotValue::setSlot (this, timeName (), val);
    }
    Float getTimeStep () const {
      return FloatSlotValue::getSlot (this, timeStepName ());
    }
    void setTimeStep (Float val) {
      FloatSlotValue::setSlot (this, timeStepName (), val);
    }
    Vec3 getCube () {
      return Vec3SlotValue::getSlot (this, cubeName());
    }
    void setCube (const Vec3 &val) {
      Vec3SlotValue::setSlot (this, cubeName(), val);
    }
  };
  typedef TypedFactory <DSceneWriterConfiguration, Action> TF;
  class DSceneWriter
    : public TF
  {
  public:
    DSceneWriter ()
      : TF ("DSceneWriter")
    {}
    SP<DSceneWriterConfiguration> typedDefaultConfiguration () const {
      return NEW (DSceneWriterConfiguration ());
    }
    SP<Action> makeIt (DSceneWriterConfiguration *conf) const {
      SP<Action> result = NEW (Action ());
      SP<SceneGraph> sg = conf->getTopLevel()->getSceneGraph();
      // The object indices to write out.
      Dynavec <int> serialOrder;
      if (conf->getSelectionOnly ()) {
	SelectionManager::getSelection (serialOrder);
      } else {
	for (int i = 0; i < sg->maxIndex (); i++) {
	  serialOrder.push (i);
	}
      }
      { // Filter out everything but the atoms.
        // Warn if any atoms are not H or C.
	int to = 0;
	const String HSym = "H";
	const String CSym = "C";
	for (int from = 0; from < serialOrder.size(); from++) {
	  const int s = serialOrder [from];
	  SP<const PhysicsObject> const o = sg->object(s);
	  if (o->isInstanceOf (BoringAtom::staticClassId)) {
	    SP<const BoringAtom> const ba =
	      static_cast <const BoringAtom *> (&*o);
	    const int num = ba->getSymbolNumber();
	    if (num != 1 & num != 6) {
	      result->setMessage (String ("Atom ")+s+" is "+
				  AtomInfo::toSymbol (ba->getSymbolNumber())+
				  ", not H or C; writing it anyway");
	    }
	    serialOrder[to] = s;
	    to++;
	  } else {
	    // Skip the non-atom.
	  }
	}
	serialOrder.extendTo (to);
      }
      ofstream out (&*(conf->getFileName()));
      if (!out.is_open()) {
	result->setProblem (String ("Failed to open output file ")+
			    conf->getFileName()+": "+strerror (errno));
      }
      // The header.
      out << " " << conf->getFileName () << endl;
      // Number of atoms.
      out << serialOrder.size() << endl;
      // Time and timestep in femtoseconds
      out << conf->getTime () << " " << conf->getTimeStep () << endl;
      // The periodic cube.
      Vec3 cube = conf->getCube ();
      out << cube [0] << " " << cube [1] << " " << cube [2] << endl;
      String error = "";
      String line;
      // The Fortran code that reads these lines uses "*" as the format, so we
      // could almost write it out with variable width and be happy.  However,
      // our code that reads the files does fixed-width reads that line up with
      // the fixed-width writes that the Fortran does, so if we want to be able
      // to read our own output without falling down we need to do fixed-width
      // writes here.
      for (int i = 0; i < serialOrder.size() && !result->isProblem (); i++) {
	const int objectIndex = serialOrder [i];
	OL<BoringAtom> ba (sg, objectIndex);
	line = "";
	line.fromInt (i+1, 0, 5, &error);
	// Be sure to have a space between the columns, so Fortran's * format
	// will read correctly.
	line.fromInt (ba->getSymbolNumber(), 6, 4, &error);
	const Float *state = ba.objectState ();
	Vec3 where = BoringAtom::center (state);
	line.fromFloat (where [0], 11, 20, 6, &error);
	line.fromFloat (where [1], 31, 20, 6, &error);
	line.fromFloat (where [2], 51, 20, 6, &error);
	BrennerAtom::MotionFlag motion;
	if (ba->isInstanceOf (BrennerAtom::staticClassId)) {
	  SP<BrennerAtom> bra = (BrennerAtom *) (&*ba);
	  motion = bra->getMotionFlag ();
	} else {
	  motion = BrennerAtomConfiguration::DYNAMIC;
	}
	line.fromInt (motion, 72, 1, &error);
	if (error.size ()) {
	  result->setProblem ("Could not format line: "+error);
	} 
	out << line << endl;
	if (out.fail ()) {
	  result->setProblem ("I/O error on " + conf->getFileName ()+": "
			      + strerror (errno));
	}
      }
      // Write velocities.
      for (int i = 0; i < serialOrder.size() && !result->isProblem(); i++) {
	const int objectIndex = serialOrder [i];
	OL<BoringAtom> ba (sg, objectIndex);
	line = "";
	line.fromInt (i+1, 0, 5, &error);
	const Float *state = ba.objectState ();
	Vec3 vel = BoringAtom::velocity (state);
	line.fromFloat (vel[0], 6, 20, 6, &error);
	line.fromFloat (vel[1], 26, 20, 6, &error);
	line.fromFloat (vel[2], 46, 20, 6, &error);
	if (error.size()) {
	  result->setProblem ("Could not format line: "+error);
	}
	out << line << endl;
	if (out.fail ()) {
	  result->setProblem ("I/O error on " + conf->getFileName ()+": "
			      + strerror (errno));
	}
      }
      // Write three lumps of zeros.
      for (int j = 0; j < 3; j++) {
	for (int i = 0; i < serialOrder.size() && !result->isProblem(); i++) {
	  line = "";
	  line.fromInt (i+1, 0, 5, &error);
	  line.fromFloat (0, 6, 20, 6, &error);
	  line.fromFloat (0, 26, 20, 6, &error);
	  line.fromFloat (0, 46, 20, 6, &error);
	  if (error.size()) {
	    result->setProblem ("Could not format line: "+error);
	  }
	  out << line << endl;
	  if (out.fail ()) {
	    result->setProblem ("I/O error on " + conf->getFileName ()+": "
				+ strerror (errno));
	  }
	}
      }
      out.close ();
      if (!result->isProblem()) {
	if (out.fail ()) {
	  result->setProblem ("I/O error while closing "+
			      conf->getFileName()+": "+strerror (errno));
	}
      }
      return result;
    }
  };
  static const bool useless =
  (FactoryTable::store ("SceneWriter", NEW (DSceneWriter ())),
   true);
}
