// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

// For strerror.
#ifndef  __string_h__
#include <string.h>
#define __string_h__
#endif

// For errno.
#ifndef __errno_h__
#include <errno.h>
#define __errno_h__
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __fstream_h__
#include <fstream.h>
#define __fstream_h__
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __FileSceneLoaderConfiguration_h__
#include "FileSceneLoaderConfiguration.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __AtomFactory_h__
#include "AtomFactory.h"
#endif

#ifndef __PhysicsObject_h__
#include "PhysicsObject.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __FileSceneLoader_h__
#include "FileSceneLoader.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __BrennerAtomConfiguration_h__
#include "BrennerAtomConfiguration.h"
#endif

namespace {
  // A SceneLoader that reads the coord.d files that are the input and output
  // of Brenner's MD fortran distribution.  Has to be called DSceneLoader
  // because of the ".d", so tomFileSceneLoader can use this at the
  // appropriate time.
  class DSceneLoader
    : public FileSceneLoader
  {
    SP<Factory> m_f;
    SP<BrennerAtomConfiguration> m_bac;
    SP<SceneGraph> m_sg;
    Vec3 m_translation;
    Dynavec <int> m_objects;
    Dynavec <Float> m_plausibleState;
    void makeAtomLine (const String &line, const int expectedNumber) {
      String error;
      int lineNumber = line.toInt (0, &error);
      if (expectedNumber != lineNumber) {
	setProblem ("Expected atom number "+String (expectedNumber)+
		    " but got "+ lineNumber);
	// But keep parsing.
      }
      int atomicNo = line.toInt (5, &error);
      const Float x = line.toFloat (11, &error);
      const Float y = line.toFloat (31, &error);
      const Float z = line.toFloat (51, &error);
      BrennerAtomConfiguration::MotionFlag motion =
	(BrennerAtomConfiguration::MotionFlag) line.toInt (71, &error);
      if (error.size()) {
	setProblem ("Could not parse the line " + line + "\n" + error);
      }
      SP<PhysicsObject> atom;
      if (!error.size()) {
	m_bac->setSymbolNumber (atomicNo);
	m_bac->setMotionFlag (motion);
	SP<Configurable> obj = m_f->makeIt (&*m_bac);
	atom = dynamic_cast <PhysicsObject *> (&*obj);
	assert (atom);
	atom->plausibleState (m_plausibleState);
	atom->translate (m_plausibleState, m_translation);
	atom->translate (m_plausibleState, Vec3 (x, y, z));
	m_objects.push (m_sg->addObject (atom, m_plausibleState));
      }
    }
  public:
    DSceneLoader (SP<FileSceneLoaderConfiguration> slc) {
      m_f = FactoryTable::load ("Atom", "BrennerAtom");
      assert (m_f);
      m_bac =
	dynamic_cast <BrennerAtomConfiguration *>
	(&*m_f->defaultConfiguration ());
      assert (m_bac);
      m_sg = slc->getTopLevel ()->getSceneGraph ();
      assert (m_sg);
      m_translation = slc->getWhere ();
      String fileName = slc->getFileName ();
      ifstream stream (&*fileName);
      if (!stream.is_open ()) {
	setProblem ("Could not open " + fileName + " for reading: "
		    + strerror (errno));
	return;
      }
      String line;
      if (!isProblem ()) {
	stream >> line;
	if (stream.eof ()) {
	  setProblem ("Premature eof");
	}
      }
      if (!isProblem ()) {
	cout << "Scene title is " << line << endl;
	stream >> line;
	if (stream.eof ()) {
	  setProblem ("Premature eof");
	}
      }
      // Silly compiler thinks that if atoms isn't initialized here, it might
      // be used before it is initialized.  This isn't really true.  The
      // problem is that the compiler doesn't know that isProblem () can't
      // become false after it is true.
      int atoms = 0;
      if (!isProblem ()) {
	String error = "";
	atoms = line.toInt (0, &error);
	if (error.size()) {
	  setProblem ("Failed to parse number of atoms: "+error);
	} else if (atoms < 0) {
	  setProblem ("Negative number " + String (atoms) + " of atoms");
	}
      }
      if (!isProblem ()) {
	// Third line has elapsed time and timestep. don't care.
	stream >> line; 
	// Fourth line has periodic box; don't care.
	stream >> line; 
	if (stream.eof ()) {
	  setProblem ("Premature eof");
	}
      }
      m_objects.extendTo (0);
      for (int i = 1; i <= atoms && !isProblem(); i++) {
	if (stream.eof ()) {
	  setProblem ("Premature eof");
	} else {
	  stream >> line;
	  makeAtomLine (line, i);
	}
      }
      // If we opened it, it's important to close even if there were problems.
      stream.close ();
      if (slc->getSelect ()) {
	// If asked to select everything we added, do it even if there were
	// problems.
	Dynavec <int> selection;
	for (int i = 0; i < m_objects.size(); i++) {
	  const int s = m_objects [i];
	  if (-1 != s && m_sg->object (s)->isSelectable ()) {
	    selection.push (s);
	  }
	}
	SelectionManager::setSelection (m_sg, selection);
      }
      if (isProblem()) {
	setProblem (getProblem () + " in " + fileName);
      }
    }
  };
  class DSceneLoaderFactory
    : public Factory
  {
  public:
    DSceneLoaderFactory ()
      : Factory ("DSceneLoader")
    {}
    SP<Configuration> defaultConfiguration () const {
      return NEW (FileSceneLoaderConfiguration ());
    }
    SP<Configurable> makeIt (const SP<Configuration> c) const {
      FileSceneLoaderConfiguration *conf =
	dynamic_cast <FileSceneLoaderConfiguration *> (&*c);
      assert (conf);
      return NEW (DSceneLoader (conf));
    }
  };
  const bool useless =
  (FactoryTable::store ("FileSceneLoader", NEW (DSceneLoaderFactory ())),
   true);
};
