// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __IncrementalStorage_h__
#include "IncrementalStorage.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Vec3i_h__
#include "Vec3i.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

#ifndef __BoundingBoxInt_h__
#include "BoundingBoxInt.h"
#endif

#ifndef NDEBUG
#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __ArrayStorageConfiguration_h__
#include "ArrayStorageConfiguration.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

// This Storage uses arrays to find the neighbors of objects.

namespace {

struct ArrayElt {
  // The indices of the objects near here.  That is, a list of all of
  // the first arguments to store that were passed.
  Dynavec <int> m_indices;
  // The position of this stored coordinate in m_storedCoords, so we
  // can remove efficiently.
  int m_coordNo;
};

class ArrayStorageImpl : public IncrementalStorage {
  // The extent of the space we're trying to cover.  The actual memory
  // required is m_size * m_size * m_size pointers.  m_size doesn't
  // need to be a power of 2.
  const int m_size;
  // The huge array, of size m_size * m_size * m_size.
  ArrayElt ** const m_hugeArray;
  // The minimum coordinate that we represent.  -m_size/2, assuming we
  // want our space to be centered at the origin.
  const int m_min;
  // Yadda maximum yadda.  m_min + m_size, yadda yadda.
  const int m_max;
  inline bool arrayInBounds (int x, int y, int z) const {
    assert (m_max == m_min + m_size);
    return (x >= m_min && x < m_max &&
	    y >= m_min && y < m_max &&
	    z >= m_min && z < m_max);
  }
  inline ArrayElt ** arrayIndex (const int x, const int y, const int z) {
    assert (arrayInBounds (x, y, z));
    return
      m_hugeArray +
      (((x - m_min) * m_size) + y - m_min) * m_size + z - m_min;
  }
  inline const ArrayElt *const*const arrayIndex
    (const int x, const int y, const int z)
    const
    {
      assert (arrayInBounds (x, y, z));
      return
	m_hugeArray +
	(((x - m_min) * m_size) + y - m_min) * m_size + z - m_min;
    }
  // The distance, in world coordiantes, of one cube that we
  // represent.  Should be the typical size of an object.  If
  // it's smaller, then we waste time storing lots of entries for the
  // smallest object.  If it's larger, then objects have more
  // neighbors than they should so we waste time discarding irrelevant
  // neighbors.
  // If there is no typical size of an object, then this is slow.
  const Float m_characteristicSize;
  // A set of booleans used to quickly eliminate duplicates.
  // When we aren't in the middle of some method, all values in
  // this array should be false.
  // Another invariant: when we aren't in the store method, the length
  // of the set array should be enough to include the largest element
  // stored.
  // Logically, this is local to the fetch method, but we put it here
  // so we don't have to reallocate and reinitialize it each time
  // around.
  // fetch logically returns a const, hence "mutable" here.
  mutable Dynavec <bool> m_set;
  // The places that we have stored nonzero pointers, so we can
  // quickly set them all back to null when empty is called.
  // When we aren't in the middle of some method, all values stored
  // here should be different from each other.  If the coordinates of
  // a point aren't in here, then that element of m_hugeArray should
  // be null.
  Dynavec <ArrayElt **> m_storedCoords;
  // A list of unused index vectors so we don't have to allocate
  // them again and again.  No element should be on here twice.
  Dynavec <ArrayElt *> m_indices;
  // Grabs a new vector, either from m_indices if availible, or by
  // calling new.  Initializes the size to 0.
  ArrayElt *newVec ();
  // Only ArrayStorageFactory can make an instance of this class.
  friend class ArrayStorageFactory;
  friend class ArrayIncrementalStorageFactory;
  ArrayStorageImpl (int size, Float characteristicSize);
  ~ArrayStorageImpl ();
 public:
  bool store (int value, const BoundingBox &box);
  bool store (int value, const BoundingBox &box, BoundingBox &includedBox);
  bool store (int value, const BoundingBox &box, BoundingBox *includedBox);
  void fetch (const BoundingBox &box, int notThis,
	      Dynavec <int> &result) const;
  void remove (int value, const BoundingBox &box);
  void empty ();
};

bool ArrayStorageImpl::store (const int value, const BoundingBox &box) {
  return store (value, box, 0);
}

bool ArrayStorageImpl::store (const int value, const BoundingBox &box, BoundingBox &includedBox) {
  return store (value, box, &includedBox);
}

bool ArrayStorageImpl::store (const int value,
			      const BoundingBox &box,
			      BoundingBox *includedBox) {
  assert (m_hugeArray);
  assert (m_size > 0);
  assert (m_min < m_max);
  assert (m_characteristicSize > 0);
  const BoundingBoxInt intBox (box, m_characteristicSize);
  if (includedBox) {
    *includedBox = BoundingBox (intBox, m_characteristicSize);
  }
  const Vec3i min = intBox.min ();
  const Vec3i max = intBox.max ();
  if (!arrayInBounds (min [0], min [1], min [2]) ||
      !arrayInBounds (max [0] - 1, max [1] - 1, max [2] - 1)) {
    return false;
  } else {
    { // Extend m_set, so we can use it freely in fetch.
      const int oldSize = m_set.size ();
      if (value >= oldSize) {
	m_set.extendTo (value+1);
	for (int i = oldSize; i <= value; i++) {
	  m_set [i] = false;
	}
      }
    }
    {
      int i, j, k;
      const int sizeSquared = m_size * m_size;
      // Minimizing number of multiplies.
      ArrayElt ** iptr;
      ArrayElt ** jptr;
      ArrayElt ** kptr;
      for (i = min [0], iptr = arrayIndex (min [0], min [1], min [2]);
	   i < max [0];
	   i++, iptr += sizeSquared) {
	for (j = min [1], jptr = iptr;
	     j < max [1];
	     j++, jptr += m_size) {
	  for (k = min [2], kptr = jptr;
	       k < max [2];
	       k++, kptr++) {
	    assert (arrayIndex (i, j, k) == kptr);
	    if (!(*kptr)) {
	      *kptr = newVec ();
	      assert (0 == (*kptr)->m_indices.size());
	      (*kptr)->m_coordNo = m_storedCoords.size();
	      m_storedCoords.push (kptr);
	    }
#ifndef NDEBUG
	    // No value should appear on its vector twice.
	    for (int z = 0; z < (*kptr)->m_indices.size(); z++) {
	      assert ((*kptr)->m_indices[z] != value);
	    }
#endif
	    (*kptr)->m_indices.push (value);
	  }
	}
      }
    }
    return true;
  }
}

void ArrayStorageImpl::remove (const int value, const BoundingBox &box) {
  assert (m_hugeArray);
  assert (m_size > 0);
  assert (m_min < m_max);
  assert (m_characteristicSize > 0);
  const BoundingBoxInt intBox (box, m_characteristicSize);
  const Vec3i min = intBox.min ();
  const Vec3i max = intBox.max ();
  assert (arrayInBounds (min [0], min [1], min [2]));
  assert (arrayInBounds (max [0] - 1, max [1] - 1, max [2] - 1));
  int i, j, k;
  const int sizeSquared = m_size * m_size;
  // Minimizing number of multiplies.
  ArrayElt ** iptr;
  ArrayElt ** jptr;
  ArrayElt ** kptr;
  for (i = min [0], iptr = arrayIndex (min [0], min [1], min [2]);
       i < max [0];
       i++, iptr += sizeSquared) {
    for (j = min [1], jptr = iptr;
	 j < max [1];
	 j++, jptr += m_size) {
      for (k = min [2], kptr = jptr;
	   k < max [2];
	   k++, kptr++) {
	assert (arrayIndex (i, j, k) == kptr);
	assert (*kptr);
	ArrayElt &kptrdest = **kptr;
#ifndef NDEBUG
	assert (VecUtil::find (value, kptrdest.m_indices));
#endif
	int s = kptrdest.m_indices.size();
	for (int z = 0; z < s; z++) {
	  if (value == kptrdest.m_indices[z]) {
	    // Note that the next two steps work even if value is
	    // the last element of kptrdest.m_indices.
	    kptrdest.m_indices [z] = kptrdest.m_indices [s-1];
	    kptrdest.m_indices.pop ();
	    break;
	  }
	}
	if (0 == s) {
	  // Take it off of m_storedCoords.
	  m_storedCoords [kptrdest.m_coordNo] =
	    m_storedCoords [m_storedCoords.size() - 1];
	  assert ((*m_storedCoords [kptrdest.m_coordNo])->m_coordNo ==
		  m_storedCoords.size() - 1);
	  (*m_storedCoords [kptrdest.m_coordNo])->m_coordNo = kptrdest.m_coordNo;
	  // Return it to the free list.
#ifndef NDEBUG
	  kptrdest.m_coordNo = -1;
#endif
	  m_indices.push (*kptr);
	}
      }
    }
  }
#ifdef VERY_SLOW_DEBUG
  // After we remove it, it shouldn't be there any more.
  for (int i = 0; i < m_storedCoords.size() - 1; i++) {
    assert (! VecUtil::find (value, (*m_storedCoords[i])->m_indices));
  }
#endif
}

void ArrayStorageImpl::fetch (const BoundingBox &box,
			      const int notThis,
			      Dynavec <int> &result) const {
  result.extendTo (0);
#ifdef VERY_SLOW_DEBUG
  // m_set is normally all false.
  for (int q = 0; q < m_set.size(); q++) {
    assert (!m_set [q]);
  }
#endif
  assert (m_hugeArray);
  assert (m_size > 0);
  assert (m_min < m_max);
  assert (m_characteristicSize > 0);
  m_set [notThis] = true;
  const BoundingBoxInt intBox (box, m_characteristicSize);
  Vec3i min = intBox.min ();
  Vec3i max = intBox.max ();
  {
    int i, j, k;
    const int sizeSquared = m_size * m_size;
    const ArrayElt *const* iptr;
    const ArrayElt *const* jptr;
    const ArrayElt *const* kptr;
    for (i = min [0], iptr = arrayIndex (min [0], min [1], min [2]);
	 i < max [0]; 
	 i++, iptr += sizeSquared) {
      for (j = min [1], jptr = iptr;
	   j < max [1];
	   j++, jptr += m_size) {
	for (k = min [2], kptr = jptr;
	     k < max [2];
	     k++, kptr++) {
	  assert (arrayIndex (i, j, k) == kptr);
	  if (*kptr) {
#ifdef VERY_SLOW_DEBUG
	    { // Non-null array elements are in m_storedCoords.
	      bool foundIt = false;
	      for (int q = 0; q < m_storedCoords.size(); q++) {
		if (m_storedCoords [q] == kptr) foundIt = true;
		assert ((*(m_storedCoords[q]))->m_coordNo == q);
	      }
	      assert (foundIt);
	    }
#endif
	    const ArrayElt &here = **kptr;
	    int s = here.m_indices.size();
	    for (int z = 0; z < s; z++) {
	      int index = here.m_indices [z];
	      if (!m_set [index]) {
		m_set [index] = true;
		result.push (index);
	      }
	    }
	  } else {
#ifdef VERY_SLOW_DEBUG
	    // Null array elements are not in m_storedCoords.
	    for (int q = 0; q < m_storedCoords.size (); q++) {
	      assert (m_storedCoords [q] != kptr);
	    }
#endif
	  }
	}
      }
    }
  }
  // Clear the bits.
  m_set [notThis] = false;
  {
    int s = result.size ();
    for (int i = 0; i < s; i++) {
      m_set [result[i]] = false;
    }
  }
#ifdef VERY_SLOW_DEBUG
  // m_set is normally all false.
  for (int q = 0; q < m_set.size(); q++) {
    assert (!m_set [q]);
  }
#endif
}

void ArrayStorageImpl::empty () {
  int s = m_storedCoords.size ();
  for (int q = 0; q < s; q++) {
    ArrayElt ** elt = m_storedCoords [q];
    assert (elt);
    assert (*elt);
    assert ((*elt)->m_coordNo == q);
#ifndef NDEBUG
    (*elt)->m_coordNo = -1;
#endif
    m_indices.push (*elt);
    *elt = 0;
  }
  m_storedCoords.extendTo (0);
#ifndef NDEBUG
  // After we've called empty, the array should be all null.
  for (int i = m_min; i < m_max; i++) {
    for (int j = m_min; j < m_max; j++) {
      for (int k = m_min; k < m_max; k++) {
	assert (0 == *arrayIndex (i, j, k));
      }
    }
  }
#endif
}

ArrayElt *ArrayStorageImpl::newVec () {
  ArrayElt *result;  
  if (m_indices.size ()) {
    result = m_indices.pop ();
    assert (-1 == result->m_coordNo);
    result->m_indices.extendTo (0);
  } else {
    result = NEW (ArrayElt ());
#ifndef NDEBUG
    result->m_coordNo = -1;
#endif
  }
  return result;
}

ArrayStorageImpl::ArrayStorageImpl (int arraySize, Float minSize) :
  m_size (arraySize),
  m_hugeArray (NEW (ArrayElt* [arraySize * arraySize * arraySize])),
  m_min (-arraySize / 2),
  m_max ((-arraySize / 2) + arraySize),
  m_characteristicSize (minSize)
{
  assert (m_size > 0);
  const int maxSize = arraySize * arraySize * arraySize;
  for (int i = 0; i < maxSize; i++) {
    m_hugeArray [i] = 0;
  }
}

ArrayStorageImpl::~ArrayStorageImpl () {
  empty ();
  for (int q = 0; q < m_storedCoords.size(); q++) {
    assert ((*m_storedCoords[q])->m_coordNo == q);
    delete *m_storedCoords [q];
  }
  for (int q = 0; q < m_indices.size(); q++) {
    delete m_indices [q];
  }
  delete [] m_hugeArray;
}

class ArrayStorageFactory : public Factory {
public:
  ArrayStorageFactory () : Factory ("ArrayStorage") {}
  SP<Configuration> defaultConfiguration () const {
    return NEW (ArrayStorageConfiguration ());
  }
  SP<Configurable> makeIt (SP<Configuration> c) const {
    ArrayStorageConfiguration *conf =
      dynamic_cast <ArrayStorageConfiguration *> (&*c);
    assert (conf);
    return NEW (ArrayStorageImpl (conf->getArraySize(),
				  conf->getCharacteristicSize()));
  }
};

class ArrayIncrementalStorageFactory : public Factory {
public:
  ArrayIncrementalStorageFactory () : Factory ("ArrayStorage") {}
  SP<Configuration> defaultConfiguration () const {
    return NEW (ArrayStorageConfiguration ());
  }
  SP<Configurable> makeIt (SP<Configuration> c) const {
    ArrayStorageConfiguration *conf =
      dynamic_cast <ArrayStorageConfiguration *> (&*c);
    assert (conf);
    return NEW (ArrayStorageImpl (conf->getArraySize(),
				  conf->getCharacteristicSize()));
  }
};

static const bool useless =
(FactoryTable::store ("Storage", NEW (ArrayStorageFactory ())),
 FactoryTable::store ("IncrementalStorage",
		      NEW (ArrayIncrementalStorageFactory ())),
 true);

} // namespace
