// No copyright; this file is in the public domain.  Tim Freeman 23 Feb 2000.

// This file is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

#include "ActionConfiguration.h"
#include "TypedFactory.h"
#include "Action.h"
#include "FactoryTable.h"
#include "MemoryUtil.h"
#include "String.h"
#include "SP.h"
#include "Factory.h"
#include "Float.h"
#include "Dynavec.h"
#include "Vec3.h"
#include "TopLevel.h"
#include "SceneGraph.h"
#include "LinkManager.h"
#include "BoringAtom.h"

namespace {
  // This sample plugin defines a command that inserts "Hi!" into the scene,
  // spelling out the letters with the default DesignAtom.
  
  // Dot represents one atom in the figure we're going to add.
  struct Dot {
    // x and y coordinates of the atom, in an arbitrary coordinate system.
    int x, y;
    // The index in dotArray of the atom we want to link this atom to, or
    // -1 if none.  We interpret hiArray from the beginning, so linkTo had
    // better be less than or equal than the index of this dot.
    int linkTo;
  };
  // Here's an ASCII drawing of figure we're going to add:
  //
  //y=4  8   10   14   18
  //y=3  7   9         17
  //y=2  2 3 4    13   16
  //y=1  1   5    12
  //y=0  0   6    11   15
  //
  //   x=0 1 2  3 4  5 6
  //
  // And here it is as a sequence of Dot's:
  const Dot hiArray [] = {
    {0, 0, -1}, // 0
    {0, 1, 0},  // 1
    {0, 2, 1},  // 2
    {1, 2, 2},  // 3
    {2, 2, 3},  // 4
    {2, 1, 4},  // 5
    {2, 0, 5},  // 6
    {0, 3, 2},  // 7
    {0, 4, 7},  // 8
    {2, 3, 4},  // 9
    {2, 4, 9},  // 10
    {4, 0, -1}, // 11
    {4, 1, 11}, // 12
    {4, 2, 12}, // 13
    {4, 4, -1}, // 14
    {6, 0, -1}, // 15
    {6, 2, -1}, // 16
    {6, 3, 16}, // 17
    {6, 4, 17}};// 18
  // Multiply all dimensions by this much.  1.4 Angstroms is a plausible bond
  // length between two Carbon atoms.
  const Float bondLength=1.4;
  class Hi
    : public TypedFactory <ActionConfiguration, Action>
  {
  public:
    // Give the factory (that is, the command) a name so we can look it up
    // later.
    Hi ()
      : TypedFactory <ActionConfiguration, Action> ("Hi")
    {}
    // Specify the default configuration for this command.  Nothing interesting
    // here.
    SP<ActionConfiguration> typedDefaultConfiguration () const {
      return NEW (ActionConfiguration ());
    }
    SP<Action> makeIt (ActionConfiguration *conf) const {
      // Get the TopLevel so we can get the SceneGraph from it.
      SP<TopLevel> const top = conf->getTopLevel ();
      // Get the SceneGraph so we can add atoms to it.
      SP<SceneGraph> const sg = top->getSceneGraph ();
      // Get the LinkManager so we can create links between the atoms. 
      SP<LinkManager> const lm = sg->getLinkManager ();
      // Get the factory that makes DesignAtom's.
      SP<Factory> const factory = FactoryTable::load ("Atom", "DesignAtom");
      // Make one.  We only need one DesignAtom because the position is stored
      // outside of the DesignAtom itself.  DesignAtom has no header file and
      // it is a subclass of BoringAtom which does have a header file, so da is
      // a BoringAtom.
      SP<BoringAtom> const da =
	dynamic_cast <BoringAtom *>
	(&*factory->makeIt (factory->defaultConfiguration ()));
      assert (da);
      // We'll hold the state in the plausibleState vector.
      Dynavec <Float> plausibleState;
      // Put a plausible state into the plausibleState vector.  For
      // BoringAtom's, this state has zero position and zero velocity.
      da->plausibleState (plausibleState);
      // A list of the indices of the atoms we've added, so we can link them
      // together properly.
      Dynavec <int> atomIndices;
      // Vectors for the x and y coordinates 
      const Vec3 dx = Vec3 (bondLength, 0, 0);
      const Vec3 dy = Vec3 (0, bondLength, 0);
      for (int i = 0; i < sizeof (hiArray) / sizeof (Dot); i++) {
	const Dot &d = hiArray [i];
	BoringAtom::setCenter (&*plausibleState, d.x*dx+d.y*dy);
	// Add the new atom, remembering its index.
	const int index = sg->addObject (da, plausibleState);
	// Save the index so we can make links later.
	atomIndices.push (index);
	if (-1 != d.linkTo) {
	  // Add the link.  An individual link is a directed edge, but bonds
	  // between atoms are undirected, so we have to do it both ways.
	  lm->addNewLink (index, atomIndices [d.linkTo]);
	  lm->addNewLink (atomIndices [d.linkTo], index);
	}
      }
      // Generate the return value.
      SP<Action> result = NEW (Action ());
      // Give a success message.
      result->setMessage ("Hi!");
      return result;
    }
  };
  // Register this factory as an Action.
  bool useless = (FactoryTable::store ("Action", NEW (Hi ())), true);
}
