// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __TheObjectDrawerFactory_h__
#include "TheObjectDrawerFactory.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __ObjectDrawerConnection_h__
#include "ObjectDrawerConnection.h"
#endif

#ifndef __ConstantRecursiveSlotValue_h__
#include "ConstantRecursiveSlotValue.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __ObjectDrawerConfiguration_h__
#include "ObjectDrawerConfiguration.h"
#endif

namespace {
  struct TheObjectDrawerFactoryData {
    Dynavec <ObjectDrawerConnection *> m_connections;
    SP<ConstantRecursiveSlotValue> m_odf;
    Dynavec <String> m_objectDrawers;
  };
  TheObjectDrawerFactoryData *s_data = 0;
  void deAllocate () {
    if (s_data) {
      delete s_data;
      s_data = 0;
    }
  }
  void checkAllocated () {
    if (!s_data) {
      s_data = NEW (TheObjectDrawerFactoryData ());
    }
  }
  bool useless = (MemoryUtil::registerDeallocator (deAllocate),
		  true);
}

void TheObjectDrawerFactory::giveObjectDrawer (SP<ObjectDrawerConnection> odconn) {
  checkAllocated ();
  assert (odconn);
  SP<ConstantRecursiveSlotValue> odf = s_data->m_odf;
  if (odf) {
    SP<ObjectDrawerConfiguration> odconf =
      dynamic_cast <ObjectDrawerConfiguration *> (&*odf->getConfiguration());
    odconf->setClassName (odconn->getObjectDrawingClass());
    odconf->setPhysicsObjectConfiguration (odconn->getConfiguration());
    SP<ObjectDrawer> od =
      dynamic_cast <ObjectDrawer *> (&*odf->makeIt ());
    odconn->setObjectDrawer (od);
  } else {
    // If they make a PhysicsObject before the object drawer factory
    // is set, or we are setting the odf to 0 just before shutting
    // down a window system, then we fall into this case. 
    odconn->setObjectDrawer (0);
  }
}

void
TheObjectDrawerFactory::registerConnection (SP<ObjectDrawerConnection> odc) {
  checkAllocated ();
  Dynavec <ObjectDrawerConnection *> &conns = s_data->m_connections;
  int pos = conns.size();
  conns.push (odc);
  odc->setObjectDrawerPosition (pos);
  giveObjectDrawer (odc);
}

void
TheObjectDrawerFactory::deRegisterConnection (SP<ObjectDrawerConnection> odc) {
  checkAllocated ();
  Dynavec <ObjectDrawerConnection *> &conns = s_data->m_connections;
  int position = odc->getObjectDrawerPosition ();
  assert (conns [position] == odc);
  if (position < conns.size() - 1) {
    conns [position] = conns [conns.size() - 1];
    conns [position]->setObjectDrawerPosition (position);
  }
  conns [conns.size() - 1] = 0;
  conns.pop ();
}

void
TheObjectDrawerFactory::setTheObjectDrawerFactory
(SP<ConstantRecursiveSlotValue> odf)
{
  checkAllocated ();
  s_data->m_odf = odf;
  Dynavec <ObjectDrawerConnection *> &conns = s_data->m_connections;
  for (int i = 0; i < conns.size(); i++) {
    giveObjectDrawer (conns [i]);
  }
}

void TheObjectDrawerFactory::registerObjectDrawer (const String &className) {
  checkAllocated ();
  s_data->m_objectDrawers.push (className);
}

Dynavec <String> TheObjectDrawerFactory::allRegisteredObjectDrawers () {
  checkAllocated ();
  return s_data->m_objectDrawers;
}
