// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __PhysicsObject_h__
#define __PhysicsObject_h__

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __ObjectDrawerConnection_h__
#include "ObjectDrawerConnection.h"
#endif

class Quaternion;
class SceneGraph;
class PhysicsObjectInfo;
class Vec3;
class BoundingSphere;
class BoundingBox;

// An object that we're going to do physics simulations with.
// We plan to have many similar objects running around, so we have
// extrinsic state.  The state is some number of Floats, and the
// number depends on exactly which PhysicsObject we're dealing with.

// "Immutable" below says that for the lifetime of a given instance of
// a PhysicsObject, the value returned by the routine will never
// change, so it may be cached by the system.
class Canvas;
class PhysicsObject
  : public Configurable, public ObjectDrawerConnection
{

 public:

  PhysicsObject (const String &objectDrawerClass, SP<Configuration> conf);

  // The number of Float's in the state information.  It's the
  // callers responsibility to allocate the state.  Immutable.
  virtual int stateSize () const = 0;

  // Whether the physics of the object intersects a bounding box.
  // The caller should be able to deal with quick-and-dirty
  // approximations here that return true when the actual answer is
  // false, but it is important that if actual answer is true, this
  // method must return true.  (This is for physics that happens because
  // objects are nearby.  If physics with an object happens because that object
  // is linked to this one, then that doesn't have to cause physicsIntersect to
  // be true.  We aren't given enough information to follow all those links and
  // find the distance to the guy at the other end, anyway.)
  virtual bool physicsIntersect (const Float *state, const BoundingBox &box)
    const = 0;

  // Return a plausible initial state for the PhysicsObject.  The size
  // of the state should be stateSize(), and the object should be
  // centered at the origin.  To move it, use translate.
  virtual void plausibleState (Dynavec <Float> &result) const = 0;

  // Need to be able to compute a bounding ball so I can compute near
  // and far clipping planes when drawing the figure.
  virtual Vec3 boundingSphereCenter (const Float *state) const = 0;

  // Like the graphicsmumble's, but for physical effects instead.
  virtual Float physicsBoundingSphereRadius () const = 0; // immutable
  virtual bool isPhysicsBounded () const = 0;
  
  // Next ones aren't the fast path, but they remain because code I'm
  // not yet ready to throw away uses them.
  BoundingBox physicsBoundingBox (const Float *state) const;
  BoundingSphere physicsBoundingSphere (const Float *state) const;

  // Update state to say the object has been translated by the given vector.
  // This method is const because it can only update the state.  It
  // can't modify the PhysicsObject.
  virtual void translate (Float *state, const Vec3 &v) const = 0;
  // Next one is for convenience.  Assert that the length of the state
  // is right, then call the virtual translate method.
  void translate (Dynavec <Float> &state, const Vec3 &v) const;
  // Rotate the object.  If the object has no rotational degrees of freedom,
  // this should be implemented to do nothing.
  virtual void rotate (Float *state, const Quaternion &q) const = 0;
  // Whether the object is exempt from the laws of physics.  Walls
  // that keep objects from falling out of bounds, mouse-controlled
  // objects, and so forth will return true; real physical objects
  // return false.  Immutable.
  virtual bool exempt () const = 0;
  
  // implement the laws of physics for this object and
  // perhaps the nearby objects that it affects.
  // If this object has an opinion about how this object should interact with
  // another object, and the other object has and opinion about what
  // should happen to this object too, the opinions should *add*.
  // The object is not given the time, so we are
  // assuming here that the laws of physics do not vary with time.

  // If an object is exempt from the laws of physics, then the
  // neighborDeriv value passed for it will be the null pointer.
  // The default implementation does nothing.
  // FIXME You can't get the neighbors of the neighbors.
  // Probably redo all this when I get around to sorting the scene graph by
  // object type.
  virtual void doPhysics (// All of the state.
			  const Float *state,
			  // All of the derivative information
			  Float *deriv,
			  // The index of me into both state and deriv.
			  int myIndex,
			  // Objects and positions for the neighbors.
			  const Dynavec<PhysicsObjectInfo> &neighbors) const;
  // Just like the previous one, except we also get a "links" argument, which
  // is a list of the objects that we have links to.  The default
  // implementation ignores the links argument and calls doPhysics without that
  // argument.  The physicsObject should be able to deal with links to deleted
  // objects that have been garbage collected.  They can be recognized because
  // their PhysicsObject is null and their index is -1.
  virtual void doPhysics (const Float *state,
			  Float *deriv,
			  int myIndex,
			  const Dynavec<PhysicsObjectInfo> &neighbors,
			  const Dynavec<PhysicsObjectInfo> &links) const;

  // A unique pointer identifying the class of the physics object.
  // A PhysicsObject class (say "BouncingBall") should have a static
  // constant member BouncingBall::staticClassId, and each instance of
  // BouncingBall should return BouncingBall::staticClassId as its
  // value.  To help with debugging, BouncingBall::staticClassId
  // should be the string "BouncingBall".
  // dynamic_cast could be used to do compute this without the added
  // method and static member, but dynamic_cast is slow.
  virtual const char *classId ()const = 0;
  // Return whether the given object is an instance of the given
  // class, where the class is given by it's classId.
  // Could be replaced by dynamic_cast, except that's too slow.
  // No need to report that everything with this method is an instance
  // of PhysicsObject.  PhysicsObject doesn't have a staticClassId
  // anyway, so you couldn't implement such reporting if you wanted to.
  virtual bool isInstanceOf (const char *classId) const = 0;
  // Here's the goals of animation:
  // 
  // Some PhysicsObjects should get a chance to tweak the scene graph or
  // themselves once per frame (for ones that are for displaying to the user)
  // or once per timestep (for ones that do physically meaningful things to the
  // scene).
  //
  // If there are PhysicsObjects that might become animated if the user gives
  // some more inputs, but they aren't animated now, then we want to be able to
  // wait for the next input from the user without using CPU time.
  //
  // A scene graph ought to be able to sort out the objects that might be
  // animated at some point in the future and treat them separately, so we
  // don't have to repeatedly scan the scene graph looking for animations.
  // Animations ought to be fairly rare.
  //
  // Default method returns false.  If this method returns false for an object,
  // it is never allowed to have an animation.
  virtual bool hasAnimationEver () const;
  // Default hasAnimationNow method returns false.  If no object claims to have
  // an animation now, then the TopLevel may relinquish the CPU and wait for
  // the next user input.  
  //
  // It's okay for an object to have an animation but to deny it
  // according to hasAnimationNow.  The result will be that the
  // animation is still called when we update the frame, but the frame isn't
  // continuously updated for the purpose of making the animation go.  This is
  // appropriate for an animation that maintains static information on the
  // screen, so it's doing its job properly when the CPU is relinquished.  For
  // instance, imagine that we started keeping track of the current simulation
  // time and made an animation that displays it in the corner of a screen.
  // That animation doesn't need to repeatedly update the screen when time is
  // stopped, so it should deny having an animation but nevertheless supply
  // one.
  //
  // In contrast, an animation that actually causes something to move, like the
  // one that makes selections blink, had better have an hasAnimationNow method
  // that returns true when there is blinking to do.
  //
  // Note that this method gets a "const SceneGraph *", so it can't modify the
  // scene graph.
  virtual bool hasAnimationNow (const SceneGraph *sg, int index) const;
  // Default method does nothing.  Called once per timestep.  The index is the
  // index of this object in the scene graph, for objects that appear more than
  // once in the scene graph.
  //
  // This method can also decide that the animation is no longer useful
  // and to delete it from the scene graph.  SceneGraph implementations ought
  // to be prepared to deal with this weird case.
  virtual void animateTimeStep (SceneGraph *sg, int index) const;
  // Default method does nothing.  Called once per frame.
  //
  // This method can also decide that the animation is no longer useful
  // and to delete it from the scene graph.  SceneGraph implementations ought
  // to be prepared to deal with this weird case.
  //
  // An animateFrame implementation can read the frame buffer from the Canvas.
  // If it does this, it will see the previous frame that was drawn.  This is
  // good for dumping the screen into a picture or movie.
  //
  // FIXME Take care that this remains true w/double buffering extension.
  virtual void animateFrame (Canvas *canvas, SceneGraph *sg, int index) const;
  // Whether it makes sense to select this PhysicsObject.  Default method
  // returns true.
  virtual bool isSelectable () const;

  // Whether it is useful to receive object o as a neighbor
  // for ourselves, when we are at myIndex and the neighbor is at neighborIndex
  // in the state array.  If this
  // method returns false, then the promise is that the given neighbor will not
  // be on the neighbors list when doPhysics is called.  This is
  // useful because neighbors change relatively rarely, and some discarding of
  // useless neighbors can therefore be done relatively rarely instead of over
  // and over again in doPhysics.  Note that this method is not given the state
  // information, since that changes during numerical integration. 
  // 
  // The default method always returns true.
  //
  // DesignAtom takes advantage of this.  BouncingBall could too.
  virtual bool isRelevantNeighbor (const PhysicsObject *o, int myIndex,
				   int neighborIndex) const;
};

#endif
