// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "ObjectDrawerConnection.h"

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __TheObjectDrawerFactory_h__
#include "TheObjectDrawerFactory.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

struct ObjectDrawerConnection::ObjectDrawerConnectionData {
  // The name of the drawer class, "Ball" or "Disk" or whatever.
  // Doesn't include characters for the window system.
  String m_objectDrawerClass;
  // When the global object drawer factory is changed, the singleton
  // will have to come by and call setObjectDrawer on this class.
  // The position is an integer that will be the position of this
  // ObjectDrawerConnection in its vector of odc's.
  int m_position;
  SP<Configuration> m_conf;
};

ObjectDrawerConnection::ObjectDrawerConnection (const String &odc,
						SP<Configuration> conf)
  : m_data (NEW (ObjectDrawerConnectionData ()))
{
  m_data->m_objectDrawerClass = odc;
  // Copy the configuration so it wont change out from under us.
  m_data->m_conf = conf->copy();
  // Don't want smart pointer nonsense in registerConnection to free this.
  ref ();
  TheObjectDrawerFactory::registerConnection (this);
  deref ();
}

ObjectDrawerConnection::~ObjectDrawerConnection () {
  // ref/deref is to make sure deRegisterConnection smart pointer
  // manipulation doesn't free the thing we're inthe middle of
  // destructing.
  ref ();
  TheObjectDrawerFactory::deRegisterConnection (this);
  deref ();
  assert (m_data);
  delete m_data;
  m_data = 0;
}

String ObjectDrawerConnection::getObjectDrawingClass () const {
  return m_data->m_objectDrawerClass;
};

void ObjectDrawerConnection::setObjectDrawer (SP<ObjectDrawer> od) {
  m_objectDrawer = od;
}

void ObjectDrawerConnection::setObjectDrawerPosition (int position) {
  m_data->m_position = position;
}

int ObjectDrawerConnection::getObjectDrawerPosition () const {
  return m_data->m_position;
}

SP<Configuration> ObjectDrawerConnection::getConfiguration () {
  return m_data->m_conf;
}

const ObjectDrawer *ObjectDrawerConnection::getObjectDrawer () const {
  assert (m_objectDrawer);
  return m_objectDrawer;
}

ObjectDrawer *ObjectDrawerConnection::getObjectDrawer () {
  assert (m_objectDrawer);
  return m_objectDrawer;
}
