// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __ObjectDrawer_h__
#define __ObjectDrawer_h__

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __PhysicsObjectInfo_h__
#include "PhysicsObjectInfo.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

class Canvas;
class Vec3;
class BoundingBox;
class BoundingSphere;

class SceneGraph;
class ObjectDrawer
  : public Configurable
{
public:
  // Determines whether a line segment (v1 to v2) intersects the 3-d
  // image of the given object.  If it does intersect the object, we
  // return true and put an intersection point in result.  (If the
  // caller does not care about the result, he can pass null there.)
  // If it does not intersect the 
  // object, we return false and leave result unmodified.  To get rid
  // of many special cases, we decree that if v1 or v2 is inside
  // the object in question, then it doesnt matter whether true or false is
  // returned, although even when true is returned in this case, we
  // still require that result is filled in and is a point in the
  // object.  To avoid more special cases, we also decree that the
  // caller must ensure that v1 is not equal to v2.
  // We return the intersection point that is closest to v1.
  virtual bool intersect (const Float *state,
			  const Vec3 &v1, const Vec3 &v2, Vec3 *result,
			  int objectIndex, const SceneGraph *sg) const
    = 0;
  // Whether the graphics of the object intersects a bounding box.
  // Same deal for approximations as physicsIntersect.
  virtual bool graphicsIntersect (const Float *state, const BoundingBox &box,
				  int objectIndex,
				  const SceneGraph *sg)
    const = 0;

  // Determine whether the point v intersects the 3d image of the
  // given object.
  virtual bool intersect (const Float *state,
			  const Vec3 &v,
			  int objectIndex,
			  const SceneGraph *sg) const = 0;

  virtual Float graphicsBoundingSphereRadius (int objectIndex,
					      const SceneGraph *sg)
    const = 0; // immutable

  // Use OpenGL primitives to draw the object.  
  // This routine can assume that the matrixmode is GL_MODELVIEW, and
  // it is not permitted to change the matrixmode, so it should not call
  // GL::matrixMode.
  // This routine should not make any assumptions about the current
  // matrial, so it should definitely call GL::material.
  // This routine should not modify the projection matrix.  We do not
  // attempt to keep the model view matrix consistent from one draw to
  // the next, so it will have to set up its model view matrix from
  // scratch.
  virtual void draw (const Float *state, const Canvas *vo,
		     int objectIndex, const SceneGraph *sg) const = 0;

  BoundingBox graphicsBoundingBox (const Float *state,
				   int objectIndex, const SceneGraph *sg) const;
  // This returns a sphere that has to enclose all of the graphics that we'll
  // draw.  
  BoundingSphere graphicsBoundingSphere (const Float *state,
					 int objectIndex,
					 const SceneGraph *sg) const;
  // selectionBoundingSphere returns a spherical region that is sensitive to
  // selections.  The default method returns the same as
  // graphicsBoundingSphere.  They are different if we are drawing bonds, since
  // the graphicsBoundingSphere has to enclose the bonds but the
  // selectionBoundingSphere doesn't.
  virtual BoundingSphere selectionBoundingSphere (const Float *state,
						  int objectIndex,
						  const SceneGraph *sg) const;
  // This is defined for both ObjectDrawer and PhysicsObject, because
  // both need it.
  virtual Vec3 boundingSphereCenter (const Float *state) const = 0;
  // Whether there is anything to see for the object.
  virtual bool isVisible () const = 0;
};

#endif
