// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "Disk.h"

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Matrix4_h__
#include "Matrix4.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __DiskConfiguration_h__
#include "DiskConfiguration.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __TheObjectDrawerFactory_h__
#include "TheObjectDrawerFactory.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

Disk::Disk (SP<DiskConfiguration> d)
  : PhysicsObject ("Disk", &*d),
    m_defaultNorm (d->getDefaultNorm ())
{
}

int Disk::stateSize () const {
  return 6;
}

void Disk::plausibleState (Dynavec <Float> &result) const {
  result.extendTo (0);
  result.extendTo (stateSize (), 0);
  setCenter (&*result, Vec3 (0,0,0));
  setNorm (&*result, m_defaultNorm);
}

Vec3 Disk::boundingSphereCenter (const Float *state) const {
  return center (state);
}

Float Disk::physicsBoundingSphereRadius () const {
  assert (0 && "No physics bounding sphere radius for a disk");
  return 0;
}

bool Disk::isPhysicsBounded () const {
  return false;
}

void Disk::translate (Float *state, const Vec3 &v) const {
  state [0] += v [0];
  state [1] += v [1];
  state [2] += v [2];
}

void Disk::rotate (Float *state, const Quaternion &q) const {
  setNorm (state, q.rotate (normUnit (state)));
}


bool Disk::exempt () const {
  return true;
}

bool Disk::physicsIntersect (const Float *state,
			     const BoundingBox &box) const
{
  // Return true if any of the corners of the box are on the
  // half-universe that experiences force from the disk. 
  Vec3 c = center (state);
  Vec3 n = normUnit (state);
  for (int i = 0; i < 8; i++) {
    if ((box.corner (i)-c) * n <= 0) return true;
  }
  return false;
}

const char *const Disk::staticClassId = "Disk";

const char *Disk::classId () const {
  return Disk::staticClassId;
}

void Disk::setCenter (Float *state, const Vec3 &v) {
  state [0] = v[0];
  state [1] = v[1];
  state [2] = v[2];
}

void Disk::setNorm (Float *state, const Vec3 &v) {
  Vec3 w = v.makeUnit();
  state [3] = w [0];
  state [4] = w [1];
  state [5] = w [2];
}

namespace {
  
  class DiskFactory: public Factory {
  public:
    DiskFactory () : Factory ("Disk")
    {}
    SP<Configuration> defaultConfiguration () const {
      return NEW (DiskConfiguration ());
    }
    SP<Configurable> makeIt (SP<Configuration> c) const {
      DiskConfiguration *conf = dynamic_cast <DiskConfiguration *> (&*c);
      assert (conf);
      return NEW (Disk (conf));
    }
  };
  
  bool Disk::isInstanceOf (const char *classId) const {
    return staticClassId == classId;
  }

  static const bool useless =
  (FactoryTable::store ("PhysicsObject", NEW (DiskFactory ())),
   TheObjectDrawerFactory::registerObjectDrawer ("Disk"),
   true);
}
