// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __BouncingBall_h__
#define __BouncingBall_h__

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Ball_h__
#include "Ball.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

class BoundingSphere;
class BoundingBox;
class BouncingBallConfiguration;
class Vec3;

#ifndef __SP_h__
#include "SP.h"
#endif

class BouncingBall: public Ball {
  // Mass of the ball.
  const Float m_mass;
  // Next constant scales how sharp the physics at the boundary of the
  // ball is.
  // High values make the balls act more like rigid object, at the
  // cost of numerical analysis that is inaccurate without a small
  // step size.
  // Low values give bouncing balls that overlap each other quite a
  // bit before they actually bounce.
  const Float m_stiffness;
  // The acceleration of gravity.  A nonnegative number.  Gravity
  // always accelerates in the -z direction.
  const Float m_gravity;
  // If two balls are closer than this, then don't try to compute
  // force between them, to avoid divide by 0.
  const Float m_smallDistance;
  const bool m_randomizeVelocities;
 public:
  // Not const because Ball updates the conf with billboarding counts. 
  BouncingBall (SP<BouncingBallConfiguration> b);
  ~BouncingBall ();
  void plausibleState (Dynavec <Float> &result) const;
  bool exempt () const;
  static inline Vec3 velocity (const Float *state) {
    return Vec3 (state + 3);
  }
  static inline void BouncingBall::setVelocity (Float *state, const Vec3 &v) {
    v.copyToPointer (state + 3);
  }
  inline static void addVelocity (Float *state, const Vec3 &v) {
    v.addToPointer (state + 3);
  }
  void doPhysics (const Float *state, 
		  Float *deriv,
		  int myIndex,
		  const Dynavec<PhysicsObjectInfo> &neighbors) const;
  int stateSize () const;
  Float physicsBoundingSphereRadius () const;
  bool isPhysicsBounded () const;
  bool physicsIntersect (const Float *state, const BoundingBox &box) const;
  static const char *const staticClassId;
  const char *classId () const;
  bool isInstanceOf (const char *classId) const;
};

#endif
