// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "BouncingBall.h"

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __BouncingBallConfiguration__h__
#include "BouncingBallConfiguration.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __Disk_h__
#include "Disk.h"
#endif

#ifndef __PhysicsObject_h__
#include "PhysicsObject.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __PhysicsObjectInfo_h__
#include "PhysicsObjectInfo.h"
#endif

#ifndef __ObjectDrawer_h__
#include "ObjectDrawer.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

BouncingBall::BouncingBall (const SP<BouncingBallConfiguration> b) :
  Ball (&*b),
  m_mass (b->getMass()),
  m_stiffness (b->getStiffness()),
  m_gravity (b->getGravity ()),
  m_smallDistance (b->getVisibleRadius()/20.0),
  m_randomizeVelocities (b->getRandomVelocities ())
{}

BouncingBall::~BouncingBall () {}

bool BouncingBall::isPhysicsBounded () const {
  return true;
}

// Repulsion force as a function of distance.  The distance should
// increase as the objects get further apart, with the goal of zero
// force at distance zero.  We also need a bounded fifth
// derivative, so we will be able to use Runga-Kutta to integrate.
// Since the bounding sphere of our bouncing balls is the visible
// sphere, the repulsion force had better be zero when the distance is
// positive.
static inline Float bounceRepel (Float distance) {
  if (distance >= 0) {
    return 0;
  } else {
    //
    Float d = - distance;
    Float d2 = d * d;
    Float d4 = d2 * d2;
    Float d5 = d4 * d;
    return d5;
  }
}

// Three state variables for the position, next three for the velocity.
int BouncingBall::stateSize () const {
  return 6;
}

// Next one for rand, RAND_MAX declarations.
#ifndef __stdlib_h__
#include <stdlib.h>
#define __stdlib_h__
#endif

// Return a random number between -1 and 1.
static Float randomVelocity () {
  return ((((Float) (rand ())) / RAND_MAX) * 2.0 - 1.0)/3.0;
}

void BouncingBall::plausibleState (Dynavec <Float> &result) const {
  result.extendTo (0);
  result.push (0); result.push (0); result.push (0); 
  if (m_randomizeVelocities) {
    result.push (randomVelocity ());
    result.push (randomVelocity ());
    result.push (randomVelocity ()); 
  } else {
    result.push (0); result.push (0); result.push (0); 
  }
}

void BouncingBall::doPhysics (const Float *state, 
			      Float *deriv,
			      int myIndex,
			      const Dynavec<PhysicsObjectInfo> &neighbors) const
{
  // We are given the velocity; make sure it is the derivative of the
  // center position.
  const Float *const myState = state + myIndex;
  Float *const myDeriv = deriv + myIndex;
  velocity(myState).addToPointer (myDeriv);
  const int neighborSize = neighbors.size();
  for (int i = 0; i < neighborSize; i++) {
    const PhysicsObjectInfo &neighborInfo = neighbors [i];
    const PhysicsObject *neighbor = neighborInfo.m_object;
    const int neighborIndex = neighborInfo.m_index;
    assert (neighborIndex != myIndex);
    const Float *neighborState = state + neighborIndex;
    const char *const neighborClassId = neighbor->classId();
    if (neighborClassId == BouncingBall::staticClassId) {
      const BouncingBall *bouncingNeighbor = (const BouncingBall *) neighbor;
      if (myIndex < neighborIndex) {
	Vec3 toNeighbor = center (myState) - center (neighborState);
	Float distToNeighborSquared = toNeighbor.lengthSquared();
	if (distToNeighborSquared <
	    FloatUtil::sqr (visibleRadius () +
				bouncingNeighbor->visibleRadius())) {
	  // This block accounts for only 5 ms of the
	  // timesteps per frame.
	  Float distToNeighbor = sqrt (distToNeighborSquared);
	  // If two objects are right on top of each other,
	  // distToNeighbor is 0, and then we divide by 0 when
	  // computing toNeighborUnit unless we have the next
	  // condition.  Debug builds crash when this happens.
	  if (distToNeighbor > m_smallDistance) {
	    // Unit vector from neighbor to me.
	    Vec3 toNeighborUnit = toNeighbor / distToNeighbor;
	    // Distance between our surfaces.  Will always be negative.
	    Float distance = distToNeighbor -
	      bouncingNeighbor->visibleRadius() -
	      visibleRadius ();
	    // Repulsion force.  Will always be positive.
	    Float force = bounceRepel (distance * m_stiffness);
	    // Force of the neighbor on me.
	    Vec3 forceVec = force * toNeighborUnit;
	    // Add acceleration to derivative of velocity.
	    addVelocity (myDeriv, forceVec / m_mass);
	    // Likewise, but opposite, for the neighbor.
	    addVelocity (deriv+neighborIndex, forceVec /
			 -(bouncingNeighbor->m_mass));
	  }
	}
      }
    } else if (neighborClassId == Disk::staticClassId) {
      const Vec3 diskCenter = Disk::center (neighborState);
      const Vec3 diskNorm = Disk::normUnit (neighborState);
      const Float distance =
	(center (myState) - diskCenter) * diskNorm - visibleRadius ();
      if (distance < 0) {
	Float force = bounceRepel (distance * m_stiffness);
	Float acceleration = force / m_mass;
	Vec3 accVec = acceleration * diskNorm;
	addVelocity (myDeriv, accVec);
      }
    } else {
      // We don't know what it is.  Don't do anything.
    }
  }
  // Make gravity happen.
  myDeriv[5] -= m_gravity;
  return;
}

bool BouncingBall::physicsIntersect (const Float *state,
				     const BoundingBox &box)
  const
{
  // Can't just do getObjectDrawer()->graphicsIntersect (state, box)
  // because the window sytsem may not have been selected yet.
  return box.overlaps (physicsBoundingBox (state));
}


bool BouncingBall::exempt () const {
  return false;
}

Float BouncingBall::physicsBoundingSphereRadius () const {
  return visibleRadius ();
}

const char *const BouncingBall::staticClassId = "BouncingBall";

const char *BouncingBall::classId () const {
  return BouncingBall::staticClassId;
}

bool BouncingBall::isInstanceOf (const char *classId) const {
  return (staticClassId == classId) | (Ball::staticClassId == classId);
}

namespace {
  bool useless = (FactoryTable::registerSubclass ("BouncingBall", "Ball"),
		  true);
}
