// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Integrator_h__
#define __Integrator_h__

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

class Evaluator;

// A numerical integrator is an object so that we can easily change
// integrators on the fly.  An integrator
class Integrator : public Refcount {
  // You can't copy or assign Integrators, they're too big.
  Integrator(Integrator &);     // Unimplemented
  void operator=(Integrator &); // Unimplemented
  
 public:
  // Need a public constructor so subclasses can be constructed.
  Integrator () {};
  // Initialize the integrator.
  // size is the number of state variables (all Float's) in
  // the system.  In other words, it is the number of coordinates in
  // the position of the system.
  // initialTime is the initial time.
  // initialStep is the initial time step.
  // state is the initial state of the system.  This will be
  // updated in place each time observe is called.
  // f is a function to compute the derivative of the state of the
  // system, given a position and a time.
  virtual void start (Float initialTime,
		      Float initialStep) = 0;
  virtual void setEvaluator (Evaluator *f) = 0;
  virtual Evaluator *getEvaluator () = 0;
  // Stop the integrator, freeing all memory.  Return the time of the
  // stop and the stepsize, so we have enough informaiton to start
  // this integrator or another one where this one left off
  virtual void stop (Float &time, Float &step) = 0;

  // Compute a next state of the system we're integrating.  Return the
  // time for the new state, and update the state vector we were given
  // when start was called in place.  Note that the integrator tells
  // us when the observation happened, so the integrator can vary the
  // timestep if it chooses.
  //
  // I am a little worried about losing precision in the time
  // coordinate if we let enough time pass.  The laws of physics do
  // not change with time, so it shouldn't be an issue there.  We
  // could have objects that wiggle around robotically, so their
  // position and velocity would be a function of time.  Hmm, if their
  // behavior is cyclic, then we could give each of them their own
  // notion of time, and allow them to reduce it by the period
  // themselves when it increases past then.  This seems too complex,
  // since I don't even have any such objects at the moment.  None of
  // my present code cares about the time, so ignore the issue.
  //
  // I can imagine an integrator that would use different timesteps
  // for different parts of space, doing error control independently
  // in different areas, with some overlap to cope with things moving
  // from one area to another.  Still, the user of the integrator
  // needs to be able to observe the state of the system, and the only
  // way I can think about that is if the observation happens at a
  // particular time, so in this case we'd have to engineer things so
  // the various parts of the system sync up periodically.  Because I
  // can imagine adaptive algorithms that use different timesteps in
  // different places, I didn't make a routine that allows the present
  // timestep to be extracted, since there might not be only one.
  // This will create problems if we add objects to a scene with
  // varying timesteps, since we'd have to make a new Integrator that
  // would not know about the previous adaptation of the timesteps, so
  // it would have to readapt.  This is too far off in the future;
  // don't worry about it yet.
  //
  // This protocol doesn't support that anyway, since the integrator
  // knows nothing about the structure of space.
  // 
  // FIXME we'd get better cache performance if we did this in blocks.
  // Right now, it's really memory bandwidth intensive.  The inputs
  // aren't organized into blocks, and they would have to be, so this
  // would require reorganizing the whole thing.
  virtual Float observe () = 0;
  virtual ~Integrator () {};
};

#endif
