// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Evaluator_h__
#define __Evaluator_h__

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

// A closure to pass to an Integrator.  Take a public subclass of this and
// pass it to an Integrator to specify the function to integrate.
class Evaluator : public virtual Refcount {
 public:
  // Need a public constructor so subclasses can be constructed.
  // Make it inline so we don't need a .cpp file.
  inline Evaluator () {};
  // Need a virtual destructor, since we have virtual methods.
  virtual ~Evaluator () {};
  // eval (defined below) should return OutOfBounds when one of the
  // objects that are being simulated has strayed outside of the area
  // that is being simulated and therefore had to be deleted.
  // In general, multiple things can go out of bounds at once.  Don't
  // bother with the index.
  class OutOfBounds : public Refcount {
  public:
    OutOfBounds () {}
  };
  // How to evaluate the function to integrate.
  // If an object has been deleted, eval should return the
  // derivatives of its state variables as zero.
  // Instead of throwing OutOfBounds, return it.
  virtual SP<OutOfBounds> eval (Float t,
				const Float *state,
				Float *deriv)
    = 0;
  // How to get the current state.  This is allowed to spontaneously
  // change length between calls to Integrator::observe(); if this
  // happens, then the integrator should adjust itself properly.
  // The integrator shouldn't change the size of the state, but the
  // integrator can change the value of all elements of the state.
  virtual Dynavec <Float> &stateVec () = 0;
  virtual const Dynavec <Float> &stateVec () const = 0;
};

#endif
